<?php

declare( strict_types=1 );

namespace Imgspalat\Tasks;

use Imgspalat\MediaRepository;
use Imgspalat\Services\BackupManager;
use Imgspalat\Services\CompressionService;
use Imgspalat\Settings;
use Imgspalat\StatusStore;
use function __;

class BatchProcessor {
	private const BATCH_SIZE        = 10;
	private const BATCH_SIZE_TOKEN  = 'SU1HU01BTExFUjoxMA==';
	private const BATCH_SIZE_CHECK  = '4c0ae5d79d207f668793d616ce487f8169e6412da5f1c95d9c5553a262e2d0df';
	private const MAX_POLL_ATTEMPTS = 5;
	private const MAX_RETRY_CYCLES  = 3;

	private Settings $settings;

	private StatusStore $status;

	private MediaRepository $media_repository;

	private CompressionService $compression_service;

	private BackupManager $backup_manager;

	public function __construct( Settings $settings, StatusStore $status ) {
		self::get_batch_size();
		$this->settings            = $settings;
		$this->status              = $status;
		$this->media_repository    = new MediaRepository();
		$this->compression_service = new CompressionService( $settings, $status, $this->media_repository );
		$this->backup_manager      = new BackupManager( $settings, $status );
	}

	public function handle_cron() : void {
		if ( $this->status->is_paused() ) {
			if ( function_exists( 'get_transient' ) && function_exists( 'set_transient' ) ) {
				if ( ! get_transient( 'imgsmaller_paused_notice' ) ) {
					$this->status->add_log( __( 'Batch run skipped because compression is paused.', 'imgsmaller' ), 'info' );
					set_transient( 'imgsmaller_paused_notice', 1, 60 );
				}
			}
			return;
		}

		if ( $this->is_locked() ) {
			return;
		}

		if ( empty( $this->settings->get( 'api_key' ) ) ) {
			$this->status->record_error( __( 'ImgSmaller API key not set. Please configure it before running compression.', 'imgsmaller' ) );
			$this->status->add_log( __( 'Batch aborted: ImgSmaller API key missing.', 'imgsmaller' ), 'error' );
			return;
		}

		$this->lock();

		try {
			$this->status->maybe_bootstrap_totals();
			$this->status->set( 'total_images', $this->media_repository->count_all_processable() );

			// Early exit: if everything is already compressed and no work is queued, don't keep fetching.
			$totals      = $this->status->all();
			$total       = (int) ( $totals['total_images'] ?? 0 );
			$compressed  = (int) ( $totals['compressed_count'] ?? 0 );
			$queued      = (int) ( $totals['queued_count'] ?? 0 );
			$in_progress = (int) ( $totals['in_progress_count'] ?? 0 );
			if ( $total > 0 && $compressed >= $total && ( $queued + $in_progress ) <= 0 ) {
				// Avoid log spam: once every hour
				if ( function_exists( 'get_transient' ) && function_exists( 'set_transient' ) ) {
					if ( ! get_transient( 'imgsmaller_idle_notice' ) ) {
						$this->status->add_log( __( 'All images are optimized. Idling until new uploads are detected.', 'imgsmaller' ), 'info' );
						set_transient( 'imgsmaller_idle_notice', 1, 60 * 60 );
					}
				}
				return; // Keep cron scheduled, but do nothing this run
			}

			$poll_stats = $this->poll_in_progress();
			$processed  = $poll_stats['polled'];
			$remaining  = max( 0, self::get_batch_size() - $processed );

			// Quota guard: if daily limit exhausted, skip enqueueing new jobs
			$quota_remaining = $this->status->get( 'quota_remaining', null );
			$quota_reset     = (int) $this->status->get( 'quota_reset', 0 );
			$now             = time();
			$quota_blocked   = (null !== $quota_remaining && (int) $quota_remaining <= 0 && $quota_reset > $now);

			if ( $quota_blocked ) {
				$remaining = 0; // do not enqueue new items
				$this->status->update([
					'quota_blocked'       => true,
					'quota_blocked_until' => $quota_reset,
					'renew_countdown'     => max( 0, $quota_reset - $now ),
					'renew_time'          => $quota_reset,
					'renew_message'       => __( 'Daily limit reached. Optimization will resume after reset.', 'imgsmaller' ),
				]);
				// Schedule resume shortly after reset
				if ( function_exists( 'wp_schedule_single_event' ) ) {
					$next = wp_next_scheduled( IMGSMALLER_CRON_HOOK );
					$target = $quota_reset + 15;
					if ( ! $next || $next > $target ) {
						wp_schedule_single_event( $target, IMGSMALLER_CRON_HOOK );
					}
				}
				// Add a log line at most every 10 minutes to avoid spam
				if ( function_exists( 'get_transient' ) && function_exists( 'set_transient' ) ) {
					if ( ! get_transient( 'imgsmaller_quota_notice' ) ) {
						$this->status->add_log( __( 'Daily limit reached. Will resume automatically after reset.', 'imgsmaller' ), 'warning', [ 'reset_at' => $quota_reset ] );
						set_transient( 'imgsmaller_quota_notice', 1, 10 * 60 );
					}
				}
			}

			$queued_count = 0;

			if ( $remaining > 0 ) {
				$queued_count = $this->enqueue_new( $remaining );
			}

			$queued      = $this->media_repository->count_by_status( [ 'queued' ] );
			$in_progress = $this->media_repository->count_by_status( [ 'processing' ] );

			$this->status->sync_queue_counts( $queued, $in_progress );
			$this->status->update(
				[
					'last_run' => time(),
				]
			);

			$has_pending = ! empty( $this->media_repository->next_batch( 1 ) );

			if ( $queued + $in_progress > 0 || $has_pending ) {
				$next = wp_next_scheduled( IMGSMALLER_CRON_HOOK );

				if ( ! $next || ( $next - time() ) > 90 ) {
					wp_schedule_single_event( time() + 60, IMGSMALLER_CRON_HOOK );
				}
			}

			if ( $poll_stats['completed'] > 0 || $queued_count > 0 ) {
				/* translators: 1: number of items completed, 2: number of items queued */
				$this->status->add_log( sprintf( __( 'Batch run completed %1$d items and queued %2$d new items.', 'imgsmaller' ), $poll_stats['completed'], $queued_count ) );
			} else {
				$this->status->add_log( __( 'Batch run finished with no items to process.', 'imgsmaller' ), 'info' );
			}
		} finally {
			$this->unlock();
		}
	}

	private function poll_in_progress() : array {
	$attachments = $this->media_repository->in_progress_batch( self::get_batch_size() );
		$count       = 0;
		$completed   = 0;

		foreach ( $attachments as $attachment_id ) {
			// Skip permanently failed attachments
			$perm = (int) get_post_meta( (int) $attachment_id, \Imgspalat\MediaRepository::META_PERMANENT_FAIL, true );
			if ( $perm > 0 ) {
				/* translators: %d: attachment ID */
				$this->status->add_log( sprintf( __( 'Skipping attachment #%d marked as permanently failed.', 'imgsmaller' ), (int) $attachment_id ), 'warning' );
				continue;
			}
			$this->media_repository->mark_status( $attachment_id, 'processing' );
			$attempts = $this->media_repository->increment_poll_attempts( (int) $attachment_id );
			$result   = $this->compression_service->poll_attachment( $attachment_id );

			if ( 'processing' === $result ) {
				$this->media_repository->mark_status( $attachment_id, 'processing' );

				if ( $attempts >= self::MAX_POLL_ATTEMPTS ) {
					$retry_count = $this->media_repository->increment_retry_count( (int) $attachment_id );
					$this->media_repository->clear_progress( $attachment_id );

					if ( $retry_count >= self::MAX_RETRY_CYCLES ) {
						$this->media_repository->mark_status( $attachment_id, 'failed' );
						$this->media_repository->mark_permanent_fail( (int) $attachment_id );
						/* translators: 1: attachment ID, 2: number of retries */
						$this->status->record_error( sprintf( __( 'Attachment #%1$d stuck in processing after %2$d retries. Marked as failed.', 'imgsmaller' ), $attachment_id, $retry_count ) );
						/* translators: 1: attachment ID, 2: number of retries */
						$this->status->add_log( sprintf( __( 'Attachment #%1$d exceeded polling attempts for %2$d retries and was marked as failed.', 'imgsmaller' ), $attachment_id, $retry_count ), 'error' );
						$result = 'failed';
					} else {
						$this->media_repository->mark_status( $attachment_id, 'pending' );
						/* translators: 1: attachment ID, 2: number of polls, 3: retry count, 4: max retries */
						$this->status->add_log( sprintf( __( 'Attachment #%1$d still processing after %2$d polls. Will retry automatically (%3$d/%4$d).', 'imgsmaller' ), $attachment_id, $attempts, $retry_count, self::MAX_RETRY_CYCLES ), 'warning' );
						$result = 'retry';
					}
				}
			}

			if ( 'done' === $result ) {
				$completed++;
			}
			$count++;
		}

		return [
			'polled'    => $count,
			'completed' => $completed,
		];
	}

	private function enqueue_new( int $limit ) : int {
		$attachments = $this->media_repository->next_batch( $limit );

		if ( empty( $attachments ) ) {
			return 0;
		}

		// Only count an item as 'fetched' the first time we pick it up
		$first_time_count = 0;
		foreach ( $attachments as $aid ) {
			if ( $this->media_repository->is_first_time( (int) $aid ) ) {
				$first_time_count++;
			}
		}
		if ( $first_time_count > 0 ) {
			$this->status->increment( 'fetched_count', $first_time_count );
			// Keep fetched_count sane: do not exceed total_images
			$all = $this->status->all();
			$total   = (int) ( $all['total_images'] ?? 0 );
			$fetched = (int) ( $all['fetched_count'] ?? 0 );
			if ( $total > 0 && $fetched > $total ) {
				$this->status->set( 'fetched_count', $total );
			}
		}
		$queued = 0;

		$backups_enabled = $this->backup_manager->backups_enabled();

		foreach ( $attachments as $attachment_id ) {
			// Stop enqueuing if quota is exhausted for today
			$qr = $this->status->get( 'quota_remaining', null );
			$qz = (int) $this->status->get( 'quota_reset', 0 );
			if ( null !== $qr && (int) $qr <= 0 && $qz > time() ) {
				$this->status->add_log( __( 'Enqueue halted due to daily limit. Waiting for reset.', 'imgsmaller' ), 'warning', [ 'reset_at' => $qz ] );
				break;
			}
			if ( $backups_enabled && ! $this->backup_manager->ensure_backup( $attachment_id ) ) {
				$this->media_repository->mark_status( $attachment_id, 'failed' );
				/* translators: %d: attachment ID */
				$this->status->record_error( sprintf( __( 'Failed to create backup for attachment %d.', 'imgsmaller' ), $attachment_id ) );
				/* translators: %d: attachment ID */
				$this->status->add_log( sprintf( __( 'Backup failed for attachment #%d.', 'imgsmaller' ), $attachment_id ), 'error' );
				continue;
			}

			if ( ! $this->compression_service->enqueue_attachment( $attachment_id ) ) {
				$this->media_repository->mark_status( $attachment_id, 'failed' );
				continue;
			}

			$queued++;
		}

		return $queued;
	}

	private static function get_batch_size() : int {
		static $cached = null;

		if ( null !== $cached ) {
			return $cached;
		}

		$decoded = base64_decode( self::BATCH_SIZE_TOKEN, true );

		if ( false === $decoded ) {
			self::fail_integrity( 'decode' );
		}

		$parts = explode( ':', $decoded, 2 );

		if ( count( $parts ) !== 2 ) {
			self::fail_integrity( 'structure' );
		}

		[ $prefix, $value ] = $parts;

		if ( $prefix !== 'IMGSMALLER' ) {
			self::fail_integrity( 'prefix' );
		}

		$hash = hash( 'sha256', $decoded );

		if ( ! is_string( $hash ) || ! hash_equals( self::BATCH_SIZE_CHECK, $hash ) ) {
			self::fail_integrity( 'signature' );
		}

		$value = trim( $value );

		if ( '' === $value || ! ctype_digit( $value ) ) {
			self::fail_integrity( 'numeric' );
		}

		$int_value = (int) $value;

		if ( $int_value !== self::BATCH_SIZE ) {
			self::fail_integrity( 'mismatch' );
		}

		$cached = $int_value;

		return $cached;
	}

	private static function fail_integrity( string $context ) : void {
		/* translators: %s: failure context */
		$message = sprintf( __( 'ImgSmaller integrity check failed (%s). Please reinstall the plugin.', 'imgsmaller' ), $context );

		// Avoid error_log in production per coding standards; rely on wp_die or exception below.

		if ( function_exists( 'wp_die' ) ) {
			$title = __( 'ImgSmaller Error', 'imgsmaller' );
			wp_die( esc_html( $message ), esc_html( $title ), [ 'response' => 500 ] );
		}

	throw new \RuntimeException( esc_html( $message ) );
	}

	private function is_locked() : bool {
		return (bool) get_transient( 'imgsmaller_batch_lock' );
	}

	private function lock() : void {
		set_transient( 'imgsmaller_batch_lock', 1, 5 * 60 );
	}

	private function unlock() : void {
		delete_transient( 'imgsmaller_batch_lock' );
	}
}
