<?php

declare( strict_types=1 );

namespace Imgspalat;

class StatusStore {
	public const OPTION_KEY = 'imgsmaller_status';
	private const LOG_OPTION = 'imgsmaller_logs';

	private array $defaults = [
		'total_images'       => 0,
		'fetched_count'      => 0,
		'sent_count'         => 0,
		'compressed_count'   => 0,
		'queued_count'       => 0,
		'in_progress_count'  => 0,
		'processing_count'   => 0,
		'last_run'           => 0,
		'last_error'         => '',
		'quota_remaining'    => null,
		'quota_reset'        => null,
		'remaining_count'    => 0,
		'progress_percent'   => 0,
		'is_paused'          => false,
		'restore_in_progress'=> false,
		'restore_total'      => 0,
		'restore_restored'   => 0,
		'restore_attempted'  => 0,
		'restore_failed'     => 0,
		'restore_message'    => '',
	];

	private ?array $cache = null;

	private ?array $log_cache = null;

	public function all() : array {
		if ( null === $this->cache ) {
			$raw = function_exists( 'get_option' ) ? get_option( self::OPTION_KEY, [] ) : [];
			$this->cache = function_exists( 'wp_parse_args' ) ? wp_parse_args( (array) $raw, $this->defaults ) : array_merge( $this->defaults, (array) $raw );
		}

		$status = $this->cache;

		return $this->with_derived_metrics( $status );
	}

	public function get( string $key, $default = null ) {
		$all = $this->all();

		return $all[ $key ] ?? $default;
	}

	public function set( string $key, $value ) : void {
		$all         = $this->all();
		$all[ $key ] = $value;
		$this->persist( $all );
	}

	public function increment( string $key, int $by = 1 ) : void {
		$all         = $this->all();
		$all[ $key ] = ( $all[ $key ] ?? 0 ) + $by;
		$this->persist( $all );
	}

	public function update( array $payload ) : void {
		$all = $this->all();

		foreach ( $payload as $key => $value ) {
			$all[ $key ] = $value;
		}

		$this->persist( $all );
	}

	public function sync_queue_counts( int $queued, int $in_progress ) : void {
		$this->update(
			[
				'queued_count'      => $queued,
				'in_progress_count'  => $in_progress,
			]
		);
	}

	public function maybe_bootstrap_totals() : void {
		$all = $this->all();

		if ( $all['total_images'] > 0 ) {
			return;
		}

		if ( ! function_exists( 'wp_count_posts' ) ) {
			return;
		}

		// Prefer WP_Query to avoid direct DB calls for counting image attachments
		if ( class_exists( '\\WP_Query' ) ) {
			$q = new \WP_Query(
				[
					'post_type'              => 'attachment',
					'post_status'            => 'inherit',
					'post_mime_type'         => [ 'image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/avif' ],
					'fields'                 => 'ids',
					'posts_per_page'         => 1,
					'no_found_rows'          => false, // we need found_posts for the total
					'update_post_meta_cache' => false,
					'update_post_term_cache' => false,
					'cache_results'          => false,
				]
			);
			$count = (int) $q->found_posts;
			if ( $count > 0 ) {
				$all['total_images'] = $count;
				$this->persist( $all );
			}
		} else {
			// Fallback to wp_count_posts
			$count = wp_count_posts( 'attachment' );
			if ( isset( $count->inherit ) ) {
				$all['total_images'] = (int) $count->inherit;
				$this->persist( $all );
			}
		}
	}

	public function record_error( string $message ) : void {
		$this->set( 'last_error', $message );
	}

	public function clear_error() : void {
		$this->set( 'last_error', '' );
	}

	public function pause() : void {
		$this->set( 'is_paused', true );
		if ( function_exists( 'set_transient' ) ) {
			set_transient( 'imgsmaller_paused_notice', 1, 60 );
		}
	}

	public function resume() : void {
		$this->set( 'is_paused', false );
		if ( function_exists( 'delete_transient' ) ) {
			delete_transient( 'imgsmaller_paused_notice' );
		}
	}

	public function is_paused() : bool {
		return (bool) $this->get( 'is_paused', false );
	}

	public function delete() : void {
		if ( function_exists( 'delete_option' ) ) {
			delete_option( self::OPTION_KEY );
		}
		$this->cache = null;
		$this->clear_logs();
	}

	public function logs( int $limit = 20 ) : array {
		if ( null === $this->log_cache ) {
			$this->log_cache = function_exists( 'get_option' ) ? get_option( self::LOG_OPTION, [] ) : [];
		}

		return array_slice( (array) $this->log_cache, 0, $limit );
	}

	public function add_log( string $message, string $level = 'info', array $context = [] ) : void {
		if ( null === $this->log_cache ) {
			$this->log_cache = function_exists( 'get_option' ) ? get_option( self::LOG_OPTION, [] ) : [];
		}

		$entry = [
			'time'    => time(),
			'level'   => $level,
			'message' => $message,
			'context' => $context,
		];

		array_unshift( $this->log_cache, $entry );
		$this->log_cache = array_slice( $this->log_cache, 0, 50 );

		if ( function_exists( 'update_option' ) ) {
			update_option( self::LOG_OPTION, $this->log_cache );
		}
	}

	public function clear_logs() : void {
		if ( function_exists( 'delete_option' ) ) {
			delete_option( self::LOG_OPTION );
		}
		$this->log_cache = [];
	}

	private function persist( array $data ) : void {
		$this->cache = $data;
		if ( function_exists( 'update_option' ) ) {
			update_option( self::OPTION_KEY, $this->cache );
		}
	}

	private function with_derived_metrics( array $status ) : array {
		$total       = max( 0, (int) ( $status['total_images'] ?? 0 ) );
		$compressed  = max( 0, (int) ( $status['compressed_count'] ?? 0 ) );
		$queued      = max( 0, (int) ( $status['queued_count'] ?? 0 ) );
		$in_progress = max( 0, (int) ( $status['in_progress_count'] ?? 0 ) );

		if ( $total > 0 ) {
			$percent = ( $compressed / $total ) * 100;
		} else {
			$percent = $compressed > 0 ? 100 : 0;
		}

		$percent                  = (int) round( min( 100, max( 0, $percent ) ) );
		$status['progress_percent'] = $percent;
		$status['remaining_count']  = max( 0, $total - min( $compressed, $total ) );
		$status['processing_count'] = $queued + $in_progress;
		$status['is_paused']        = ! empty( $status['is_paused'] );
		$status['restore_in_progress'] = ! empty( $status['restore_in_progress'] );
		$status['restore_total']       = max( 0, (int) ( $status['restore_total'] ?? 0 ) );
		$status['restore_restored']    = max( 0, (int) ( $status['restore_restored'] ?? 0 ) );
		$status['restore_message']     = (string) ( $status['restore_message'] ?? '' );

		if ( $status['restore_restored'] > $status['restore_total'] ) {
			$status['restore_restored'] = $status['restore_total'];
		}

		return $status;
	}
}
