<?php

declare( strict_types=1 );

namespace Imgspalat;

class Settings {
	public const OPTION_KEY = 'imgsmaller_settings';

	private array $defaults = [
		'api_key'        => '',
		'output_format'  => 'webp',
		'target_size'    => 'auto',
		'connected'      => false,
		'checked_at'     => 0,
		'enable_backup'  => true,
		'cron_token'     => '',
		'exclude_ids'    => '',
		// onboarding/tour state
		'onboarding_dismissed' => false,
		'onboarding_completed' => false,
	];

	private ?array $cache = null;

	public function all() : array {
		if ( null === $this->cache ) {
			$raw = function_exists( 'get_option' ) ? get_option( self::OPTION_KEY, [] ) : [];
			$this->cache = function_exists( 'wp_parse_args' ) ? wp_parse_args( (array) $raw, $this->defaults ) : array_merge( $this->defaults, (array) $raw );
			// Ensure we have a cron token; generate once and persist.
			if ( empty( $this->cache['cron_token'] ) ) {
				$this->cache['cron_token'] = $this->generate_cron_token();
				if ( function_exists( 'update_option' ) ) {
					update_option( self::OPTION_KEY, $this->cache );
				}
			}
			$this->cache['enable_backup'] = (bool) ( $this->cache['enable_backup'] ?? true );
			if ( isset( $this->cache['file_url'] ) ) {
				unset( $this->cache['file_url'] );
			}
		}

		return $this->cache;
	}

	public function get( string $key, $default = null ) {
		$settings = $this->all();

		return $settings[ $key ] ?? $default;
	}

	public function set( string $key, $value ) : void {
		$settings        = $this->all();
		if ( 'enable_backup' === $key ) {
			$value = (bool) $value;
		}
		$settings[ $key ] = $value;
		$this->persist( $settings );
	}

	public function set_api_key( string $api_key ) : void {
		if ( ! function_exists( 'sanitize_text_field' ) ) {
			require_once ABSPATH . 'wp-includes/formatting.php';
		}
		$this->set( 'api_key', sanitize_text_field( $api_key ) );
	}

	public function set_output_format( string $format ) : void {
		$allowed = [ 'jpg', 'jpeg', 'png', 'webp', 'avif', 'original' ];

		if ( ! in_array( strtolower( $format ), $allowed, true ) ) {
			$format = 'webp';
		}

		$this->set( 'output_format', $format );
	}

	public function get_cron_token() : string {
		return (string) ( $this->get( 'cron_token', '' ) ?: $this->generate_cron_token_and_persist() );
	}

	/**
	 * Regenerate and persist a new cron token. Returns the new token.
	 */
	public function regenerate_cron_token() : string {
		return $this->generate_cron_token_and_persist();
	}

	private function generate_cron_token_and_persist() : string {
		$token = $this->generate_cron_token();
		$this->set( 'cron_token', $token );
		return $token;
	}

	private function generate_cron_token() : string {
		$bytes = function_exists( 'wp_generate_password' ) ? wp_generate_password( 32, false, false ) : bin2hex( random_bytes( 16 ) );
		return substr( preg_replace( '/[^A-Za-z0-9]/', '', (string) $bytes ), 0, 32 );
	}

	/**
	 * Return an array of excluded attachment IDs (ints), sanitized from settings.
	 */
	public function get_excluded_ids() : array {
		$raw = (string) ( $this->get( 'exclude_ids', '' ) );
		$ids = [];
		if ( $raw !== '' ) {
			foreach ( preg_split( '/[\s,]+/', $raw ) as $part ) {
				$int = (int) trim( (string) $part );
				if ( $int > 0 ) {
					$ids[] = $int;
				}
			}
		}
		return array_values( array_unique( $ids ) );
	}

	public function set_target_size( string $target ) : void {
		$target = trim( $target );

		if ( '' === $target ) {
			$target = 'auto';
		}

		$this->set( 'target_size', $target );
	}

	public function set_enable_backup( bool $enabled ) : void {
		$this->set( 'enable_backup', $enabled );
	}

	public function is_backup_enabled() : bool {
		return (bool) $this->get( 'enable_backup', true );
	}

	public function mark_connection_status( bool $connected ) : void {
		$settings                = $this->all();
		$settings['connected']   = $connected;
		$settings['checked_at']  = time();
		$this->persist( $settings );
	}

	public function delete_all() : void {
		if ( function_exists( 'delete_option' ) ) {
			delete_option( self::OPTION_KEY );
		}
		$this->cache = null;
	}

	public function replace( array $settings ) : void {
		if ( isset( $settings['file_url'] ) ) {
			unset( $settings['file_url'] );
		}
		$this->cache = function_exists( 'wp_parse_args' ) ? wp_parse_args( $settings, $this->defaults ) : array_merge( $this->defaults, $settings );
		$this->cache['enable_backup'] = (bool) ( $this->cache['enable_backup'] ?? true );
		if ( function_exists( 'update_option' ) ) {
			update_option( self::OPTION_KEY, $this->cache );
		}
	}

	private function persist( array $settings ) : void {
		if ( isset( $settings['file_url'] ) ) {
			unset( $settings['file_url'] );
		}
		if ( isset( $settings['enable_backup'] ) ) {
			$settings['enable_backup'] = (bool) $settings['enable_backup'];
		}
		$this->cache = $settings;
		if ( function_exists( 'update_option' ) ) {
			update_option( self::OPTION_KEY, $this->cache );
		}
	}
}
