<?php

declare( strict_types=1 );

namespace Imgspalat\Services;

use Imgspalat\MediaRepository;
use Imgspalat\Settings;
use Imgspalat\StatusStore;
use function __;

class CompressionService {
	private const BASE_URL = 'https://imgsmaller.com';

	private Settings $settings;

	private StatusStore $status;

	private MediaRepository $media_repository;

	public function __construct( Settings $settings, StatusStore $status, ?MediaRepository $repository = null ) {
		$this->settings         = $settings;
		$this->status           = $status;
		$this->media_repository = $repository ?: new MediaRepository();
	}

	public function enqueue_attachment( int $attachment_id ) : bool {
		$api_key = (string) $this->settings->get( 'api_key', '' );

		if ( empty( $api_key ) ) {
			$this->status->record_error( __( 'ImgSmaller API key is missing.', 'imgsmaller' ) );
			return false;
		}
		$file_url = wp_get_attachment_url( $attachment_id );

		if ( empty( $file_url ) ) {
			/* translators: %d: attachment ID */
			$this->status->record_error( sprintf( __( 'Public URL for attachment %d not found.', 'imgsmaller' ), $attachment_id ) );
			return false;
		}

		if ( ! function_exists( 'wp_check_filetype' ) ) {
			require_once ABSPATH . 'wp-includes/functions.php';
		}

		$parts   = wp_parse_url( (string) $file_url );
		$mime    = wp_check_filetype( (string) ( $parts['path'] ?? '' ) )['type'] ?? 'application/octet-stream';
		$payload = [
			'file_url'    => $file_url,
			'target_size' => (string) $this->settings->get( 'target_size', 'auto' ),
			'output'      => $this->normalize_output_format( (string) $this->settings->get( 'output_format', 'webp' ), $mime ),
		];

		$response = wp_remote_post(
			self::BASE_URL . '/api/v1/compress',
			[
				'timeout'     => 60,
				'headers'     => $this->headers( $api_key ),
				'body'        => wp_json_encode( $payload ),
				'data_format' => 'body',
			]
		);

		if ( is_wp_error( $response ) ) {
			$this->status->record_error( $response->get_error_message() );
			return false;
		}

		$code = (int) wp_remote_retrieve_response_code( $response );
		$body = json_decode( (string) wp_remote_retrieve_body( $response ), true );

		// If the API could not fetch the file_url (common on private sites), retry with a proxy URL.
		if ( 422 === $code ) {
			$err_msg = '';
			if ( is_array( $body ) ) {
				$err_msg = (string) ( $body['message'] ?? ( $body['error'] ?? '' ) );
			}
			if ( '' !== $err_msg && ( false !== stripos( $err_msg, 'file_url' ) || false !== stripos( $err_msg, 'fetch' ) ) ) {
				$proxy_url = $this->build_proxy_url( $attachment_id );
				if ( $proxy_url ) {
					$payload_proxy = $payload;
					$payload_proxy['file_url'] = $proxy_url;
					/* translators: %d: attachment ID */
					$this->log( sprintf( __( 'Retrying with proxy URL for attachment #%d.', 'imgsmaller' ), $attachment_id ) );
					$response = wp_remote_post(
						self::BASE_URL . '/api/v1/compress',
						[
							'timeout'     => 60,
							'headers'     => $this->headers( $api_key ),
							'body'        => wp_json_encode( $payload_proxy ),
							'data_format' => 'body',
						]
					);

					if ( ! is_wp_error( $response ) ) {
						$code = (int) wp_remote_retrieve_response_code( $response );
						$body = json_decode( (string) wp_remote_retrieve_body( $response ), true );
					}
				}
			}
		}

		if ( 202 === $code && is_array( $body ) ) {
			// Derive and persist the result URL used for polling.
			$progress = [];
			if ( ! empty( $body['result_url'] ) ) {
				$progress['result_url'] = (string) $body['result_url'];
			} elseif ( ! empty( $body['key'] ) ) {
				$progress['result_url'] = trailingslashit( self::BASE_URL ) . 'result/' . sanitize_text_field( (string) $body['key'] );
			}

			if ( ! empty( $body['download_url_template'] ) ) {
				$progress['download_url_template'] = (string) $body['download_url_template'];
			}

			$this->media_repository->reset_retry_count( $attachment_id );
			$this->media_repository->reset_poll_attempts( $attachment_id );
			$this->media_repository->mark_status( $attachment_id, 'queued' );
			if ( ! empty( $progress ) ) {
				$this->media_repository->store_progress( $attachment_id, $progress );
			}
			$this->status->increment( 'sent_count' );
			$this->status->clear_error();
			/* translators: 1: attachment ID, 2: attachment title */
			$this->log( sprintf( __( 'Queued attachment #%1$d (%2$s) for compression.', 'imgsmaller' ), $attachment_id, get_the_title( $attachment_id ) ) );

			return true;
		}

		if ( 429 === $code ) {
			$reset = isset( $body['reset_in_seconds'] ) ? (int) $body['reset_in_seconds'] : 3600;
			$this->status->update(
				[
					'quota_remaining' => 0,
					'quota_reset'     => time() + $reset,
					'quota_blocked'   => true,
					'renew_countdown' => $reset,
					'renew_time'      => time() + $reset,
					'renew_message'   => __( 'Daily limit reached. Optimization will resume after reset.', 'imgsmaller' ),
				]
			);
		}

		// Handle repeated 422 failures: increment retry count and mark permanent after threshold.
		if ( 422 === $code ) {
			$count = $this->media_repository->increment_retry_count( $attachment_id );
			if ( $count >= 3 && function_exists( 'update_post_meta' ) ) {
				update_post_meta( $attachment_id, MediaRepository::META_PERMANENT_FAIL, 1 );
				$this->media_repository->mark_status( $attachment_id, 'failed' );
				/* translators: %d: attachment ID */
				$this->log( sprintf( __( 'Attachment #%d marked as permanently skipped after repeated fetch failures (422).', 'imgsmaller' ), $attachment_id ), 'warning' );
			}
		}

		$error = $body['message'] ?? ( $body['error'] ?? __( 'Unexpected response from ImgSmaller API.', 'imgsmaller' ) );
	/* translators: %d: HTTP status code */
	$message = sprintf( '(%d) %s', $code, $error );
		$this->status->record_error( $message );
		/* translators: 1: attachment ID, 2: error message */
		$this->log( sprintf( __( 'Failed to queue attachment #%1$d: %2$s', 'imgsmaller' ), $attachment_id, $message ), 'error' );

		return false;
	}

	private function build_proxy_url( int $attachment_id ) : ?string {
		$token = (string) $this->settings->get_cron_token();
		if ( '' === $token ) {
			return null;
		}
		$base = trailingslashit( site_url() ) . 'wp-json/imgsmaller/v1/file';
		$args = [ 'token' => $token, 'id' => $attachment_id ];
		if ( function_exists( 'add_query_arg' ) ) {
			return esc_url_raw( add_query_arg( $args, $base ) );
		}
		$qs = http_build_query( $args );
		return esc_url_raw( $base . ( false === strpos( $base, '?' ) ? '?' : '&' ) . $qs );
	}

	public function poll_attachment( int $attachment_id ) : string {
		$progress = $this->media_repository->get_progress( $attachment_id );
		$result   = $progress['result_url'] ?? '';

		if ( empty( $result ) ) {
			return 'failed';
		}

		$api_key = (string) $this->settings->get( 'api_key', '' );

		$response = wp_remote_get(
			(string) $result,
			[
				'timeout' => 20,
				'headers' => $api_key ? [ 'X-API-Key' => $api_key ] : [],
			]
		);

		if ( is_wp_error( $response ) ) {
			$this->status->record_error( $response->get_error_message() );
			return 'processing';
		}

		$code = (int) wp_remote_retrieve_response_code( $response );
		$body = json_decode( (string) wp_remote_retrieve_body( $response ), true );

		if ( 200 !== $code || ! is_array( $body ) ) {
			if ( in_array( $code, [ 400, 401, 403, 404, 410, 422 ], true ) || $code >= 500 ) {
				$context_body = is_array( $body ) ? $body : [];
				/* translators: %d: HTTP status code */
				/* translators: %d: HTTP status code */
				$poll_msg = __( 'ImgSmaller API returned HTTP %d while polling.', 'imgsmaller' );
				$message  = $this->format_failure_message( $context_body, sprintf( $poll_msg, $code ) );
				$this->status->record_error( $message );
				/* translators: 1: attachment ID, 2: error message */
				$this->log( sprintf( __( 'Polling failed for attachment #%1$d: %2$s', 'imgsmaller' ), $attachment_id, $message ), 'error', [ 'http_code' => $code, 'response' => $context_body ] );
				$this->media_repository->mark_status( $attachment_id, 'failed' );
				$this->media_repository->clear_progress( $attachment_id );

				return 'failed';
			}

			return 'processing';
		}

		// Handle new API style: ready boolean, download_url, and url path.
		if ( array_key_exists( 'ready', $body ) ) {
			if ( true !== $body['ready'] ) {
				return 'processing';
			}
		}

		$status_value = $this->normalize_result_status( $body );
		$download_url = $this->build_download_url( $body, $progress );

		if ( '' === $status_value && ! empty( $download_url ) ) {
			$status_value = 'done';
		}

		$processing_statuses = [ 'processing', 'queued', 'pending', 'running', 'in_progress', 'in-progress', 'waiting' ];
		if ( in_array( $status_value, $processing_statuses, true ) ) {
			return 'processing';
		}

		$failure_statuses = [ 'error', 'failed', 'failure', 'cancelled', 'canceled', 'timeout', 'expired' ];
		if ( in_array( $status_value, $failure_statuses, true ) ) {
			$message = $this->format_failure_message( $body, __( 'Compression failed to complete.', 'imgsmaller' ), $status_value );
			$this->status->record_error( $message );
			/* translators: 1: attachment ID, 2: error message */
			$this->log( sprintf( __( 'Compression failed for attachment #%1$d: %2$s', 'imgsmaller' ), $attachment_id, $message ), 'error', [ 'status' => $status_value, 'response' => $body ] );
			$this->media_repository->mark_status( $attachment_id, 'failed' );
			$this->media_repository->clear_progress( $attachment_id );

			return 'failed';
		}

		if ( '' === $status_value && empty( $download_url ) ) {
			return 'processing';
		}

		$success_statuses = [ 'done', 'completed', 'complete', 'finished', 'success', 'succeeded', 'ready' ];
		if ( '' !== $status_value && ! in_array( $status_value, $success_statuses, true ) ) {
			$message = $this->format_failure_message( $body, __( 'Compression failed to complete.', 'imgsmaller' ), $status_value );
			$this->status->record_error( $message );
			/* translators: 1: attachment ID, 2: error message */
			$this->log( sprintf( __( 'Compression failed for attachment #%1$d: %2$s', 'imgsmaller' ), $attachment_id, $message ), 'error', [ 'status' => $status_value, 'response' => $body ] );
			$this->media_repository->mark_status( $attachment_id, 'failed' );
			$this->media_repository->clear_progress( $attachment_id );

			return 'failed';
		}

		if ( empty( $download_url ) ) {
			$this->status->record_error( __( 'Download URL missing from ImgSmaller response.', 'imgsmaller' ) );
			/* translators: %d: attachment ID */
			$this->log( sprintf( __( 'No download URL returned for attachment #%d.', 'imgsmaller' ), $attachment_id ), 'error' );
			return 'failed';
		}

		$download = wp_remote_get(
			$download_url,
			[
				'timeout' => 60,
			]
		);

		if ( is_wp_error( $download ) ) {
			$this->status->record_error( $download->get_error_message() );
			return 'processing';
		}

		if ( 200 !== (int) wp_remote_retrieve_response_code( $download ) ) {
			$this->status->record_error( __( 'Unable to download optimized image.', 'imgsmaller' ) );
			return 'processing';
		}

		$body_contents   = wp_remote_retrieve_body( $download );
		$current_path    = get_attached_file( $attachment_id );

		if ( ! $current_path ) {
			$this->status->record_error( __( 'Attachment file path missing.', 'imgsmaller' ) );
			/* translators: %d: attachment ID */
			$this->log( sprintf( __( 'Attachment #%d is missing a stored file path.', 'imgsmaller' ), $attachment_id ), 'error' );
			return 'failed';
		}

		$old_metadata    = function_exists( 'wp_get_attachment_metadata' ) ? (array) wp_get_attachment_metadata( $attachment_id ) : [];
		$target_extension = $this->determine_target_extension( $attachment_id, $download, $download_url, $current_path );
		$destination      = $this->prepare_destination_path( $current_path, $target_extension );
		$write_path       = $destination['path'];
		$extension_changed = $destination['changed'];

	$dir_writable = function_exists( 'wp_is_writable' ) ? wp_is_writable( dirname( (string) $write_path ) ) : false;
		if ( ! $write_path || ! $dir_writable ) {
			$this->status->record_error( __( 'Attachment destination is not writable.', 'imgsmaller' ) );
			/* translators: %d: attachment ID */
			$this->log( sprintf( __( 'Destination not writable for attachment #%d.', 'imgsmaller' ), $attachment_id ), 'error' );
			return 'failed';
		}

		if ( false === file_put_contents( $write_path, $body_contents ) ) {
			$this->status->record_error( __( 'Failed to write optimized file.', 'imgsmaller' ) );
			/* translators: %d: attachment ID */
			$this->log( sprintf( __( 'Unable to write optimized file for attachment #%d.', 'imgsmaller' ), $attachment_id ), 'error' );
			return 'failed';
		}

		if ( $extension_changed ) {
			update_attached_file( $attachment_id, $write_path );
		}

		if ( ! function_exists( 'wp_generate_attachment_metadata' ) ) {
			require_once ABSPATH . 'wp-admin/includes/image.php';
		}

		$metadata = wp_generate_attachment_metadata( $attachment_id, $write_path );
		wp_update_attachment_metadata( $attachment_id, $metadata );

		if ( $extension_changed ) {
			$this->cleanup_previous_files( $old_metadata, $current_path, $write_path );
		}

		$this->media_repository->mark_status( $attachment_id, 'done' );
		$this->media_repository->reset_retry_count( $attachment_id );
		$this->media_repository->clear_progress( $attachment_id );
		$this->status->increment( 'compressed_count' );
		$this->status->clear_error();
		/* translators: 1: attachment ID, 2: attachment title */
		$this->log( sprintf( __( 'Compression complete for attachment #%1$d (%2$s).', 'imgsmaller' ), $attachment_id, get_the_title( $attachment_id ) ) );

		return 'done';
	}

	public function test_connection() : bool {
		$api_key = (string) $this->settings->get( 'api_key', '' );

		if ( empty( $api_key ) ) {
			return false;
		}

		$response = wp_remote_get(
			self::BASE_URL . '/api/v1/info',
			[
				'timeout' => 15,
				'headers' => [ 'X-API-Key' => $api_key ],
			]
		);

		if ( is_wp_error( $response ) ) {
			$this->status->record_error( $response->get_error_message() );
			return false;
		}

		$code = (int) wp_remote_retrieve_response_code( $response );

		if ( 200 === $code ) {
			$body = json_decode( (string) wp_remote_retrieve_body( $response ), true );
			if ( isset( $body['remaining_today'] ) ) {
				$reset = (int) ( $body['reset_in_seconds'] ?? 0 );
				$payload = [ 'quota_remaining' => (int) $body['remaining_today'] ];
				if ( $reset > 0 ) {
					$payload['quota_reset']     = time() + $reset;
					$payload['renew_countdown'] = $reset;
					$payload['renew_time']      = time() + $reset;
				}
				$this->status->update( $payload );
			}

			$this->status->clear_error();
			return true;
		}

		if ( 401 === $code ) {
			$this->status->record_error( __( 'Invalid ImgSmaller API key.', 'imgsmaller' ) );
		}

		return false;
	}

	private function headers( string $api_key ) : array {
		return [
			'Content-Type' => 'application/json',
			'X-API-Key'    => $api_key,
		];
	}

	private function normalize_output_format( string $format, string $mime ) : string {
		$format = strtolower( $format );
		$allowed = [ 'jpeg', 'jpg', 'png', 'webp', 'avif', 'original' ];

		if ( ! in_array( $format, $allowed, true ) ) {
			$format = 'webp';
		}

		if ( 'original' === $format ) {
			return 'original';
		}

		if ( 'jpg' === $format ) {
			return 'jpg';
		}

		if ( 'jpeg' === $format ) {
			return 'jpeg';
		}

		if ( 'png' === $format ) {
			return 'png';
		}

		if ( 'webp' === $format ) {
			return 'webp';
		}

		if ( 'avif' === $format ) {
			return 'avif';
		}

		// Fallback to input type when allowed.
		if ( strpos( $mime, 'jpeg' ) !== false ) {
			return 'jpeg';
		}

		if ( strpos( $mime, 'png' ) !== false ) {
			return 'png';
		}

		return 'webp';
	}

	private function build_download_url( array $result, array $progress ) : ?string {
		// New API returns absolute download_url
		if ( ! empty( $result['download_url'] ) ) {
			return esc_url_raw( (string) $result['download_url'] );
		}

		// Some variants return 'download'
		if ( ! empty( $result['download'] ) ) {
			return esc_url_raw( (string) $result['download'] );
		}

		// If only a relative 'url' is returned, try to make it absolute
		if ( ! empty( $result['url'] ) && is_string( $result['url'] ) ) {
			$path = ltrim( (string) $result['url'], '/' );
			return trailingslashit( self::BASE_URL ) . $path;
		}

		if ( isset( $result['image_id'] ) && ! empty( $progress['download_url_template'] ) ) {
			return str_replace( '{imageId}', (string) $result['image_id'], (string) $progress['download_url_template'] );
		}

		return null;
	}

	private function normalize_result_status( array $body ) : string {
		$possible_statuses = [];

		foreach ( [ 'status', 'state', 'result' ] as $key ) {
			if ( isset( $body[ $key ] ) && is_string( $body[ $key ] ) && '' !== $body[ $key ] ) {
				$possible_statuses[] = strtolower( (string) $body[ $key ] );
			}
		}

		if ( isset( $body['data'] ) && is_array( $body['data'] ) ) {
			foreach ( [ 'status', 'state', 'result' ] as $key ) {
				if ( isset( $body['data'][ $key ] ) && is_string( $body['data'][ $key ] ) && '' !== $body['data'][ $key ] ) {
					$possible_statuses[] = strtolower( (string) $body['data'][ $key ] );
				}
			}
		}

		if ( isset( $body['success'] ) ) {
			if ( true === $body['success'] ) {
				$possible_statuses[] = 'success';
			} elseif ( false === $body['success'] ) {
				$possible_statuses[] = 'failed';
			}
		}

		return $possible_statuses[0] ?? '';
	}

	private function format_failure_message( array $body, string $fallback, string $status_value = '' ) : string {
		$message = '';

		if ( isset( $body['message'] ) && is_string( $body['message'] ) ) {
			$message = $body['message'];
		} elseif ( isset( $body['error'] ) && is_string( $body['error'] ) ) {
			$message = $body['error'];
		} elseif ( isset( $body['errors'] ) && is_array( $body['errors'] ) ) {
			$first_error = reset( $body['errors'] );

			if ( is_string( $first_error ) ) {
				$message = $first_error;
			} elseif ( is_array( $first_error ) ) {
				foreach ( [ 'message', 'detail', 'error' ] as $error_key ) {
					if ( isset( $first_error[ $error_key ] ) && is_string( $first_error[ $error_key ] ) ) {
						$message = $first_error[ $error_key ];
						break;
					}
				}
			}
		}

		if ( '' !== $status_value ) {
			/* translators: %s: status value */
			$status_tpl    = __( 'Compression failed with status "%s".', 'imgsmaller' );
			$status_message = sprintf( $status_tpl, $status_value );
			$message        = $message ? sprintf( '%s (%s)', $message, $status_message ) : $status_message;
		}

		return $message ?: $fallback;
	}

	private function determine_target_extension( int $attachment_id, array $download_response, string $download_url, string $current_path ) : string {
		$current_extension = strtolower( (string) pathinfo( $current_path, PATHINFO_EXTENSION ) );
		$requested         = strtolower( (string) $this->settings->get( 'output_format', 'webp' ) );

		if ( 'original' === $requested ) {
			return $current_extension;
		}

		$header_extension = $this->extension_from_content_type( wp_remote_retrieve_header( $download_response, 'content-type' ) );
		$url_extension    = $this->extension_from_url( $download_url );
		$requested        = $this->normalize_extension( $requested );

		foreach ( [ $header_extension, $url_extension, $requested, $current_extension ] as $candidate ) {
			$candidate = $this->normalize_extension( $candidate );
			if ( null !== $candidate ) {
				return $candidate;
			}
		}

		return $current_extension;
	}

	private function extension_from_content_type( $content_type ) : ?string {
		if ( empty( $content_type ) ) {
			return null;
		}

		if ( is_array( $content_type ) ) {
			$content_type = reset( $content_type );
		}

		$content_type = strtolower( trim( (string) $content_type ) );
		$content_type = explode( ';', $content_type )[0];

		$map = [
			'image/jpeg' => 'jpg',
			'image/jpg'  => 'jpg',
			'image/png'  => 'png',
			'image/webp' => 'webp',
			'image/avif' => 'avif',
		];

		return $map[ $content_type ] ?? null;
	}

	private function extension_from_url( string $url ) : ?string {
		$parts = wp_parse_url( $url );
		$path  = (string) ( $parts['path'] ?? '' );

		if ( '' === $path ) {
			return null;
		}

		$extension = pathinfo( $path, PATHINFO_EXTENSION );

		return $extension ? strtolower( (string) $extension ) : null;
	}

	private function normalize_extension( $extension ) : ?string {
		if ( null === $extension ) {
			return null;
		}

		$extension = strtolower( ltrim( (string) $extension, '.' ) );

		if ( '' === $extension ) {
			return null;
		}

		if ( 'jpeg' === $extension ) {
			return 'jpg';
		}

		if ( 'tif' === $extension ) {
			return 'tiff';
		}

		return $extension;
	}

	private function prepare_destination_path( string $current_path, ?string $target_extension ) : array {
		$current_extension = strtolower( (string) pathinfo( $current_path, PATHINFO_EXTENSION ) );
		$target_extension  = $this->normalize_extension( $target_extension ) ?: $current_extension;

		if ( $target_extension === $current_extension || '' === $target_extension ) {
			return [
				'path'    => $current_path,
				'changed' => false,
			];
		}

		$info     = pathinfo( $current_path );
		$dirname  = isset( $info['dirname'] ) && '.' !== $info['dirname'] ? $info['dirname'] : '';
		$filename = $info['filename'] ?? '';

		if ( '' === $filename ) {
			return [
				'path'    => $current_path,
				'changed' => false,
			];
		}

		$separator = DIRECTORY_SEPARATOR;
		if ( false !== strpos( $current_path, '/' ) ) {
			$separator = '/';
		} elseif ( false !== strpos( $current_path, '\\' ) ) {
			$separator = '\\';
		}

		$new_path = ( '' !== $dirname ? $dirname . $separator : '' ) . $filename . '.' . $target_extension;

		return [
			'path'    => $new_path,
			'changed' => $new_path !== $current_path,
		];
	}

	private function cleanup_previous_files( array $old_metadata, string $old_path, string $new_path ) : void {
		if ( $old_path === $new_path ) {
			return;
		}

		$paths_to_remove = [];

		if ( file_exists( $old_path ) ) {
			$paths_to_remove[] = $old_path;
		}

		if ( ! empty( $old_metadata ) ) {
			$upload_dir = wp_upload_dir();
			$base_dir   = isset( $upload_dir['basedir'] ) ? trailingslashit( $upload_dir['basedir'] ) : '';

			if ( ! empty( $old_metadata['file'] ) ) {
				$primary_path = $base_dir . ltrim( (string) $old_metadata['file'], '/\\' );
				if ( $primary_path !== $new_path && file_exists( $primary_path ) ) {
					$paths_to_remove[] = $primary_path;
				}
				$primary_dir = dirname( $primary_path );
			} else {
				$primary_dir = dirname( $old_path );
			}

			if ( ! empty( $old_metadata['sizes'] ) && is_array( $old_metadata['sizes'] ) ) {
				$dir_prefix = $primary_dir ? trailingslashit( $primary_dir ) : '';
				foreach ( $old_metadata['sizes'] as $size ) {
					if ( isset( $size['file'] ) && '' !== $size['file'] ) {
						$file = $dir_prefix . ltrim( (string) $size['file'], '/\\' );
						if ( $file !== $new_path && file_exists( $file ) ) {
							$paths_to_remove[] = $file;
						}
					}
				}
			}
		}

		$paths_to_remove = array_unique( $paths_to_remove );

		foreach ( $paths_to_remove as $path ) {
			wp_delete_file( $path );
		}
	}

	private function log( string $message, string $level = 'info', array $context = [] ) : void {
		$this->status->add_log( $message, $level, $context );
	}
}
