<?php

declare( strict_types=1 );

namespace Imgspalat\Services;

use Imgspalat\MediaRepository;
use Imgspalat\Settings;
use Imgspalat\StatusStore;
use WP_Error;
use function __;

class BackupManager {
	private const META_BACKUP_PATH = '_imgsmaller_backup_path';
	private const JOB_OPTION       = 'imgsmaller_restore_job';
	private const META_ORIGINAL_FILE = '_imgsmaller_original_file';

	private Settings $settings;
	private StatusStore $status;

	public function __construct( ?Settings $settings = null, ?StatusStore $status = null ) {
		$this->settings = $settings ?: new Settings();
		$this->status   = $status ?: new StatusStore();
	}

	private function log( string $message, string $level = 'info', array $context = [] ) : void {
		$this->status->add_log( $message, $level, $context );
	}

	public function backups_enabled() : bool {
		return $this->settings->is_backup_enabled();
	}

	public function ensure_backup( int $attachment_id ) : bool {
		if ( ! $this->backups_enabled() ) {
			return true;
		}

		$path = get_post_meta( $attachment_id, self::META_BACKUP_PATH, true );

		if ( $path && file_exists( $path ) ) {
			return true;
		}

		if ( $path && ! file_exists( $path ) ) {
			delete_post_meta( $attachment_id, self::META_BACKUP_PATH );
		}

		$file = get_attached_file( $attachment_id );

		if ( ! $file || ! file_exists( $file ) ) {
			/* translators: %d: attachment ID */
			$msg = __( 'Unable to locate original file for attachment #%d while creating backup.', 'imgsmaller' );
			$this->log( sprintf( $msg, $attachment_id ), 'warning' );
			return false;
		}

		$backup_dir = $this->backup_dir_for( $attachment_id );

		if ( ! wp_mkdir_p( $backup_dir ) ) {
			/* translators: %d: attachment ID */
			$msg = __( 'Unable to create backup directory for attachment #%d.', 'imgsmaller' );
			$this->log( sprintf( $msg, $attachment_id ), 'error' );
			return false;
		}

		if ( ! function_exists( 'wp_basename' ) ) {
			require_once ABSPATH . 'wp-includes/formatting.php';
		}
		$original_relative = get_post_meta( $attachment_id, '_wp_attached_file', true );
		if ( $original_relative && ! get_post_meta( $attachment_id, self::META_ORIGINAL_FILE, true ) ) {
			update_post_meta( $attachment_id, self::META_ORIGINAL_FILE, $original_relative );
		}

		$backup_path = trailingslashit( $backup_dir ) . wp_basename( $file );

		if ( ! copy( $file, $backup_path ) ) {
			/* translators: %d: attachment ID */
			$msg = __( 'Failed to copy attachment #%d into backup directory.', 'imgsmaller' );
			$this->log( sprintf( $msg, $attachment_id ), 'error' );
			return false;
		}

		update_post_meta( $attachment_id, self::META_BACKUP_PATH, $backup_path );

		return true;
	}

	public function restore( int $attachment_id ) : bool {
		$path = get_post_meta( $attachment_id, self::META_BACKUP_PATH, true );

		if ( ! $path || ! file_exists( $path ) ) {
			/* translators: %d: attachment ID */
			$msg = __( 'Backup file missing for attachment #%d during restore.', 'imgsmaller' );
			$this->log( sprintf( $msg, $attachment_id ), 'warning' );
			delete_post_meta( $attachment_id, self::META_BACKUP_PATH );
			delete_post_meta( $attachment_id, self::META_ORIGINAL_FILE );
			return false;
		}

		$upload_dir        = wp_upload_dir();
		$original_relative = get_post_meta( $attachment_id, self::META_ORIGINAL_FILE, true );

		if ( $original_relative ) {
			$dest = trailingslashit( $upload_dir['basedir'] ) . ltrim( $original_relative, '/\\' );
		} else {
			$dest = get_attached_file( $attachment_id );
		}

		if ( ! $dest ) {
			/* translators: %d: attachment ID */
			$msg = __( 'Unable to determine restore destination for attachment #%d.', 'imgsmaller' );
			$this->log( sprintf( $msg, $attachment_id ), 'error' );
			return false;
		}

		if ( ! wp_mkdir_p( dirname( $dest ) ) ) {
			/* translators: %d: attachment ID */
			$msg = __( 'Failed to prepare destination directory for attachment #%d during restore.', 'imgsmaller' );
			$this->log( sprintf( $msg, $attachment_id ), 'error' );
			return false;
		}

		$current_file = get_attached_file( $attachment_id );

		if ( $current_file && file_exists( $current_file ) && $current_file !== $dest ) {
			wp_delete_file( $current_file );
		}

		if ( ! copy( $path, $dest ) ) {
			/* translators: %d: attachment ID */
			$msg = __( 'Failed to copy backup for attachment #%d during restore.', 'imgsmaller' );
			$this->log( sprintf( $msg, $attachment_id ), 'error' );
			return false;
		}

		update_attached_file( $attachment_id, $dest );

		if ( ! function_exists( 'wp_generate_attachment_metadata' ) ) {
			require_once ABSPATH . 'wp-admin/includes/image.php';
		}

		$metadata = wp_generate_attachment_metadata( $attachment_id, $dest );
		wp_update_attachment_metadata( $attachment_id, $metadata );

		// After a successful restore, remove the stored backup for this attachment
		try {
			if ( $path && file_exists( $path ) ) {
				wp_delete_file( $path );
			}
			$dir = $this->backup_dir_for( $attachment_id );
			if ( is_dir( $dir ) ) {
				$this->delete_directory( $dir );
			}
			delete_post_meta( $attachment_id, self::META_BACKUP_PATH );
			delete_post_meta( $attachment_id, self::META_ORIGINAL_FILE );
			/* translators: %d: attachment ID */
			$this->log( sprintf( __( 'Restored and removed backup for attachment #%d.', 'imgsmaller' ), $attachment_id ), 'info' );
		} catch ( \Throwable $e ) {
			// Non-fatal if backup cleanup fails
			/* translators: 1: attachment ID, 2: error message */
			$this->log( sprintf( __( 'Restore succeeded but cleanup failed for attachment #%1$d: %2$s', 'imgsmaller' ), $attachment_id, $e->getMessage() ), 'warning' );
		}

		return true;
		}


	public function restore_all( MediaRepository $repository ) : int {
		$restored = 0;

		$query = new \WP_Query(
			[
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'fields'         => 'ids',
				'nopaging'       => true,
				'post_mime_type' => [ 'image/jpeg', 'image/png', 'image/webp', 'image/avif' ],
				'no_found_rows'  => true,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Admin-only backup restore scan filtered by a single meta key; minimal fields and no_found_rows.
				'meta_query'     => [
					[
						'key'     => self::META_BACKUP_PATH,
						'compare' => 'EXISTS',
					],
				],
			]
		);

		foreach ( (array) $query->posts as $attachment_id ) {
			if ( $this->restore( (int) $attachment_id ) ) {
				$repository->mark_status( (int) $attachment_id, 'pending' );
				$repository->clear_progress( (int) $attachment_id );
				$restored++;
			}
		}

		return $restored;
	}

	public function backup_base_dir() : string {
		$base = get_option( 'imgsmaller_backup_dir' );

		if ( ! $base ) {
			$upload_dir = wp_upload_dir();
			$base       = trailingslashit( $upload_dir['basedir'] ) . 'imgsmaller-backups';
		}

		return trailingslashit( wp_normalize_path( $base ) );
	}

	public function create_backup_archive() {
		if ( ! class_exists( '\\ZipArchive' ) ) {
			return new WP_Error( 'imgsmaller_zip_missing', __( 'The ZipArchive PHP extension is required to export backups.', 'imgsmaller' ) );
		}

		$base_dir = $this->backup_base_dir();

		if ( ! is_dir( $base_dir ) ) {
			return new WP_Error( 'imgsmaller_no_backups', __( 'No backup directory found.', 'imgsmaller' ) );
		}

		$files = $this->get_backup_files( $base_dir );

		if ( empty( $files ) ) {
			return new WP_Error( 'imgsmaller_no_files', __( 'No backup files were found to export.', 'imgsmaller' ) );
		}

		$temp = wp_tempnam( 'imgsmaller-backups' );

		if ( ! $temp ) {
			return new WP_Error( 'imgsmaller_temp_failed', __( 'Unable to create a temporary file for the backup archive.', 'imgsmaller' ) );
		}

		$zip = new \ZipArchive();
		$result = $zip->open( $temp, \ZipArchive::CREATE | \ZipArchive::OVERWRITE );

		if ( true !== $result ) {
			return new WP_Error( 'imgsmaller_zip_open_failed', __( 'Unable to create the backup archive.', 'imgsmaller' ) );
		}

		foreach ( $files as $absolute => $relative ) {
			$zip->addFile( $absolute, $relative );
		}

		$zip->close();

		return $temp;
	}

	public function import_backup_archive( string $zip_path ) {
		if ( ! class_exists( '\\ZipArchive' ) ) {
			return new WP_Error( 'imgsmaller_zip_missing', __( 'The ZipArchive PHP extension is required to import backups.', 'imgsmaller' ) );
		}

		if ( ! is_readable( $zip_path ) ) {
			return new WP_Error( 'imgsmaller_zip_unreadable', __( 'Unable to read the uploaded backup archive.', 'imgsmaller' ) );
		}

		$zip = new \ZipArchive();
		$result = $zip->open( $zip_path );

		if ( true !== $result ) {
			return new WP_Error( 'imgsmaller_zip_open_failed', __( 'Unable to open the uploaded archive.', 'imgsmaller' ) );
		}

		$target_base = $this->backup_base_dir();
		$temp_dir    = trailingslashit( wp_normalize_path( get_temp_dir() ) ) . 'imgsmaller-backup-' . wp_generate_uuid4();

		if ( ! wp_mkdir_p( $temp_dir ) ) {
			$zip->close();
			return new WP_Error( 'imgsmaller_temp_dir_failed', __( 'Unable to prepare a temporary directory for extraction.', 'imgsmaller' ) );
		}

		if ( ! $zip->extractTo( $temp_dir ) ) {
			$zip->close();
			$this->delete_directory( $temp_dir );
			return new WP_Error( 'imgsmaller_zip_extract_failed', __( 'Failed to extract the backup archive.', 'imgsmaller' ) );
		}

		$zip->close();

		if ( ! $this->replace_directory( $temp_dir, $target_base ) ) {
			$this->delete_directory( $temp_dir );
			return new WP_Error( 'imgsmaller_backup_replace_failed', __( 'Unable to replace the existing backup directory.', 'imgsmaller' ) );
		}

		return true;
	}

	/**
	 * Import a ZIP of compressed images and apply them as current attachment files.
	 * Matching strategy: by original filename (basename) against attachments' current file basename.
	 * On replace, update attached file path and regenerate metadata; mark status to 'done'.
	 * Returns array { replaced: int } or WP_Error.
	 */
	public function import_replacements_archive( string $zip_path ) {
		if ( ! class_exists( '\ZipArchive' ) ) {
			return new WP_Error( 'imgsmaller_zip_missing', __( 'The ZipArchive PHP extension is required to import compressed images.', 'imgsmaller' ) );
		}

		if ( ! is_readable( $zip_path ) ) {
			return new WP_Error( 'imgsmaller_zip_unreadable', __( 'Unable to read the uploaded archive.', 'imgsmaller' ) );
		}

		$zip = new \ZipArchive();
		$result = $zip->open( $zip_path );
		if ( true !== $result ) {
			return new WP_Error( 'imgsmaller_zip_open_failed', __( 'Unable to open the uploaded archive.', 'imgsmaller' ) );
		}

		$temp_dir = trailingslashit( wp_normalize_path( get_temp_dir() ) ) . 'imgsmaller-replace-' . wp_generate_uuid4();
		if ( ! wp_mkdir_p( $temp_dir ) ) {
			$zip->close();
			return new WP_Error( 'imgsmaller_temp_dir_failed', __( 'Unable to prepare a temporary directory for extraction.', 'imgsmaller' ) );
		}

		if ( ! $zip->extractTo( $temp_dir ) ) {
			$zip->close();
			$this->delete_directory( $temp_dir );
			return new WP_Error( 'imgsmaller_zip_extract_failed', __( 'Failed to extract the compressed archive.', 'imgsmaller' ) );
		}
		$zip->close();

		// Build basename => absolute map of extracted files
		$extracted = [];
		$iter = new \RecursiveIteratorIterator( new \RecursiveDirectoryIterator( $temp_dir, \FilesystemIterator::SKIP_DOTS ) );
		foreach ( $iter as $file ) {
			if ( $file->isDir() ) { continue; }
			$abs = wp_normalize_path( $file->getPathname() );
			$extracted[ strtolower( wp_basename( $abs ) ) ] = $abs;
		}

		if ( empty( $extracted ) ) {
			$this->delete_directory( $temp_dir );
			return new WP_Error( 'imgsmaller_no_files', __( 'No image files were found in the archive.', 'imgsmaller' ) );
		}

		// Query all processable attachments to match basenames
		$q = new \WP_Query([
			'post_type'      => 'attachment',
			'post_status'    => 'inherit',
			'fields'         => 'ids',
			'nopaging'       => true,
			'no_found_rows'  => true,
			'post_mime_type' => [ 'image/jpeg', 'image/png', 'image/webp', 'image/avif' ],
		]);

		$replaced = 0;
		$repository = new \Imgspalat\MediaRepository();

		foreach ( (array) $q->posts as $attachment_id ) {
			$dest = get_attached_file( (int) $attachment_id );
			if ( ! $dest ) { continue; }
			$basename = strtolower( wp_basename( $dest ) );
			if ( ! isset( $extracted[ $basename ] ) ) { continue; }

			$src = $extracted[ $basename ];
			if ( ! wp_mkdir_p( dirname( $dest ) ) ) {
				/* translators: %d: attachment ID */
				$msg = __( 'Failed to prepare destination for attachment #%d.', 'imgsmaller' );
				$this->log( sprintf( $msg, $attachment_id ), 'error' );
				continue;
			}

			// Replace file
			if ( ! @copy( $src, $dest ) ) {
				/* translators: %d: attachment ID */
				$msg = __( 'Failed to replace file for attachment #%d.', 'imgsmaller' );
				$this->log( sprintf( $msg, $attachment_id ), 'error' );
				continue;
			}

			// Update metadata
			if ( ! function_exists( 'wp_generate_attachment_metadata' ) ) {
				require_once ABSPATH . 'wp-admin/includes/image.php';
			}
			$meta = wp_generate_attachment_metadata( (int) $attachment_id, $dest );
			wp_update_attachment_metadata( (int) $attachment_id, $meta );

			// Mark as done in plugin state
			$repository->mark_status( (int) $attachment_id, 'done' );
			$repository->clear_progress( (int) $attachment_id );
			$replaced++;
		}

		$this->delete_directory( $temp_dir );

		return [ 'replaced' => $replaced ];
	}

	public function delete_all_backups() {
		$base_dir = $this->backup_base_dir();
		$summary  = [
			'files_deleted'        => 0,
			'attachments_cleared'  => 0,
			'backup_directory'     => $base_dir,
		];

		if ( is_dir( $base_dir ) ) {
			$files = $this->get_backup_files( $base_dir );
			$summary['files_deleted'] = count( $files );

			$this->delete_directory( $base_dir );

			if ( is_dir( $base_dir ) ) {
				return new WP_Error( 'imgsmaller_backup_delete_failed', __( 'Unable to remove the backup directory. Please verify file permissions.', 'imgsmaller' ) );
			}
		}

		$ids = $this->get_attachment_ids_with_backups();

		foreach ( $ids as $attachment_id ) {
			delete_post_meta( $attachment_id, self::META_BACKUP_PATH );
			delete_post_meta( $attachment_id, self::META_ORIGINAL_FILE );
		}

		$summary['attachments_cleared'] = count( $ids );

		$this->clear_restore_job();

		return $summary;
	}

	private function get_backup_files( string $base_dir ) : array {
		$files = [];

		$iterator = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator( $base_dir, \FilesystemIterator::SKIP_DOTS )
		);

		foreach ( $iterator as $file ) {
			if ( $file->isDir() ) {
				continue;
			}

			$absolute = wp_normalize_path( $file->getPathname() );
			$relative = ltrim( str_replace( wp_normalize_path( $base_dir ), '', $absolute ), '/\\' );
			$files[ $absolute ] = $relative;
		}

		return $files;
	}

	private function get_attachment_ids_with_backups() : array {
		// Use WP_Query to avoid direct database calls

		$query = new \WP_Query(
			[
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'fields'         => 'ids',
				'nopaging'       => true,
				'no_found_rows'  => true,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Admin-only query to detect attachments with backups by a single meta key; minimal fields and no_found_rows.
				'meta_query'     => [
					[
						'key'     => self::META_BACKUP_PATH,
						'compare' => 'EXISTS',
					],
				],
			]
		);
		return array_map( 'intval', (array) $query->posts );
	}

	private function delete_directory( string $dir ) : void {
		$dir = wp_normalize_path( $dir );

		if ( ! is_dir( $dir ) ) {
			return;
		}

		$iterator = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator( $dir, \FilesystemIterator::SKIP_DOTS ),
			\RecursiveIteratorIterator::CHILD_FIRST
		);

		foreach ( $iterator as $path ) {
			$pathname = $path->getPathname();
			if ( $path->isDir() ) {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
				@rmdir( $pathname );
			} else {
				// Prefer core helper when available
				wp_delete_file( $pathname );
			}
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
		@rmdir( $dir );
	}

	private function replace_directory( string $source, string $destination ) : bool {
		$source      = wp_normalize_path( $source );
		$destination = wp_normalize_path( $destination );

		if ( is_dir( $destination ) ) {
			$this->delete_directory( $destination );
		}

		if ( ! wp_mkdir_p( dirname( $destination ) ) ) {
			return false;
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename
		if ( @rename( $source, $destination ) ) {
			return true;
		}

		if ( ! wp_mkdir_p( $destination ) ) {
			return false;
		}

		$iterator = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator( $source, \FilesystemIterator::SKIP_DOTS ),
			\RecursiveIteratorIterator::SELF_FIRST
		);

		foreach ( $iterator as $item ) {
			$target_path = wp_normalize_path( $destination . DIRECTORY_SEPARATOR . $iterator->getSubPathName() );

			if ( $item->isDir() ) {
				if ( ! is_dir( $target_path ) && ! wp_mkdir_p( $target_path ) ) {
					return false;
				}
			} else {
				$stream = file_get_contents( $item->getPathname() );

				if ( false === $stream ) {
					return false;
				}

				if ( false === file_put_contents( $target_path, $stream ) ) {
					return false;
				}
			}
		}

		$this->delete_directory( $source );

		return true;
	}

	public function prepare_restore_job() : array {
		$meta_key = self::get_backup_meta_key();
		$query = new \WP_Query([
			'post_type'      => 'attachment',
			'post_status'    => 'inherit',
			'fields'         => 'ids',
			'posts_per_page' => -1,
			'no_found_rows'  => true,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Preparing a restore job by scanning a single meta key; admin-only, minimal fields, no_found_rows.
			'meta_query'     => [
				[
					'key'     => $meta_key,
					'compare' => 'EXISTS',
				],
			],
		]);
		$valid_ids = [];
		foreach ( (array) $query->posts as $id ) {
			$path = get_post_meta( $id, $meta_key, true );
			if ( $path && file_exists( $path ) ) {
				$valid_ids[] = (int)$id;
			} else {
				if ( $path ) { delete_post_meta( $id, $meta_key ); }
			}
		}
		if ( empty( $valid_ids ) ) {
			return [ 'success' => false, 'message' => __( 'No valid backups found to restore.', 'imgsmaller' ) ];
		}
		update_option( self::JOB_OPTION, [ 'ids' => $valid_ids, 'total' => count( $valid_ids ), 'processed' => 0, 'restored_success' => 0, 'failed_count' => 0, 'failed_ids' => [] ] );
		return [ 'success' => true, 'total' => count( $valid_ids ) ];
	}

	public function get_restore_job() : array {
		$job = get_option( self::JOB_OPTION, [] );

		if ( empty( $job ) || ! is_array( $job ) ) {
			return [];
		}

		if ( empty( $job['ids'] ) || ! is_array( $job['ids'] ) ) {
			$job['ids'] = [];
		}

		$job['ids']       = array_values( array_map( 'intval', $job['ids'] ) );
		$job['total']     = isset( $job['total'] ) ? (int) $job['total'] : count( $job['ids'] );
		$job['processed'] = isset( $job['processed'] ) ? (int) $job['processed'] : 0;
		$job['restored_success'] = isset( $job['restored_success'] ) ? (int) $job['restored_success'] : 0;
		$job['failed_count']     = isset( $job['failed_count'] ) ? (int) $job['failed_count'] : 0;
		$job['failed_ids']       = isset( $job['failed_ids'] ) && is_array( $job['failed_ids'] ) ? array_values( array_unique( array_map( 'intval', $job['failed_ids'] ) ) ) : [];

		return $job;
	}

	public static function get_backup_meta_key() : string { return self::META_BACKUP_PATH; }

	/**
	 * Prepare a restore job from an explicit list of attachment IDs.
	 * Filters IDs to those that exist, are images, and have a backup recorded.
	 */
	public function prepare_restore_job_from_ids( array $ids ) : array {
		$meta_key = self::get_backup_meta_key();
		$valid_ids = [];
		foreach ( $ids as $id ) {
			$id = (int) $id;
			if ( $id <= 0 ) { continue; }
			$path = get_post_meta( $id, $meta_key, true );
			if ( $path && file_exists( $path ) ) {
				$valid_ids[] = $id;
			} else {
				if ( $path ) { delete_post_meta( $id, $meta_key ); }
			}
		}
		$valid_ids = array_values( array_unique( $valid_ids ) );
		if ( empty( $valid_ids ) ) {
			return [ 'success' => false, 'message' => __( 'None of the selected items have an existing backup file to restore.', 'imgsmaller' ) ];
		}
		update_option( self::JOB_OPTION, [ 'ids' => $valid_ids, 'total' => count( $valid_ids ), 'processed' => 0, 'restored_success' => 0, 'failed_count' => 0, 'failed_ids' => [] ] );
		return [ 'success' => true, 'total' => count( $valid_ids ) ];
	}

	public function save_restore_job( array $job ) : void {
		update_option( self::JOB_OPTION, $job, false );
	}

	public function clear_restore_job() : void {
		delete_option( self::JOB_OPTION );
	}

	public function process_restore_batch( MediaRepository $repository, int $batch_size = 10 ) : array {
		$job = $this->get_restore_job();

		if ( empty( $job ) ) {
			$job = $this->prepare_restore_job();
		}

		$total = (int) ( $job['total'] ?? 0 );
		$processed_before = (int) ( $job['processed'] ?? 0 );

		if ( $total <= 0 || empty( $job['ids'] ) ) {
			$this->clear_restore_job();

			return [
				'restored'  => 0,
				'processed' => $processed_before,
				'total'     => $total,
				'remaining' => 0,
				'done'      => true,
			];
		}

		$batch_size = max( 1, min( 200, $batch_size ) );
		$chunk      = array_splice( $job['ids'], 0, $batch_size );
		$restored   = 0;
		$attempted  = 0;
		$failed     = [];

		foreach ( $chunk as $attachment_id ) {
			$attachment_id = (int) $attachment_id;
			if ( $attachment_id <= 0 ) {
				continue;
			}

			$attempted++;

			if ( $this->restore( $attachment_id ) ) {
				$repository->mark_status( $attachment_id, 'pending' );
				$repository->clear_progress( $attachment_id );
				$restored++;
			} else {
				// Instrumentation: capture context for failure diagnostics.
				$backup_path   = get_post_meta( $attachment_id, self::META_BACKUP_PATH, true );
				$original_meta = get_post_meta( $attachment_id, self::META_ORIGINAL_FILE, true );
				$attached_file = get_attached_file( $attachment_id );
				$exists        = ( $backup_path && file_exists( $backup_path ) ) ? 'yes' : 'no';
				$this->log(
					sprintf( 'Restore debug: ID %d failed. backup_path=%s exists=%s original_meta=%s attached_file=%s',
						$attachment_id,
						$backup_path ?: '[none]',
						$exists,
						$original_meta ?: '[none]',
						$attached_file ?: '[none]'
					),
					'error'
				);
				$failed[] = $attachment_id;
			}
		}

		$job['processed']        = min( $total, $processed_before + $attempted );
		$job['restored_success'] = min( $total, ( $job['restored_success'] ?? 0 ) + $restored );
		$job['failed_count']     = min( $total, ( $job['failed_count'] ?? 0 ) + count( $failed ) );
		$existing_failed_ids     = isset( $job['failed_ids'] ) && is_array( $job['failed_ids'] ) ? $job['failed_ids'] : [];
		$job['failed_ids']       = array_values( array_unique( array_merge( $existing_failed_ids, $failed ) ) );

		$remaining = max( 0, $total - $job['processed'] );
		$done      = $remaining <= 0 || empty( $job['ids'] );

		if ( $done ) {
			$this->clear_restore_job();
		} else {
			$this->save_restore_job( $job );
		}

		return [
			'restored'      => (int) $job['restored_success'],
			'processed'     => (int) $job['processed'],
			'attempted'     => (int) $job['processed'],
			'just_restored' => $restored,
			'just_failed'   => $failed,
			'failed_total'  => (int) $job['failed_count'],
			'total'         => $total,
			'remaining'     => $remaining,
			'done'          => $done,
			'failed_ids'    => $job['failed_ids'],
		];
	}

	private function backup_dir_for( int $attachment_id ) : string {
		return trailingslashit( $this->backup_base_dir() . $attachment_id );
	}

}
