<?php

declare( strict_types=1 );

namespace Imgspalat\Rest;

use Imgspalat\MediaRepository;
use Imgspalat\Services\CompressionService;
use Imgspalat\Services\BackupManager;
use Imgspalat\Settings;
use Imgspalat\StatusStore;
use Imgspalat\Tasks\BatchProcessor;
use function __;
use function _n;

class Controller {
    private Settings $settings;

    private StatusStore $status;

    private BatchProcessor $batch_processor;

    private CompressionService $compression_service;

    private BackupManager $backup_manager;

    public function __construct( Settings $settings, StatusStore $status, BatchProcessor $batch_processor ) {
        $this->settings            = $settings;
        $this->status              = $status;
        $this->batch_processor     = $batch_processor;
        $this->compression_service = new CompressionService( $settings, $status );
        $this->backup_manager      = new BackupManager( $settings, $status );
    }

    public function register_routes() : void {
        add_action( 'rest_api_init', function () : void {
            register_rest_route(
                'imgsmaller/v1',
                '/status',
                [
                    'methods'             => 'GET',
                    'callback'            => [ $this, 'rest_status' ],
                    'permission_callback' => function () {
                        return current_user_can( 'manage_options' );
                    },
                ]
            );

            // Public cron endpoint guarded by token; triggers one batch run.
            register_rest_route(
                'imgsmaller/v1',
                '/cron',
                [
                    'methods'             => 'GET',
                    'callback'            => [ $this, 'rest_cron' ],
                    'permission_callback' => '__return_true',
                ]
            );

            // Public restore endpoint guarded by token; triggers one restore step.
            register_rest_route(
                'imgsmaller/v1',
                '/restore',
                [
                    'methods'             => 'GET',
                    'callback'            => [ $this, 'rest_restore' ],
                    'permission_callback' => '__return_true',
                ]
            );

            // Public file proxy endpoint guarded by token; streams attachment bytes
            register_rest_route(
                'imgsmaller/v1',
                '/file',
                [
                    'methods'             => 'GET',
                    'callback'            => [ $this, 'rest_file_proxy' ],
                    'permission_callback' => '__return_true',
                ]
            );
        } );
        // Register admin-ajax handlers for plan info and allowed domains
        add_action( 'wp_ajax_imgsmaller_plan_info', [ $this, 'handle_plan_info' ] );
        add_action( 'wp_ajax_imgsmaller_set_domain', [ $this, 'handle_set_domain' ] );
        // Onboarding tour state
        add_action( 'wp_ajax_imgsmaller_tour_dismiss', [ $this, 'handle_tour_dismiss' ] );
        add_action( 'wp_ajax_imgsmaller_tour_complete', [ $this, 'handle_tour_complete' ] );
    // Failed items list
    add_action( 'wp_ajax_imgsmaller_failed_list', [ $this, 'handle_failed_list' ] );
    }

    public function handle_tour_dismiss() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }
        $s = $this->settings->all();
        $s['onboarding_dismissed'] = true;
        $this->settings->replace( $s );
        wp_send_json_success( [ 'settings' => $this->settings->all() ] );
    }

    public function handle_tour_complete() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }
        $s = $this->settings->all();
        $s['onboarding_completed'] = true;
        $this->settings->replace( $s );
        wp_send_json_success( [ 'settings' => $this->settings->all() ] );
    }

    /**
     * Extract bearer token from request query or headers.
     * Supports: token, cron_token, t query params; Authorization: Bearer <token>; X-ImgSmaller-Token header.
     */
    private function extract_request_token( \WP_REST_Request $request ) : string {
        $token = (string) ( $request->get_param( 'token' ) ?? '' );
        if ( '' === $token ) {
            $token = (string) ( $request->get_param( 'cron_token' ) ?? '' );
        }
        if ( '' === $token ) {
            $token = (string) ( $request->get_param( 't' ) ?? '' );
        }

        if ( '' === $token ) {
            $auth = (string) ( $request->get_header( 'authorization' ) ?? $request->get_header( 'Authorization' ) ?? '' );
            if ( $auth && stripos( $auth, 'Bearer ' ) === 0 ) {
                $token = substr( $auth, 7 );
            }
        }

        if ( '' === $token ) {
            $hdr = (string) ( $request->get_header( 'x-imgsmaller-token' ) ?? $request->get_header( 'X-ImgSmaller-Token' ) ?? '' );
            if ( $hdr ) {
                $token = $hdr;
            }
        }

        return trim( $token );
    }

    public function cron_restore_step() : void {
        try {
            $repository = new MediaRepository();
            $chunk = $this->backup_manager->process_restore_batch( $repository, 200 );
            $done = ! empty( $chunk['done'] );
            if ( ! $done ) {
                wp_schedule_single_event( time() + 60, IMGSMALLER_RESTORE_CRON_HOOK );
            } else {
                $this->status->update(
                    [
                        'restore_in_progress' => false,
                        'restore_message'     => __( 'Restore complete.', 'imgsmaller' ),
                    ]
                );
            }
        } catch ( \Throwable $e ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Restore cron step failed: %s', 'imgsmaller' ), $e->getMessage() ), 'error' );
        }
    }

    public function rest_cron( \WP_REST_Request $request ) : \WP_REST_Response {
        $token = $this->extract_request_token( $request );
        $expected = (string) $this->settings->get_cron_token();

        if ( '' === $token || $token !== $expected ) {
            return new \WP_REST_Response( [ 'ok' => false, 'message' => __( 'Invalid or missing token.', 'imgsmaller' ) ], 403 );
        }

        if ( $this->status->is_paused() ) {
            return new \WP_REST_Response( [ 'ok' => false, 'message' => __( 'Compression is paused.', 'imgsmaller' ) ], 200 );
        }

        try {
            $this->batch_processor->handle_cron();
        } catch ( \Throwable $exception ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Cron endpoint failed: %s', 'imgsmaller' ), $exception->getMessage() ), 'error' );
            return new \WP_REST_Response( [ 'ok' => false, 'message' => $exception->getMessage() ], 500 );
        }

        return new \WP_REST_Response(
            [
                'ok'      => true,
                'message' => __( 'Batch processed.', 'imgsmaller' ),
                'status'  => $this->status->all(),
            ],
            200
        );
    }

    public function rest_restore( \WP_REST_Request $request ) : \WP_REST_Response {
        $token = $this->extract_request_token( $request );
        $expected = (string) $this->settings->get_cron_token();

        if ( '' === $token || $token !== $expected ) {
            return new \WP_REST_Response( [ 'ok' => false, 'message' => __( 'Invalid or missing token.', 'imgsmaller' ) ], 403 );
        }

        $time_limit = (int) $request->get_param( 'time' );
        $time_limit = $time_limit > 0 ? min( 20, max( 3, $time_limit ) ) : 8;

        try {
            $repository = new MediaRepository();
            $job        = $this->backup_manager->prepare_restore_job();
            $total      = (int) ( $job['total'] ?? 0 );

            if ( $total <= 0 ) {
                $this->backup_manager->clear_restore_job();
                $this->status->update(
                    [
                        'restore_in_progress' => false,
                        'restore_total'       => 0,
                        'restore_restored'    => 0,
                        'restore_attempted'   => 0,
                        'restore_failed'      => 0,
                        'restore_message'     => __( 'No backups found to restore.', 'imgsmaller' ),
                    ]
                );

                return new \WP_REST_Response(
                    [
                        'ok'      => true,
                        'done'    => true,
                        'message' => $this->status->get( 'restore_message', '' ),
                        'total'   => 0,
                        'status'  => $this->status->all(),
                    ],
                    200
                );
            }

            $this->status->update( [ 'restore_in_progress' => true, 'restore_total' => $total ] );

            $start      = microtime( true );
            $done       = false;
            $restored   = 0;
            $processed  = 0;
            $failed     = 0;

            while ( ! $done ) {
                $chunk = $this->backup_manager->process_restore_batch( $repository, 200 );
                $restored  = (int) ( $chunk['restored'] ?? $restored );
                $processed = (int) ( $chunk['processed'] ?? $processed );
                $failed    = (int) ( $chunk['failed_total'] ?? $failed );
                $done      = ! empty( $chunk['done'] );

                if ( ( microtime( true ) - $start ) >= $time_limit ) {
                    break;
                }
            }

            $remaining = max( 0, $total - $processed );

            if ( $done ) {
                $message = __( 'Restore complete.', 'imgsmaller' );
                if ( $failed > 0 ) {
                    /* translators: %d: number of images failed to restore */
                    $message = sprintf( _n( 'Restore finished with %d image that could not be restored.', 'Restore finished with %d images that could not be restored.', $failed, 'imgsmaller' ), $failed );
                }

                $this->status->update(
                    [
                        'restore_in_progress' => false,
                        'restore_total'       => $total,
                        'restore_restored'    => $restored,
                        'restore_attempted'   => $processed,
                        'restore_failed'      => $failed,
                        'restore_message'     => $message,
                    ]
                );

                return new \WP_REST_Response(
                    [
                        'ok'        => true,
                        'done'      => true,
                        'message'   => $this->status->get( 'restore_message', '' ),
                        'restored'  => $restored,
                        'processed' => $processed,
                        'failed'    => $failed,
                        'total'     => $total,
                        'remaining' => $remaining,
                        'status'    => $this->status->all(),
                    ],
                    200
                );
            }

            /* translators: 1: processed items, 2: total items */
            $this->status->update( [ 'restore_attempted' => $processed, 'restore_failed' => $failed, 'restore_message' => sprintf( __( 'Restoring… %1$d/%2$d processed', 'imgsmaller' ), $processed, $total ) ] );

            return new \WP_REST_Response(
                [
                    'ok'        => true,
                    'done'      => false,
                    'message'   => $this->status->get( 'restore_message', '' ),
                    'restored'  => $restored,
                    'processed' => $processed,
                    'failed'    => $failed,
                    'total'     => $total,
                    'remaining' => $remaining,
                    'status'    => $this->status->all(),
                ],
                200
            );
        } catch ( \Throwable $exception ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Restore endpoint failed: %s', 'imgsmaller' ), $exception->getMessage() ), 'error' );
            return new \WP_REST_Response( [ 'ok' => false, 'message' => $exception->getMessage() ], 500 );
        }
    }

    public function handle_status() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }
        
        wp_send_json_success(
            [
                'settings' => $this->settings->all(),
                'status'   => $this->status->all(),
                'logs'     => $this->status->logs(),
            ]
        );
    }

    public function handle_start() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $api_key = (string) $this->settings->get( 'api_key', '' );
        if ( '' === $api_key ) {
            $this->status->add_log( __( 'Compression start aborted: API key not configured.', 'imgsmaller' ), 'error' );
            wp_send_json_error(
                [
                    'message' => __( 'Please add your ImgSmaller API key before starting compression.', 'imgsmaller' ),
                ]
            );
        }

        if ( $this->status->is_paused() ) {
            $this->status->resume();
            $this->status->add_log( __( 'Compression resumed automatically when starting a new batch.', 'imgsmaller' ) );
        }

        $repository     = new MediaRepository();
        $total_images   = $repository->count_all_processable();
        $pending_sample = $repository->next_batch( 1 );

        if ( empty( $pending_sample ) ) {
            /* translators: %d: number of processable images */
            $msg = __( 'Compression start aborted: no eligible images found. Total processable images: %d', 'imgsmaller' );
            $this->status->add_log( sprintf( $msg, $total_images ), 'warning' );
            
            // If we have processable images but no pending ones, they might all be processed
            if ( $total_images > 0 ) {
                wp_send_json_error(
                    [
                        'message' => __( 'All images appear to have been processed already. Use "Restore Originals" to reset and reprocess.', 'imgsmaller' ),
                    ]
                );
            } else {
                wp_send_json_error(
                    [
                        'message' => __( 'No processable image files found in your media library. Please upload some JPEG, PNG, WebP, or AVIF images first.', 'imgsmaller' ),
                    ]
                );
            }
        }

        wp_schedule_single_event( time() + 5, IMGSMALLER_CRON_HOOK );

        // Execute one batch immediately so users see progress without waiting for cron.
        try {
            // Clear a potentially stale lock so the manual trigger can proceed.
            if ( function_exists( 'delete_transient' ) ) {
                delete_transient( 'imgsmaller_batch_lock' );
            }
            $this->batch_processor->handle_cron();
        } catch ( \Throwable $exception ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Immediate batch failed: %s', 'imgsmaller' ), $exception->getMessage() ), 'error' );
        }

        wp_send_json_success(
            [
                'message' => __( 'Compression batch triggered.', 'imgsmaller' ),
                'status'  => $this->status->all(),
                'logs'    => $this->status->logs(),
            ]
        );
    }

    public function handle_pause() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $this->status->pause();
        $this->status->add_log( __( 'Compression paused from dashboard.', 'imgsmaller' ), 'warning' );

        if ( function_exists( 'wp_clear_scheduled_hook' ) ) {
            wp_clear_scheduled_hook( IMGSMALLER_CRON_HOOK );
        }

        if ( function_exists( 'delete_transient' ) ) {
            delete_transient( 'imgsmaller_batch_lock' );
        }

        wp_send_json_success(
            [
                'paused'  => true,
                'issues'  => $this->status->get( 'last_error', '' ),
                'logs'    => $this->status->logs(),
                'status'  => $this->status->all(),
                'settings'=> $this->settings->all(),
            ]
        );
    }

    public function handle_resume() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $was_paused = $this->status->is_paused();
        $this->status->resume();

        if ( $was_paused ) {
            $this->status->add_log( __( 'Compression resumed from dashboard.', 'imgsmaller' ), 'info' );
        }

        if ( function_exists( 'wp_schedule_single_event' ) ) {
            wp_schedule_single_event( time() + 5, IMGSMALLER_CRON_HOOK );
        }

        if ( function_exists( 'delete_transient' ) ) {
            delete_transient( 'imgsmaller_batch_lock' );
        }

        try {
            $this->batch_processor->handle_cron();
        } catch ( \Throwable $exception ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Resume trigger failed to process immediately: %s', 'imgsmaller' ), $exception->getMessage() ), 'error' );
        }

        wp_send_json_success(
            [
                'paused'   => false,
                'logs'     => $this->status->logs(),
                'status'   => $this->status->all(),
                'settings' => $this->settings->all(),
            ]
        );
    }

    public function handle_restore() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        try {
            $repository = new MediaRepository();
            $job        = $this->backup_manager->prepare_restore_job();
            $total      = (int) ( $job['total'] ?? 0 );

            if ( $total <= 0 ) {
                $this->backup_manager->clear_restore_job();
                $this->status->update(
                    [
                        'restore_in_progress' => false,
                        'restore_total'       => 0,
                        'restore_restored'    => 0,
                        'restore_attempted'   => 0,
                        'restore_failed'      => 0,
                        'restore_message'     => __( 'No backups found to restore.', 'imgsmaller' ),
                    ]
                );
                $this->status->add_log( __( 'Restore skipped because no backups were available.', 'imgsmaller' ), 'warning' );

                wp_send_json_success(
                    [
                        'restored'  => 0,
                        'processed' => 0,
                        'attempted' => 0,
                        'failed_total' => 0,
                        'total'     => 0,
                        'remaining' => 0,
                        'done'      => true,
                        'message'   => __( 'No backups found to restore.', 'imgsmaller' ),
                        'status'    => $this->status->all(),
                        'logs'      => $this->status->logs(),
                    ]
                );
            }

            $time_limit = isset( $_POST['time_limit'] ) ? max( 3, min( 20, (int) $_POST['time_limit'] ) ) : 8;
            $start      = microtime( true );
            $done       = false;
            $restored   = 0;
            $processed  = 0;
            $failed_total = 0;
            $failed_ids = [];

            $this->status->update( [ 'restore_in_progress' => true, 'restore_total' => $total ] );

            while ( ! $done ) {
                $chunk = $this->backup_manager->process_restore_batch( $repository, 200 );
                $restored     = (int) ( $chunk['restored'] ?? $restored );
                $processed    = (int) ( $chunk['processed'] ?? $processed );
                $failed_total = (int) ( $chunk['failed_total'] ?? $failed_total );
                if ( ! empty( $chunk['just_failed'] ) && is_array( $chunk['just_failed'] ) ) {
                    $failed_ids = array_merge( $failed_ids, array_map( 'intval', $chunk['just_failed'] ) );
                }
                $done = ! empty( $chunk['done'] );

                if ( ( microtime( true ) - $start ) >= $time_limit ) {
                    break; // return partial progress; UI will re-call until done
                }
            }

            $restored     = min( $restored, $total );
            $processed    = min( $processed, $total );
            $failed_total = min( $failed_total, $total );
            $failed_ids   = array_values( array_unique( $failed_ids ) );
            $remaining    = max( 0, $total - $processed );

            if ( $done ) {
                $message = __( 'Restore complete.', 'imgsmaller' );
                if ( $failed_total > 0 ) {
                    /* translators: %d: number of images failed to restore */
                    $fail_tpl = _n( 'Restore finished with %d image that could not be restored.', 'Restore finished with %d images that could not be restored.', $failed_total, 'imgsmaller' );
                    $message  = sprintf( $fail_tpl, $failed_total );
                }

                $this->status->update(
                    [
                        'restore_in_progress' => false,
                        'restore_total'       => $total,
                        'restore_restored'    => $restored,
                        'restore_attempted'   => $processed,
                        'restore_failed'      => $failed_total,
                        'restore_message'     => $message,
                    ]
                );

                $this->status->update(
                    [
                        'fetched_count'     => 0,
                        'sent_count'        => 0,
                        'compressed_count'  => 0,
                        'queued_count'      => 0,
                        'in_progress_count' => 0,
                    ]
                );
                $this->status->clear_error();

                /* translators: 1: number restored, 2: number failed */
                $restore_completed_tpl = __( 'Restore completed. %1$d restored, %2$d failed.', 'imgsmaller' );
                $this->status->add_log(
                    sprintf( $restore_completed_tpl, $restored, $failed_total ),
                    $failed_total > 0 ? 'warning' : 'success',
                    [
                        'restored' => $restored,
                        'failed'   => $failed_total,
                    ]
                );
                $message_out = $message;
            } else {
                /* translators: 1: processed items, 2: total items */
                $message_out = sprintf( __( 'Restoring… %1$d/%2$d processed', 'imgsmaller' ), $processed, $total );
                // schedule next step so progress continues in background even if UI is closed
                wp_schedule_single_event( time() + 30, IMGSMALLER_RESTORE_CRON_HOOK );
            }

            wp_send_json_success(
                [
                    'restored'      => $restored,
                    'processed'     => $processed,
                    'attempted'     => $processed,
                    'failed_total'  => $failed_total,
                    'total'         => $total,
                    'remaining'     => $remaining,
                    'done'          => (bool) $done,
                    'failed_ids'    => $failed_ids,
                    'message'       => $message_out,
                    'status'        => $this->status->all(),
                    'logs'          => $this->status->logs(),
                ]
            );
        } catch ( \Throwable $e ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Restore failed: %s', 'imgsmaller' ), $e->getMessage() ), 'error' );
            wp_send_json_error( [ 'message' => $e->getMessage() ] );
        }
    }

    public function handle_restore_step() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        try {
            $repository = new MediaRepository();
            $job        = $this->backup_manager->prepare_restore_job();
            $total      = (int) ( $job['total'] ?? 0 );

            if ( $total <= 0 ) {
                $this->backup_manager->clear_restore_job();
                $this->status->update(
                    [
                        'restore_in_progress' => false,
                        'restore_total'       => 0,
                        'restore_restored'    => 0,
                        'restore_attempted'   => 0,
                        'restore_failed'      => 0,
                        'restore_message'     => __( 'No backups found to restore.', 'imgsmaller' ),
                    ]
                );

                wp_send_json_success(
                    [
                        'done'      => true,
                        'processed' => 0,
                        'restored'  => 0,
                        'failed'    => 0,
                        'total'     => 0,
                        'message'   => __( 'No backups found to restore.', 'imgsmaller' ),
                        'status'    => $this->status->all(),
                        'logs'      => $this->status->logs(),
                    ]
                );
            }

            // Process a single batch (no loop/time budget)
            $chunk      = $this->backup_manager->process_restore_batch( $repository, 200 );
            $restored   = (int) ( $chunk['just_restored'] ?? 0 );
            $processed  = (int) ( $chunk['processed'] ?? 0 );
            $failed     = (int) ( $chunk['failed_total'] ?? 0 );
            $done       = ! empty( $chunk['done'] );
            $remaining  = max( 0, $total - $processed );

            if ( $done ) {
                $message = __( 'Restore complete.', 'imgsmaller' );
                if ( $failed > 0 ) {
                    /* translators: %d: number of images failed to restore */
                    $message = sprintf( _n( 'Restore finished with %d image that could not be restored.', 'Restore finished with %d images that could not be restored.', $failed, 'imgsmaller' ), $failed );
                }
                $this->status->update(
                    [
                        'restore_in_progress' => false,
                        'restore_total'       => $total,
                        'restore_restored'    => (int) ( $chunk['restored'] ?? 0 ),
                        'restore_attempted'   => $processed,
                        'restore_failed'      => $failed,
                        'restore_message'     => $message,
                    ]
                );
            } else {
                $this->status->update(
                    [
                        'restore_in_progress' => true,
                        'restore_total'       => $total,
                        'restore_attempted'   => $processed,
                        'restore_failed'      => $failed,
                        /* translators: 1: processed items, 2: total items */
                        'restore_message'     => sprintf( __( 'Restoring… %1$d/%2$d processed', 'imgsmaller' ), $processed, $total ),
                    ]
                );
            }

            wp_send_json_success(
                [
                    'done'      => (bool) $done,
                    'restored'  => $restored,
                    'processed' => $processed,
                    'failed'    => $failed,
                    'remaining' => $remaining,
                    'total'     => $total,
                    'message'   => $this->status->get( 'restore_message', '' ),
                    'status'    => $this->status->all(),
                    'logs'      => $this->status->logs(),
                ]
            );
        } catch ( \Throwable $e ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Restore step failed: %s', 'imgsmaller' ), $e->getMessage() ), 'error' );
            wp_send_json_error( [ 'message' => $e->getMessage() ] );
        }
    }

    public function handle_process_now() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        if ( $this->status->is_paused() ) {
            wp_send_json_error( [ 'message' => __( 'Compression is currently paused.', 'imgsmaller' ) ] );
        }

    $cycles = isset( $_POST['cycles'] ) ? absint( wp_unslash( $_POST['cycles'] ) ) : 2;
        $cycles = max( 1, min( 5, $cycles ) );
        $time_limit = isset( $_POST['time_limit'] ) ? absint( wp_unslash( $_POST['time_limit'] ) ) : 10;
        $time_limit = max( 3, min( 30, $time_limit ) );

        $start      = microtime( true );
        $executed   = 0;
        $errors     = [];

        // Clear a potentially stale batch lock before manual processing
        if ( function_exists( 'delete_transient' ) ) {
            delete_transient( 'imgsmaller_batch_lock' );
        }

        while ( $executed < $cycles ) {
            $unlock = isset( $_POST['unlock'] ) ? sanitize_text_field( wp_unslash( (string) $_POST['unlock'] ) ) : '';
            if ( '1' === $unlock && function_exists( 'delete_transient' ) ) {
                delete_transient( 'imgsmaller_batch_lock' );
            }

            try {
                $this->batch_processor->handle_cron();
            } catch ( \Throwable $exception ) {
                $errors[] = $exception->getMessage();
                /* translators: %s: error message */
                $this->status->add_log( sprintf( __( 'Manual batch run failed: %s', 'imgsmaller' ), $exception->getMessage() ), 'error' );
                break;
            }

            $executed++;

            if ( ( microtime( true ) - $start ) >= $time_limit ) {
                break;
            }

            $status_snapshot = $this->status->all();
            $queued          = (int) ( $status_snapshot['queued_count'] ?? 0 );
            $in_progress     = (int) ( $status_snapshot['in_progress_count'] ?? 0 );
            $remaining       = (int) ( $status_snapshot['remaining_count'] ?? 0 );

            if ( $queued <= 0 && $in_progress <= 0 && $remaining <= 0 ) {
                break;
            }
        }

        if ( $executed > 0 ) {
            /* translators: %d: number of executed cycles */
            $this->status->add_log( sprintf( __( 'Manual batch processed %d cycle(s).', 'imgsmaller' ), $executed ), ! empty( $errors ) ? 'warning' : 'info', [ 'errors' => $errors ] );
        }

        $response = [
            'runs'        => $executed,
            'errors'      => $errors,
            'status'      => $this->status->all(),
            'settings'    => $this->settings->all(),
            'logs'        => $this->status->logs(),
        ];

        wp_send_json_success( $response );
    }

    public function handle_test_connection() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $connected = $this->compression_service->test_connection();
        $this->settings->mark_connection_status( $connected );

        wp_send_json_success(
            [
                'connected' => $connected,
            ]
        );
    }

    public function handle_cancel_restore() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        try {
            $this->backup_manager->clear_restore_job();
            if ( function_exists( 'wp_clear_scheduled_hook' ) ) {
                wp_clear_scheduled_hook( IMGSMALLER_RESTORE_CRON_HOOK );
            }
            $this->status->update(
                [
                    'restore_in_progress' => false,
                    'restore_message'     => __( 'Restore cancelled by user.', 'imgsmaller' ),
                ]
            );

            $this->status->add_log( __( 'Restore cancelled from dashboard.', 'imgsmaller' ), 'warning' );

            wp_send_json_success(
                [
                    'status' => $this->status->all(),
                    'logs'   => $this->status->logs(),
                ]
            );
        } catch ( \Throwable $e ) {
            wp_send_json_error( [ 'message' => $e->getMessage() ] );
        }
    }

    public function handle_debug_media() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $repository = new MediaRepository();
        
        $debug_info = [
            'total_processable' => $repository->count_all_processable(),
            'next_batch_1' => $repository->next_batch( 1 ),
            'next_batch_10' => $repository->next_batch( 10 ),
            'queued_count' => $repository->count_by_status( [ 'queued' ] ),
            'processing_count' => $repository->count_by_status( [ 'processing' ] ),
            'done_count' => $repository->count_by_status( [ 'done' ] ),
            'failed_count' => $repository->count_by_status( [ 'failed' ] ),
            'pending_count' => $repository->count_by_status( [ 'pending' ] ),
        ];
        
        // Sample query to see what's in the database
        $sample_query = new \WP_Query([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'fields'         => 'ids',
            'posts_per_page' => 5,
            'post_mime_type' => [ 'image/jpeg', 'image/png', 'image/webp', 'image/avif' ],
        ]);
        
        $debug_info['sample_attachments'] = array_map('intval', (array) $sample_query->posts);
        $debug_info['total_attachments'] = $sample_query->found_posts;

        wp_send_json_success( $debug_info );
    }

    public function handle_failed_list() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $page = isset( $_POST['page'] ) ? max( 1, (int) $_POST['page'] ) : 1;
        $per  = isset( $_POST['per'] ) ? max( 1, min( 50, (int) $_POST['per'] ) ) : 10;

        $repo = new \Imgspalat\MediaRepository();
        $data = $repo->get_failed( $page, $per );

        $items = [];
        foreach ( $data['ids'] as $id ) {
            $thumb = wp_get_attachment_image_url( (int) $id, 'thumbnail' );
            $items[] = [
                'id'    => (int) $id,
                'title' => get_the_title( (int) $id ),
                'url'   => $thumb ?: '',
            ];
        }

        wp_send_json_success( [
            'items'      => $items,
            'total'      => (int) $data['total'],
            'totalPages' => (int) $data['totalPages'],
            'page'       => $page,
            'per'        => $per,
        ] );
    }

    private function api_get( string $path ) {
        $api_key = (string) $this->settings->get( 'api_key', '' );
        if ( '' === $api_key ) {
            return new \WP_Error( 'imgsmaller_api_key_missing', __( 'API key not configured.', 'imgsmaller' ) );
        }
        $url = 'https://imgsmaller.com' . $path;
        $res = wp_remote_get( $url, [ 'timeout' => 15, 'headers' => [ 'X-API-Key' => $api_key ] ] );
        if ( is_wp_error( $res ) ) { return $res; }
        $code = (int) wp_remote_retrieve_response_code( $res );
        $body = json_decode( (string) wp_remote_retrieve_body( $res ), true );
        if ( 200 !== $code ) {
            return new \WP_Error( 'imgsmaller_http_' . $code, is_array( $body ) ? ( $body['message'] ?? ( $body['error'] ?? __( 'Unexpected response from API.', 'imgsmaller' ) ) ) : __( 'Unexpected response from API.', 'imgsmaller' ), [ 'status' => $code, 'body' => $body ] );
        }
        return is_array( $body ) ? $body : [];
    }

    private function api_post( string $path, array $payload ) {
        $api_key = (string) $this->settings->get( 'api_key', '' );
        if ( '' === $api_key ) {
            return new \WP_Error( 'imgsmaller_api_key_missing', __( 'API key not configured.', 'imgsmaller' ) );
        }
        $url = 'https://imgsmaller.com' . $path;
        $res = wp_remote_post( $url, [
            'timeout' => 20,
            'headers' => [ 'X-API-Key' => $api_key, 'Content-Type' => 'application/json' ],
            'body'    => wp_json_encode( $payload ),
        ] );
        if ( is_wp_error( $res ) ) { return $res; }
        $code = (int) wp_remote_retrieve_response_code( $res );
        $raw  = (string) wp_remote_retrieve_body( $res );
        $body = '' !== $raw ? json_decode( $raw, true ) : null;
        // Treat any 2xx as success; 204 may return no content
        if ( $code < 200 || $code >= 300 ) {
            return new \WP_Error( 'imgsmaller_http_' . $code, is_array( $body ) ? ( $body['message'] ?? ( $body['error'] ?? __( 'Unexpected response from API.', 'imgsmaller' ) ) ) : __( 'Unexpected response from API.', 'imgsmaller' ), [ 'status' => $code, 'body' => $body ] );
        }
        return is_array( $body ) ? $body : [];
    }

    private function normalize_domain( string $host ) : string {
        $host = strtolower( trim( $host ) );
        // Strip schema and path if accidentally included
        if ( false !== strpos( $host, '://' ) ) {
            $parsed = wp_parse_url( $host );
            $host = (string) ( $parsed['host'] ?? $host );
        }
        // Remove common prefixes
        if ( 0 === strpos( $host, 'www.' ) ) {
            $host = substr( $host, 4 );
        }
        $host = preg_replace( '/:\d+$/', '', $host ); // drop port
        // Attempt to reduce to registrable domain (eTLD+1) for common ccTLDs
        $parts = array_values( array_filter( explode( '.', $host ), static fn( $p ) => $p !== '' ) );
        $count = count( $parts );
        if ( $count <= 2 ) {
            return implode( '.', $parts );
        }
        $tld = $parts[ $count - 1 ];
        $sld = $parts[ $count - 2 ];
        $second_level_buckets = [ 'co', 'com', 'net', 'org', 'gov', 'ac', 'edu' ];
        if ( strlen( $tld ) === 2 && in_array( $sld, $second_level_buckets, true ) ) {
            // e.g. example.co.uk => keep last three labels
            return implode( '.', array_slice( $parts, -3 ) );
        }
        // default: keep last two labels
        return implode( '.', array_slice( $parts, -2 ) );
    }

    public function handle_plan_info() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $api_key = (string) $this->settings->get( 'api_key', '' );
        if ( '' === $api_key ) {
            wp_send_json_error( [ 'message' => __( 'Please enter your API key to fetch plan info.', 'imgsmaller' ) ] );
        }

    $plan   = $this->api_get( '/api/v1/plan' );
    $info   = $this->api_get( '/api/v1/info' );
    $allows = $this->api_get( '/api/v1/allowed-domains' );
    $plans  = $this->api_get( '/api/v1/plans' );

        if ( is_wp_error( $plan ) ) { wp_send_json_error( [ 'message' => $plan->get_error_message(), 'code' => $plan->get_error_code() ] ); }
        if ( is_wp_error( $info ) ) { wp_send_json_error( [ 'message' => $info->get_error_message(), 'code' => $info->get_error_code() ] ); }
    if ( is_wp_error( $allows ) ) { wp_send_json_error( [ 'message' => $allows->get_error_message(), 'code' => $allows->get_error_code() ] ); }
    if ( is_wp_error( $plans ) ) { wp_send_json_error( [ 'message' => $plans->get_error_message(), 'code' => $plans->get_error_code() ] ); }

    $site_url = site_url();
    $parsed   = wp_parse_url( $site_url );
    $host     = (string) ( $parsed['host'] ?? '' );
        $site_domain = $this->normalize_domain( $host );

        wp_send_json_success( [
            'plan'   => $plan,
            'plans'  => $plans,
            'info'   => $info,
            'allows' => $allows,
            'site'   => [ 'url' => $site_url, 'host' => $host, 'domain' => $site_domain ],
        ] );
    }

    public function handle_set_domain() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $domain = isset( $_POST['domain'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['domain'] ) ) : '';
        if ( '' === $domain ) {
            $parsed = wp_parse_url( site_url() );
            $host   = (string) ( $parsed['host'] ?? '' );
            $domain = $this->normalize_domain( $host );
        } else {
            $domain = $this->normalize_domain( $domain );
        }

        // Try primary payload shape
        $result = $this->api_post( '/api/v1/allowed-domains', [ 'domains' => [ $domain ] ] );
        if ( is_wp_error( $result ) ) {
            // Fallback to alternate payload shape if API expects single key
            $alt = $this->api_post( '/api/v1/allowed-domains', [ 'domain' => $domain ] );
            if ( is_wp_error( $alt ) ) {
                wp_send_json_error( [ 'message' => $result->get_error_message(), 'code' => $result->get_error_code() ] );
            }
        }

        // Re-fetch allowed domains to return authoritative list
        $allows = $this->api_get( '/api/v1/allowed-domains' );
        $domains = [];
        if ( is_array( $allows ) ) {
            if ( isset( $allows['domains'] ) && is_array( $allows['domains'] ) ) {
                $domains = array_values( array_unique( array_map( 'strval', $allows['domains'] ) ) );
            } elseif ( isset( $allows[0] ) ) {
                // Some APIs may return a plain array
                $domains = array_values( array_unique( array_map( 'strval', $allows ) ) );
            }
        }
        if ( empty( $domains ) ) {
            $domains = [ $domain ];
        }

        wp_send_json_success( [ 'message' => __( 'Allowed domains updated.', 'imgsmaller' ), 'domains' => $domains ] );
    }

    public function handle_media_search() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $search = isset( $_POST['q'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['q'] ) ) : '';
        $page   = isset( $_POST['page'] ) ? max( 1, (int) $_POST['page'] ) : 1;
        $per    = isset( $_POST['per'] ) ? max( 1, min( 50, (int) $_POST['per'] ) ) : 10;
        $size   = isset( $_POST['size'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['size'] ) ) : 'all';

        $mime_types = [ 'image/jpeg', 'image/png', 'image/webp', 'image/avif' ];
        $type  = isset( $_POST['type'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['type'] ) ) : 'all';

        $args = [
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'post_mime_type' => ( 'all' === $type ? $mime_types : $type ),
            'fields'         => 'ids',
            'posts_per_page' => $per,
            'paged'          => $page,
            'orderby'        => 'date',
            'order'          => 'DESC',
            's'              => $search,
            'no_found_rows'  => false,
        ];

        $q = new \WP_Query( $args );
        $ids = array_map( 'intval', (array) $q->posts );

        $selected = array_flip( $this->settings->get_excluded_ids() );
        $items = [];
        foreach ( $ids as $id ) {
            $thumb = wp_get_attachment_image_url( $id, 'thumbnail' );
            $path  = get_attached_file( $id );
            $bytes = ( $path && file_exists( $path ) ) ? @filesize( $path ) : false;
            $match = true;
            if ( false !== $bytes && is_int( $bytes ) ) {
                // size filter buckets
                if ( 'small' === $size ) {
                    $match = $bytes < 500 * 1024; // < 500KB
                } elseif ( 'medium' === $size ) {
                    $match = $bytes >= 500 * 1024 && $bytes <= 1024 * 1024; // 500KB–1MB
                } elseif ( 'large' === $size ) {
                    $match = $bytes > 1024 * 1024; // > 1MB
                }
            } else {
                // If no size available, only include in 'all'
                $match = ( 'all' === $size );
            }

            if ( ! $match ) {
                continue;
            }

            $items[] = [
                'id'         => $id,
                'title'      => get_the_title( $id ),
                'url'        => $thumb ?: '',
                'selected'   => isset( $selected[ $id ] ),
                'size_bytes' => ( false !== $bytes && is_int( $bytes ) ) ? $bytes : null,
                'size_h'     => ( false !== $bytes && is_int( $bytes ) ) ? size_format( $bytes ) : '',
            ];
        }

        wp_send_json_success(
            [
                'items'      => array_values( $items ),
                'page'       => $page,
                'per'        => $per,
                'total'      => (int) $q->found_posts,
                'totalPages' => (int) $q->max_num_pages,
            ]
        );
    }

    public function handle_restore_search() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

        $search = isset( $_POST['q'] ) ? sanitize_text_field( (string) wp_unslash( $_POST['q'] ) ) : '';
        $page   = isset( $_POST['page'] ) ? max( 1, (int) $_POST['page'] ) : 1;
        $per    = isset( $_POST['per'] ) ? max( 1, min( 50, (int) $_POST['per'] ) ) : 10;

        $meta_key = \Imgspalat\Services\BackupManager::get_backup_meta_key();

        $args = [
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'fields'         => 'ids',
            'posts_per_page' => $per * 2, // over-fetch to allow filtering of missing files
            'paged'          => $page,
            'orderby'        => 'date',
            'order'          => 'DESC',
            's'              => $search,
            'no_found_rows'  => false,
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Admin-only REST query with small pages; meta_query is required to find attachments having a backup entry.
            'meta_query'     => [
                [
                    'key'     => $meta_key,
                    'compare' => 'EXISTS',
                ],
            ],
        ];

        $q = new \WP_Query( $args );
        $raw_ids = array_map( 'intval', (array) $q->posts );
        $items = [];
        $seen = 0;
        foreach ( $raw_ids as $id ) {
            if ( count( $items ) >= $per ) { break; }
            $seen++;
            $backup_path = get_post_meta( $id, $meta_key, true );
            if ( ! $backup_path || ! file_exists( $backup_path ) ) {
                // stale meta, cleanup for future queries
                if ( $backup_path ) {
                    delete_post_meta( $id, $meta_key );
                }
                continue;
            }
            $thumb = wp_get_attachment_image_url( $id, 'thumbnail' );
            $items[] = [
                'id'    => $id,
                'title' => get_the_title( $id ),
                'url'   => $thumb ?: '',
            ];
        }

        wp_send_json_success(
            [
                'items'      => $items,
                'page'       => $page,
                'per'        => $per,
                'total'      => (int) $q->found_posts, // note: includes stale entries maybe being cleaned
                'totalPages' => (int) $q->max_num_pages,
                'filtered'   => count( $items ),
            ]
        );
    }

    public function handle_restore_selected() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Raw array is sanitized to integers immediately below.
    $ids_raw = isset( $_POST['ids'] ) ? (array) wp_unslash( $_POST['ids'] ) : [];
    $ids = array_values( array_unique( array_map( 'intval', $ids_raw ) ) );
        $ids = array_filter( $ids, static fn( $v ) => $v > 0 );

        if ( empty( $ids ) ) {
            wp_send_json_error( [ 'message' => __( 'No images selected.', 'imgsmaller' ) ] );
        }

        try {
            $job = $this->backup_manager->prepare_restore_job_from_ids( $ids );
            if ( empty( $job ) || ( isset( $job['success'] ) && false === $job['success'] ) ) {
                $msg = is_array( $job ) && isset( $job['message'] ) ? (string) $job['message'] : __( 'Selected items have no backups to restore.', 'imgsmaller' );
                wp_send_json_error( [ 'message' => $msg ] );
            }

            // Schedule background continuation
            if ( function_exists( 'wp_schedule_single_event' ) ) {
                wp_schedule_single_event( time() + 5, IMGSMALLER_RESTORE_CRON_HOOK );
            }

            // Kick one batch immediately for responsiveness
            $repository = new \Imgspalat\MediaRepository();
            $chunk      = $this->backup_manager->process_restore_batch( $repository, 200 );
            $done       = ! empty( $chunk['done'] );

            $this->status->update( [
                'restore_in_progress' => ! $done,
                'restore_total'       => (int) ( $chunk['total'] ?? $job['total'] ?? 0 ),
                'restore_attempted'   => (int) ( $chunk['processed'] ?? 0 ),
                'restore_failed'      => (int) ( $chunk['failed_total'] ?? 0 ),
                'restore_restored'    => (int) ( $chunk['restored'] ?? 0 ),
                'restore_message'     => $done ? __( 'Restore complete.', 'imgsmaller' ) : __( 'Selected restore started…', 'imgsmaller' ),
            ] );

            $this->status->add_log( __( 'Selected restore queued.', 'imgsmaller' ), 'info', [ 'count' => (int) ( $job['total'] ?? 0 ) ] );

            wp_send_json_success( [
                'started'   => true,
                'done'      => (bool) $done,
                'counts'    => [
                    'total'     => (int) ( $chunk['total'] ?? $job['total'] ?? 0 ),
                    'processed' => (int) ( $chunk['processed'] ?? 0 ),
                    'restored'  => (int) ( $chunk['restored'] ?? 0 ),
                    'failed'    => (int) ( $chunk['failed_total'] ?? 0 ),
                ],
                'logs'     => $this->status->logs(),
                'status'   => $this->status->all(),
                'message'  => $done ? __( 'Restore complete.', 'imgsmaller' ) : __( 'Selected restore started…', 'imgsmaller' ),
            ] );
        } catch ( \Throwable $e ) {
            /* translators: %s: error message */
            $this->status->add_log( sprintf( __( 'Restore selected failed: %s', 'imgsmaller' ), $e->getMessage() ), 'error' );
            wp_send_json_error( [ 'message' => $e->getMessage() ] );
        }
    }

    public function handle_scan() : void {
        if ( ! check_ajax_referer( 'imgsmaller_dashboard', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'imgsmaller' ) ] );
        }

    $time_limit = isset( $_POST['time_limit'] ) ? absint( wp_unslash( $_POST['time_limit'] ) ) : 8;
    $time_limit = max( 3, min( 20, $time_limit ) );
    $reset_param = isset( $_POST['reset'] ) ? wp_unslash( $_POST['reset'] ) : 0; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $reset       = absint( $reset_param ) === 1;

        $option_key = 'imgsmaller_scan_job';
        $job = get_option( $option_key, [] );
        if ( $reset || empty( $job ) || ! is_array( $job ) ) {
            $job = [
                'last_id'  => 0,
                'started'  => time(),
                'totals'   => [
                    'total_images'              => 0,
                    'total_bytes'               => 0,
                    'optimized_images'          => 0,
                    'optimized_bytes'           => 0,
                    'non_optimized_images'      => 0,
                    'non_optimized_bytes'       => 0,
                    'saved_bytes'               => 0,
                    'estimated_potential_bytes' => 0,
                ],
            ];
            update_option( $option_key, $job, false );
        }

        $allowed_mimes = [ 'image/jpeg', 'image/png', 'image/webp', 'image/avif' ];
        $batch_size    = 500;
        $start         = microtime( true );
        $done          = false;

        global $wpdb;
        if ( ! $wpdb ) {
            wp_send_json_error( [ 'message' => __( 'Database not available.', 'imgsmaller' ) ] );
        }

        while ( true ) {
            // Build a placeholder list for the IN() clause and compose the SQL string with only placeholders.
            $in_placeholders = implode( ', ', array_fill( 0, count( $allowed_mimes ), '%s' ) );
            $sql = sprintf(
                "SELECT ID, post_mime_type FROM {$wpdb->posts} WHERE post_type='attachment' AND post_status='inherit' AND post_mime_type IN (%s) AND ID > %%d ORDER BY ID ASC LIMIT %%d",
                $in_placeholders
            );
            $args = array_map( 'strval', $allowed_mimes );
            $args[] = (int) $job['last_id'];
            $args[] = (int) $batch_size;
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- SQL string uses placeholders only and is prepared with $args; $wpdb is used for efficient ID>... pagination; caching is intentionally avoided because the scan reflects real-time media changes in short loops.
            $rows = $wpdb->get_results( $wpdb->prepare( $sql, ...$args ) );

            if ( empty( $rows ) ) {
                $done = true;
                break;
            }

            foreach ( $rows as $row ) {
                $id   = (int) $row->ID;
                $mime = (string) $row->post_mime_type;
                $job['last_id'] = $id;

                $file = get_attached_file( $id );
                $bytes = ( $file && file_exists( $file ) ) ? (int) @filesize( $file ) : 0;
                $job['totals']['total_images']++;
                $job['totals']['total_bytes'] += max( 0, $bytes );

                $status = (string) get_post_meta( $id, \Imgspalat\MediaRepository::META_STATUS, true );
                $optimized = ( 'done' === $status );

                if ( $optimized ) {
                    $job['totals']['optimized_images']++;
                    $job['totals']['optimized_bytes'] += max( 0, $bytes );
                    $backup = (string) get_post_meta( $id, \Imgspalat\Services\BackupManager::get_backup_meta_key(), true );
                    if ( $backup && file_exists( $backup ) ) {
                        $orig = (int) @filesize( $backup );
                        if ( $orig > 0 && $orig > $bytes ) {
                            $job['totals']['saved_bytes'] += ( $orig - $bytes );
                        }
                    }
                } else {
                    $job['totals']['non_optimized_images']++;
                    $job['totals']['non_optimized_bytes'] += max( 0, $bytes );
                    // Rough potential savings by mime
                    $factor = 0.3; // default
                    if ( 'image/webp' === $mime ) { $factor = 0.1; }
                    elseif ( 'image/avif' === $mime ) { $factor = 0.05; }
                    $job['totals']['estimated_potential_bytes'] += (int) round( $bytes * $factor );
                }
            }

            update_option( $option_key, $job, false );

            if ( ( microtime( true ) - $start ) >= $time_limit ) {
                break;
            }
        }

        if ( $done ) {
            delete_option( $option_key );
        }

        // Mirror key totals into StatusStore for dashboard visibility
        $t = $job['totals'];
        $this->status->update( [
            'scan_total_images'         => (int) ( $t['total_images'] ?? 0 ),
            'scan_optimized_images'     => (int) ( $t['optimized_images'] ?? 0 ),
            'scan_non_optimized_images' => (int) ( $t['non_optimized_images'] ?? 0 ),
            'scan_total_bytes'          => (int) ( $t['total_bytes'] ?? 0 ),
            'scan_saved_bytes'          => (int) ( $t['saved_bytes'] ?? 0 ),
            'scan_potential_bytes'      => (int) ( $t['estimated_potential_bytes'] ?? 0 ),
        ] );

        wp_send_json_success( [
            'done'   => (bool) $done,
            'totals' => $job['totals'],
            'status' => $this->status->all(),
        ] );
    }

    public function rest_status() {
        return [
            'settings' => $this->settings->all(),
            'status'   => $this->status->all(),
            'logs'     => $this->status->logs(),
        ];
    }

    public function rest_file_proxy( \WP_REST_Request $request ) : \WP_REST_Response {
        $token = $this->extract_request_token( $request );
        $expected = (string) $this->settings->get_cron_token();
        if ( '' === $token || $token !== $expected ) {
            return new \WP_REST_Response( [ 'ok' => false, 'message' => __( 'Invalid or missing token.', 'imgsmaller' ) ], 403 );
        }

        $id = (int) $request->get_param( 'id' );
        if ( $id <= 0 ) {
            return new \WP_REST_Response( [ 'ok' => false, 'message' => __( 'Missing id parameter.', 'imgsmaller' ) ], 400 );
        }

        $path = get_attached_file( $id );
        if ( ! $path || ! file_exists( $path ) ) {
            return new \WP_REST_Response( [ 'ok' => false, 'message' => __( 'Attachment file not found.', 'imgsmaller' ) ], 404 );
        }

        $mime = function_exists( 'wp_check_filetype' ) ? ( wp_check_filetype( (string) $path )['type'] ?? 'application/octet-stream' ) : 'application/octet-stream';
        $size = @filesize( $path );

        if ( function_exists( 'nocache_headers' ) ) {
            nocache_headers();
        }

        header( 'Content-Type: ' . $mime );
        if ( $size && $size > 0 ) { header( 'Content-Length: ' . (string) $size ); }
        header( 'Content-Disposition: inline; filename="' . basename( $path ) . '"' );

    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile -- Streaming file contents directly; WP_Filesystem does not provide an equivalent streaming method.
        readfile( $path );
        // We already sent output
        exit;
    }
}
