<?php

declare( strict_types=1 );

namespace Imgspalat;

use Imgspalat\Admin\DashboardPage;
use Imgspalat\Rest\Controller as RestController;
use Imgspalat\Tasks\BatchProcessor;

class Plugin {
	private static ?Plugin $instance = null;

	private Settings $settings;

	private StatusStore $status_store;

	private BatchProcessor $batch_processor;

	private DashboardPage $dashboard_page;

	private RestController $rest_controller;

	private bool $booted = false;

	public static function instance() : Plugin {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	private function __construct() {}

	public function boot() : void {
		if ( $this->booted ) {
			return;
		}

		// Ensure we're running in WordPress context
		if ( ! function_exists( 'add_action' ) ) {
			return;
		}

		$this->settings       = new Settings();
		$this->status_store   = new StatusStore();
		$this->batch_processor = new BatchProcessor( $this->settings, $this->status_store );
		$this->dashboard_page = new DashboardPage( $this->settings, $this->status_store, $this->batch_processor );
		$this->rest_controller = new RestController( $this->settings, $this->status_store, $this->batch_processor );

	// Since WP 4.6, WordPress loads translations automatically for plugins hosted on WordPress.org.
	// No explicit load_plugin_textdomain call to satisfy Plugin Check.
		add_action( 'init', [ $this->status_store, 'maybe_bootstrap_totals' ] );
		add_action( 'admin_menu', [ $this->dashboard_page, 'register_menu_page' ] );
		add_action( 'admin_init', [ $this->dashboard_page, 'register_settings' ] );
		add_action( 'wp_ajax_imgsmaller_status', [ $this->rest_controller, 'handle_status' ] );
		add_action( 'wp_ajax_imgsmaller_start', [ $this->rest_controller, 'handle_start' ] );
		add_action( 'wp_ajax_imgsmaller_pause', [ $this->rest_controller, 'handle_pause' ] );
		add_action( 'wp_ajax_imgsmaller_resume', [ $this->rest_controller, 'handle_resume' ] );
		add_action( 'wp_ajax_imgsmaller_restore', [ $this->rest_controller, 'handle_restore' ] );
		add_action( 'wp_ajax_imgsmaller_restore_step', [ $this->rest_controller, 'handle_restore_step' ] );
		add_action( 'wp_ajax_imgsmaller_process_now', [ $this->rest_controller, 'handle_process_now' ] );
		add_action( 'wp_ajax_imgsmaller_test_connection', [ $this->rest_controller, 'handle_test_connection' ] );
		add_action( 'wp_ajax_imgsmaller_debug_media', [ $this->rest_controller, 'handle_debug_media' ] );
		add_action( 'wp_ajax_imgsmaller_media_search', [ $this->rest_controller, 'handle_media_search' ] );
		add_action( 'wp_ajax_imgsmaller_cancel_restore', [ $this->rest_controller, 'handle_cancel_restore' ] );
		add_action( 'wp_ajax_imgsmaller_restore_search', [ $this->rest_controller, 'handle_restore_search' ] );
		add_action( 'wp_ajax_imgsmaller_restore_selected', [ $this->rest_controller, 'handle_restore_selected' ] );
		add_action( 'wp_ajax_imgsmaller_scan', [ $this->rest_controller, 'handle_scan' ] );
		add_action( IMGSMALLER_CRON_HOOK, [ $this->batch_processor, 'handle_cron' ] );
		add_action( IMGSMALLER_RESTORE_CRON_HOOK, [ $this->rest_controller, 'cron_restore_step' ] );

		$this->rest_controller->register_routes();

		// Ensure a 5-minute cron interval exists for restore scheduling
		add_filter(
			'cron_schedules',
			static function ( array $schedules ) : array {
				if ( ! isset( $schedules['five_minutes'] ) ) {
					$schedules['five_minutes'] = [
						'interval' => 5 * 60,
						'display'  => __( 'Every Five Minutes', 'imgsmaller' ),
					];
				}

				return $schedules;
			}
		);

		$this->booted = true;
	}

	// Intentionally no register_text_domain(): WordPress auto-loads translations for .org plugins.

	public static function activate() : void {
		self::instance()->ensure_backup_directory();
		self::instance()->schedule_cron();
		self::instance()->schedule_restore_cron();
	}

	public static function deactivate() : void {
		self::instance()->unschedule_cron();
	}

	public static function uninstall() : void {
		$settings = new Settings();
		$status   = new StatusStore();

		$settings->delete_all();
		$status->delete();

		self::cleanup_backup_directory();
	}

	public function ensure_backup_directory() : void {
		$upload_dir = wp_upload_dir();
		$path       = trailingslashit( $upload_dir['basedir'] ) . 'imgsmaller-backups';

		if ( ! wp_mkdir_p( $path ) ) {
			wp_die( esc_html__( 'Failed to create ImgSmaller backup directory.', 'imgsmaller' ) );
		}

		update_option( 'imgsmaller_backup_dir', $path );
	}

	private static function cleanup_backup_directory() : void {
		$path = get_option( 'imgsmaller_backup_dir' );

		if ( ! $path || ! is_dir( $path ) ) {
			return;
		}

		self::rrmdir( $path );
		delete_option( 'imgsmaller_backup_dir' );
	}

	private static function rrmdir( string $dir ) : void {
		$items = array_diff( scandir( $dir ) ?: [], [ '.', '..' ] );

		foreach ( $items as $item ) {
			$path = trailingslashit( $dir ) . $item;

			if ( is_dir( $path ) ) {
				self::rrmdir( $path );
			} else {
				wp_delete_file( $path );
			}
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
		@rmdir( $dir );
	}

	private function schedule_cron() : void {
		if ( ! wp_next_scheduled( IMGSMALLER_CRON_HOOK ) ) {
			wp_schedule_event( time() + 60, 'hourly', IMGSMALLER_CRON_HOOK );
		}
	}

	private function schedule_restore_cron() : void {
		if ( ! wp_next_scheduled( IMGSMALLER_RESTORE_CRON_HOOK ) ) {
			wp_schedule_event( time() + 60, 'five_minutes', IMGSMALLER_RESTORE_CRON_HOOK );
		}
	}

	private function unschedule_cron() : void {
		$timestamp = wp_next_scheduled( IMGSMALLER_CRON_HOOK );

		while ( false !== $timestamp ) {
			wp_unschedule_event( $timestamp, IMGSMALLER_CRON_HOOK );
			$timestamp = wp_next_scheduled( IMGSMALLER_CRON_HOOK );
		}
	}

	private function unschedule_restore_cron() : void {
		$timestamp = wp_next_scheduled( IMGSMALLER_RESTORE_CRON_HOOK );

		while ( false !== $timestamp ) {
			wp_unschedule_event( $timestamp, IMGSMALLER_RESTORE_CRON_HOOK );
			$timestamp = wp_next_scheduled( IMGSMALLER_RESTORE_CRON_HOOK );
		}
	}

	public function settings() : Settings {
		return $this->settings;
	}

	public function status_store() : StatusStore {
		return $this->status_store;
	}

	public function batch_processor() : BatchProcessor {
		return $this->batch_processor;
	}
}
