<?php

declare( strict_types=1 );

namespace Imgspalat;

class MediaRepository {
	public const META_STATUS            = '_imgsmaller_status';
	public const META_PROGRESS_KEY      = '_imgsmaller_progress_key';
	public const META_RESULT_URL        = '_imgsmaller_result_url';
	public const META_DOWNLOAD_TEMPLATE = '_imgsmaller_download_template';
	public const META_POLL_ATTEMPTS     = '_imgsmaller_poll_attempts';
	public const META_RETRY_COUNT       = '_imgsmaller_retry_count';
	public const META_PERMANENT_FAIL    = '_imgsmaller_permanent_fail';

	private const SUPPORTED_MIMES = [
		'image/jpeg',
		'image/png',
		'image/webp',
		'image/avif',
	];

	public function next_batch( int $limit = 10 ) : array {
		if ( ! class_exists( 'WP_Query' ) ) {
			return [];
		}

		$settings = new \Imgspalat\Settings();
		$exclude = $settings->get_excluded_ids();

		$args = [
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => self::SUPPORTED_MIMES,
				'fields'         => 'ids',
				'posts_per_page' => $limit,
				'orderby'        => 'ID',
				'order'          => 'ASC',
				'no_found_rows'  => true,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- meta_query is required to select by plugin meta; query is tightly scoped and limited.
				'meta_query'     => [
					'relation' => 'AND',
					[
						'relation' => 'OR',
						[
							'key'     => self::META_STATUS,
							'compare' => 'NOT EXISTS',
						],
						[
							'key'     => self::META_STATUS,
							'value'   => [ 'pending', 'failed' ],
							'compare' => 'IN',
						],
					],
					[
						'key'     => self::META_PERMANENT_FAIL,
						'compare' => 'NOT EXISTS',
					],
				],
		];

		if ( ! empty( $exclude ) ) {
			$exclude = array_values( array_unique( array_map( 'intval', (array) $exclude ) ) );
			// Cap the exclusion list to minimize performance impact.
			$exclude = array_slice( $exclude, 0, 200 );
			// phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in -- Small, capped exclusion list for admin-only processing use case.
			$args['post__not_in'] = $exclude;
		}

		$query = new \WP_Query( $args );

		$results = array_map( 'intval', (array) $query->posts );
		
		// Optional: integrate your own debugging or logging here if needed.
		
		return $results;
	}

	public function is_first_time( int $attachment_id ) : bool {
		if ( ! function_exists( 'get_post_meta' ) ) {
			return false;
		}
		$meta = get_post_meta( $attachment_id, self::META_STATUS, true );
		return empty( $meta );
	}

	public function count_all_processable() : int {
		if ( ! class_exists( 'WP_Query' ) ) {
			return 0;
		}

		$settings = new \Imgspalat\Settings();
		$exclude = $settings->get_excluded_ids();

		$args = [
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => self::SUPPORTED_MIMES,
				'fields'         => 'ids',
				'posts_per_page' => 1,
				'no_found_rows'  => false,
		];

		if ( ! empty( $exclude ) ) {
			$exclude = array_values( array_unique( array_map( 'intval', (array) $exclude ) ) );
			$exclude = array_slice( $exclude, 0, 200 );
			// phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in -- Small, capped exclusion list for admin-only processing use case.
			$args['post__not_in'] = $exclude;
		}

		$query = new \WP_Query( $args );

		return (int) $query->found_posts;
	}

	/**
	 * Reset all image statuses to pending for debugging
	 */
	public function reset_all_statuses() : int {
		if ( ! class_exists( 'WP_Query' ) ) {
			return 0;
		}

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Reset operation needs to filter by existing plugin meta; scope is limited to attachments.
		$query = new \WP_Query(
			[
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => self::SUPPORTED_MIMES,
				'fields'         => 'ids',
				'posts_per_page' => -1,
				'no_found_rows'  => true,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Reset operation filters by plugin meta; admin-only and limited to attachments.
				'meta_query'     => [
					[
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Needed to filter by plugin status meta; query is limited to attachments with minimal fields.
						'key'     => self::META_STATUS,
						'compare' => 'EXISTS',
					],
				],
			]
		);

		$reset_count = 0;
		foreach ( (array) $query->posts as $attachment_id ) {
			$this->mark_status( (int) $attachment_id, 'pending' );
			$this->clear_progress( (int) $attachment_id );
			$this->reset_retry_count( (int) $attachment_id );
			$reset_count++;
		}

		return $reset_count;
	}

	public function in_progress_batch( int $limit = 10 ) : array {
		if ( ! class_exists( 'WP_Query' ) ) {
			return [];
		}

		$query = new \WP_Query(
			[
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => self::SUPPORTED_MIMES,
				'fields'         => 'ids',
				'posts_per_page' => $limit,
				'orderby'        => 'ID',
				'order'          => 'ASC',
				'no_found_rows'  => true,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Meta filter required to find queued/processing attachments; small, admin-only batches.
				'meta_query'     => [
					[
						'key'     => self::META_STATUS,
						'value'   => [ 'queued', 'processing' ],
						'compare' => 'IN',
					],
				],
			]
		);

		return array_map( 'intval', (array) $query->posts );
	}

	public function mark_status( int $attachment_id, string $status ) : void {
		if ( function_exists( 'update_post_meta' ) ) {
			update_post_meta( $attachment_id, self::META_STATUS, sanitize_text_field( $status ) );
		}
	}

	public function store_progress( int $attachment_id, array $data ) : void {
		if ( isset( $data['progress_key'] ) ) {
			if ( function_exists( 'update_post_meta' ) ) {
				update_post_meta( $attachment_id, self::META_PROGRESS_KEY, sanitize_text_field( (string) $data['progress_key'] ) );
			}
		}

		if ( isset( $data['result_url'] ) ) {
			if ( function_exists( 'update_post_meta' ) && function_exists( 'esc_url_raw' ) ) {
				update_post_meta( $attachment_id, self::META_RESULT_URL, esc_url_raw( (string) $data['result_url'] ) );
			}
		}

		if ( isset( $data['download_url_template'] ) ) {
			if ( function_exists( 'update_post_meta' ) && function_exists( 'esc_url_raw' ) ) {
				update_post_meta( $attachment_id, self::META_DOWNLOAD_TEMPLATE, esc_url_raw( (string) $data['download_url_template'] ) );
			}
		}
	}

	public function get_progress( int $attachment_id ) : array {
		return [
			'progress_key'          => function_exists( 'get_post_meta' ) ? get_post_meta( $attachment_id, self::META_PROGRESS_KEY, true ) : '',
			'result_url'            => function_exists( 'get_post_meta' ) ? get_post_meta( $attachment_id, self::META_RESULT_URL, true ) : '',
			'download_url_template' => function_exists( 'get_post_meta' ) ? get_post_meta( $attachment_id, self::META_DOWNLOAD_TEMPLATE, true ) : '',
		];
	}

	public function clear_progress( int $attachment_id ) : void {
		if ( function_exists( 'delete_post_meta' ) ) {
			delete_post_meta( $attachment_id, self::META_PROGRESS_KEY );
			delete_post_meta( $attachment_id, self::META_RESULT_URL );
			delete_post_meta( $attachment_id, self::META_DOWNLOAD_TEMPLATE );
			delete_post_meta( $attachment_id, self::META_POLL_ATTEMPTS );
		}
	}

	public function reset_retry_count( int $attachment_id ) : void {
		if ( function_exists( 'delete_post_meta' ) ) {
			delete_post_meta( $attachment_id, self::META_RETRY_COUNT );
		}
	}

	public function increment_retry_count( int $attachment_id ) : int {
		if ( ! function_exists( 'get_post_meta' ) || ! function_exists( 'update_post_meta' ) ) {
			return 0;
		}

		$current = (int) get_post_meta( $attachment_id, self::META_RETRY_COUNT, true );
		$current++;
		update_post_meta( $attachment_id, self::META_RETRY_COUNT, $current );

		return $current;
	}

	public function get_retry_count( int $attachment_id ) : int {
		if ( ! function_exists( 'get_post_meta' ) ) {
			return 0;
		}

		return (int) get_post_meta( $attachment_id, self::META_RETRY_COUNT, true );
	}

	public function increment_poll_attempts( int $attachment_id ) : int {
		if ( ! function_exists( 'get_post_meta' ) || ! function_exists( 'update_post_meta' ) ) {
			return 0;
		}

		$current = (int) get_post_meta( $attachment_id, self::META_POLL_ATTEMPTS, true );
		$current++;
		update_post_meta( $attachment_id, self::META_POLL_ATTEMPTS, $current );

		return $current;
	}

	public function reset_poll_attempts( int $attachment_id ) : void {
		if ( function_exists( 'delete_post_meta' ) ) {
			delete_post_meta( $attachment_id, self::META_POLL_ATTEMPTS );
		}
	}

	public function mark_permanent_fail( int $attachment_id ) : void {
		if ( function_exists( 'update_post_meta' ) ) {
			update_post_meta( $attachment_id, self::META_PERMANENT_FAIL, 1 );
		}
	}

	public function get_poll_attempts( int $attachment_id ) : int {
		if ( ! function_exists( 'get_post_meta' ) ) {
			return 0;
		}

		return (int) get_post_meta( $attachment_id, self::META_POLL_ATTEMPTS, true );
	}

	public function count_by_status( array $statuses ) : int {
		if ( ! class_exists( 'WP_Query' ) ) {
			return 0;
		}

		$query = new \WP_Query(
			[
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => self::SUPPORTED_MIMES,
				'fields'         => 'ids',
				'posts_per_page' => 1,
				'no_found_rows'  => false,
				'update_post_term_cache' => false,
				'update_post_meta_cache' => false,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Filter by plugin status meta; limited fields and attachment scope.
				'meta_query'     => [
					[
						'key'     => self::META_STATUS,
						'value'   => $statuses,
						'compare' => 'IN',
					],
				],
			]
		);

		return (int) $query->found_posts;
	}

	/**
	 * Return IDs of attachments that are permanently failed (exceeded retries).
	 */
	public function get_failed( int $page = 1, int $per = 20 ) : array {
		if ( ! class_exists( 'WP_Query' ) ) {
			return [ 'ids' => [], 'total' => 0, 'totalPages' => 0 ];
		}

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Filter by plugin permanent-fail meta; minimal fields and paging.
		$q = new \WP_Query([
			'post_type'      => 'attachment',
			'post_status'    => 'inherit',
			'fields'         => 'ids',
			'posts_per_page' => max( 1, $per ),
			'paged'          => max( 1, $page ),
			'no_found_rows'  => false,
			'update_post_term_cache' => false,
			'update_post_meta_cache' => false,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Listing permanently failed attachments requires meta_query; scoped, paged, minimal fields.
			'meta_query'     => [
				[
					'key'     => self::META_PERMANENT_FAIL,
					'compare' => 'EXISTS',
				],
			],
		]);

		return [
			'ids'        => array_map( 'intval', (array) $q->posts ),
			'total'      => (int) $q->found_posts,
			'totalPages' => (int) $q->max_num_pages,
		];
	}

	/**
	 * Clear all ImgSmaller-related post meta keys from attachments.
	 *
	 * Returns an array with counts for reporting: [ 'attachments' => int, 'rows_deleted' => int ].
	 */
	public function clear_all_plugin_meta() : array {
		$keys = [
			self::META_STATUS,
			self::META_PROGRESS_KEY,
			self::META_RESULT_URL,
			self::META_DOWNLOAD_TEMPLATE,
			self::META_POLL_ATTEMPTS,
			self::META_RETRY_COUNT,
		];

		$attachments = 0;
		$rows_deleted = 0;

			if ( isset( $GLOBALS['wpdb'] ) && $GLOBALS['wpdb'] instanceof \wpdb ) {
				global $wpdb;

				// Count attachments having any of our meta keys using WP_Query to avoid direct DB reads.
				if ( class_exists( '\\WP_Query' ) ) {
					$meta_or = array_map( static function ( $k ) {
						return [ 'key' => $k, 'compare' => 'EXISTS' ];
					}, $keys );
					$q = new \WP_Query(
						[
							'post_type'              => 'attachment',
							'post_status'            => 'inherit',
							'fields'                 => 'ids',
							'posts_per_page'         => 1,
							'no_found_rows'          => false,
							'update_post_meta_cache' => false,
							'update_post_term_cache' => false,
							'cache_results'          => false,
							// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Counting attachments by presence of any plugin meta requires a meta_query; scope is attachments with minimal fields and no caches.
							'meta_query'             => array_merge( [ 'relation' => 'OR' ], $meta_or ),
						]
					);
					$attachments = (int) $q->found_posts;
				}

				// Bulk delete meta rows directly for performance
				$in_placeholders = implode( ', ', array_fill( 0, count( $keys ), '%s' ) );
				$sql_delete = sprintf(
					"DELETE pm FROM {$wpdb->postmeta} pm INNER JOIN {$wpdb->posts} p ON p.ID = pm.post_id WHERE p.post_type = 'attachment' AND pm.meta_key IN (%s)",
					$in_placeholders
				);
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Bulk cleanup uses a direct, prepared DELETE for performance; caching is not applicable to a mutating DELETE. A per-post delete_post_meta() fallback exists below.
				$rows_deleted = (int) $wpdb->query( $wpdb->prepare( $sql_delete, ...$keys ) );
		} else {
			// Fallback: iterate via WP_Query and delete per attachment
			if ( class_exists( 'WP_Query' ) ) {
				$q = new \WP_Query(
					[
						'post_status'    => 'inherit',
						'fields'         => 'ids',
						'posts_per_page' => -1,
						'no_found_rows'  => true,
						// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Fallback cleanup needs to find attachments with plugin meta; admin-only and bounded by -1 within attachments.
						'meta_query'     => [
							[
								'key'     => self::META_STATUS,
								'compare' => 'EXISTS',
							],
						],
					]
				);
				foreach ( (array) $q->posts as $attachment_id ) {
					foreach ( $keys as $k ) {
						if ( function_exists( 'delete_post_meta' ) ) {
							$deleted = delete_post_meta( (int) $attachment_id, $k );
							if ( $deleted ) {
								$rows_deleted++;
							}
						}
					}
					$attachments++;
				}
			}
		}

		return [ 'attachments' => $attachments, 'rows_deleted' => $rows_deleted ];
	}
}
