<?php

declare( strict_types=1 );

namespace Imgspalat\Admin;

use Imgspalat\MediaRepository;
use Imgspalat\Services\BackupManager;
use Imgspalat\Settings;
use Imgspalat\StatusStore;
use Imgspalat\Tasks\BatchProcessor;

class DashboardPage {
	private Settings $settings;

	private StatusStore $status;

	private BatchProcessor $batch_processor;

	private string $page_hook = '';

	private string $backup_page_hook = '';

	private string $support_page_hook = '';

	public function __construct( Settings $settings, StatusStore $status, BatchProcessor $batch_processor ) {
		$this->settings        = $settings;
		$this->status          = $status;
		$this->batch_processor = $batch_processor;

		add_action( 'admin_post_imgsmaller_save_settings', [ $this, 'handle_settings_form' ] );
		add_action( 'admin_post_imgsmaller_restore', [ $this, 'handle_restore_form' ] );
		add_action( 'admin_post_imgsmaller_download_backup', [ $this, 'handle_download_backup' ] );
		add_action( 'admin_post_imgsmaller_import_backup', [ $this, 'handle_import_backup' ] );
		add_action( 'admin_post_imgsmaller_delete_backups', [ $this, 'handle_delete_backups' ] );
		add_action( 'admin_post_imgsmaller_import_replacements', [ $this, 'handle_import_replacements' ] );
		add_action( 'admin_post_imgsmaller_reset_plugin', [ $this, 'handle_reset_plugin' ] );
		add_action( 'admin_post_imgsmaller_regenerate_cron_token', [ $this, 'handle_regenerate_cron_token' ] );
	}

	public function register_menu_page() : void {
		$this->page_hook = add_menu_page(
			esc_html__( 'ImgSmaller', 'imgsmaller' ),
			esc_html__( 'ImgSmaller', 'imgsmaller' ),
			'manage_options',
			'imgsmaller-optimizer',
			[ $this, 'render_page' ],
			'dashicons-images-alt2',
			58
		);

		// New Backup & Restore subpage
		$this->backup_page_hook = add_submenu_page(
			'imgsmaller-optimizer',
			esc_html__( 'Backup & Restore', 'imgsmaller' ),
			esc_html__( 'Backup & Restore', 'imgsmaller' ),
			'manage_options',
			'imgsmaller-backup-restore',
			[ $this, 'render_backup_restore_page' ]
		);

		// Support subpage
		$this->support_page_hook = add_submenu_page(
			'imgsmaller-optimizer',
			esc_html__( 'Support', 'imgsmaller' ),
			esc_html__( 'Support', 'imgsmaller' ),
			'manage_options',
			'imgsmaller-support',
			[ $this, 'render_support_page' ]
		);

		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
	}

	public function render_backup_restore_page() : void {
		$settings = $this->settings->all();
		$status   = $this->status->all();
		$logs     = $this->status->logs();
		require IMGSMALLER_PLUGIN_DIR . 'views/backup-restore.php';
	}

	public function register_settings() : void {
		register_setting(
			'imgsmaller_settings_group',
			Settings::OPTION_KEY,
			[
				'sanitize_callback' => [ $this, 'sanitize_settings' ],
			]
		);
	}

	public function enqueue_assets( string $hook ) : void {
		$target_hooks = array_filter( [ $this->page_hook, $this->backup_page_hook, $this->support_page_hook ] );
		if ( ! in_array( $hook, $target_hooks, true ) ) {
			return;
		}

		wp_enqueue_style(
			'imgsmaller-dashboard',
			IMGSMALLER_PLUGIN_URL . 'assets/css/dashboard.css',
			[],
			IMGSMALLER_VERSION
		);

		wp_enqueue_script(
			'imgsmaller-dashboard',
			IMGSMALLER_PLUGIN_URL . 'assets/js/dashboard.js',
			[ 'jquery', 'wp-a11y' ],
			IMGSMALLER_VERSION,
			true
		);

		wp_localize_script(
			'imgsmaller-dashboard',
			'ImgSmallerDashboard',
			[
				'ajaxUrl'  => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'imgsmaller_dashboard' ),
				'settings' => $this->settings->all(),
				'status'   => $this->status->all(),
				'logs'     => $this->status->logs(),
				'i18n'     => [
					'connected'        => esc_html__( 'Connected', 'imgsmaller' ),
					'notConnected'     => esc_html__( 'Not Connected', 'imgsmaller' ),
					'notSet'           => esc_html__( 'Not set', 'imgsmaller' ),
					'errorGeneric'     => esc_html__( 'Something went wrong. Please try again.', 'imgsmaller' ),
					'testing'          => esc_html__( 'Testing…', 'imgsmaller' ),
					'testConnection'   => esc_html__( 'Test Connection', 'imgsmaller' ),
					'batchQueued'      => esc_html__( 'Compression batch queued.', 'imgsmaller' ),
					'startCompression' => esc_html__( 'Start Compression', 'imgsmaller' ),
					'starting'         => esc_html__( 'Starting…', 'imgsmaller' ),
					'pause'            => esc_html__( 'Pause', 'imgsmaller' ),
					'pausing'          => esc_html__( 'Pausing…', 'imgsmaller' ),
					'pausedMessage'    => esc_html__( 'Compression paused.', 'imgsmaller' ),
					'resume'           => esc_html__( 'Resume', 'imgsmaller' ),
					'resuming'         => esc_html__( 'Resuming…', 'imgsmaller' ),
					'resumedMessage'   => esc_html__( 'Compression resumed.', 'imgsmaller' ),
					'noLogs'           => esc_html__( 'No activity logged yet.', 'imgsmaller' ),
					/* translators: %d: percent completed */
					'progressPercent'  => esc_html__( '%d%% complete', 'imgsmaller' ),
					'imagesRemainingZero'     => esc_html__( 'No images remaining', 'imgsmaller' ),
					'imagesRemainingSingular' => esc_html__( '1 image remaining', 'imgsmaller' ),
					/* translators: %d: number of images remaining */
					'imagesRemainingPlural'   => esc_html__( '%d images remaining', 'imgsmaller' ),
					'startedBackground' => esc_html__( 'Your images are being optimized in the background. You can now safely close the admin dashboard.', 'imgsmaller' ),
					'restoreStart'      => esc_html__( 'Preparing restore…', 'imgsmaller' ),
					'restoreWorking'    => esc_html__( 'Restoring…', 'imgsmaller' ),
					'restoreComplete'   => esc_html__( 'Restore complete.', 'imgsmaller' ),
					'restoreFailed'     => esc_html__( 'Restore failed. Please try again.', 'imgsmaller' ),
					'restoreNothing'    => esc_html__( 'No backups found to restore.', 'imgsmaller' ),
					'restoreButton'     => esc_html__( 'Restore Originals', 'imgsmaller' ),
					'deleteBackupsConfirm' => esc_html__( 'This will permanently delete all stored backup originals. Continue?', 'imgsmaller' ),
					'deleteBackupsText'    => esc_html__( 'DELETE', 'imgsmaller' ),
					'processNow'           => esc_html__( 'Process Next Batch Now', 'imgsmaller' ),
					'processingNow'        => esc_html__( 'Processing…', 'imgsmaller' ),
					'processNowDone'       => esc_html__( 'Manual batch finished.', 'imgsmaller' ),
					'processNowFailed'     => esc_html__( 'Unable to run manual batch. Please try again.', 'imgsmaller' ),
					/* translators: %d: number of errors */
					'processNowPartial'    => esc_html__( 'Manual batch completed with %d errors. Check the activity log for details.', 'imgsmaller' ),
					/* translators: %d: seconds remaining */
					'processNowCooldown'   => esc_html__( 'Please wait %d seconds before triggering another manual batch.', 'imgsmaller' ),
					/* translators: %d: seconds remaining */
					'processNowCooldownShort' => esc_html__( 'Available in %d s', 'imgsmaller' ),
					// Tour strings
					'tourWelcomeTitle'   => esc_html__( 'Welcome to ImgSmaller', 'imgsmaller' ),
					'tourWelcomeBody'    => esc_html__( 'Let’s quickly set up your image optimization. This takes less than a minute.', 'imgsmaller' ),
					'tourStepApiTitle'   => esc_html__( 'Add your API Key', 'imgsmaller' ),
					'tourStepApiBody'    => esc_html__( 'Paste your ImgSmaller API key and click Connect.', 'imgsmaller' ),
					'tourStepDomainTitle'=> esc_html__( 'Allow your domain', 'imgsmaller' ),
					'tourStepDomainBody' => esc_html__( 'Add your site’s domain to the allowed list so the service can fetch your images.', 'imgsmaller' ),
					'tourStepStartTitle' => esc_html__( 'Start optimizing', 'imgsmaller' ),
					'tourStepStartBody'  => esc_html__( 'Click Start Compression. It runs safely in the background.', 'imgsmaller' ),
					'tourContinue'       => esc_html__( 'Continue', 'imgsmaller' ),
					'tourNoThanks'       => esc_html__( 'No thanks', 'imgsmaller' ),
					'tourBack'           => esc_html__( 'Back', 'imgsmaller' ),
					'tourDone'           => esc_html__( 'Done', 'imgsmaller' ),
					'tourButton'         => esc_html__( 'Show quick tour', 'imgsmaller' ),
				],
			]
		);
	}

	public function render_page() : void {
		$settings = $this->settings->all();
		$status   = $this->status->all();
		$logs     = $this->status->logs();

		require IMGSMALLER_PLUGIN_DIR . 'views/dashboard.php';
	}

	public function render_support_page() : void {
		$settings = $this->settings->all();
		$status   = $this->status->all();
		require IMGSMALLER_PLUGIN_DIR . 'views/support.php';
	}

	public function sanitize_settings( $input ) : array {
		$input = (array) $input;

		if ( isset( $input['api_key'] ) ) {
			$input['api_key'] = sanitize_text_field( $input['api_key'] );
		}

		if ( isset( $input['output_format'] ) ) {
			$allowed                 = [ 'jpg', 'jpeg', 'png', 'webp', 'avif', 'original' ];
			$input['output_format'] = strtolower( sanitize_text_field( $input['output_format'] ) );

			if ( ! in_array( $input['output_format'], $allowed, true ) ) {
				$input['output_format'] = 'webp';
			}
		}

		$input['enable_backup'] = isset( $input['enable_backup'] ) ? (bool) $input['enable_backup'] : false;

		if ( isset( $input['target_size'] ) ) {
			$target = trim( sanitize_text_field( (string) $input['target_size'] ) );
			$input['target_size'] = $target === '' ? 'auto' : $target;
		}

		if ( isset( $input['exclude_ids'] ) ) {
			$raw = sanitize_text_field( (string) $input['exclude_ids'] );
			// Normalize to a comma-separated list of integers
			$ids = [];
			foreach ( preg_split( '/[\s,]+/', $raw ) as $part ) {
				$int = (int) trim( (string) $part );
				if ( $int > 0 ) {
					$ids[] = (string) $int;
				}
			}
			$input['exclude_ids'] = implode( ',', array_unique( $ids ) );
		}


		if ( isset( $input['file_url'] ) ) {
			unset( $input['file_url'] );
		}

		return wp_parse_args( $input, $this->settings->all() );
	}

	public function handle_settings_form() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_save_settings' );

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Data is sanitized in sanitize_settings(); nonce verified above.
		$raw   = isset( $_POST[ Settings::OPTION_KEY ] ) ? (array) wp_unslash( $_POST[ Settings::OPTION_KEY ] ) : [];
		$data = $this->sanitize_settings( $raw );
		$this->settings->replace( $data );
		$this->settings->mark_connection_status( false );

		$this->redirect_to_dashboard( [ 'imgsmaller-updated' => '1' ] );
	}

	public function handle_restore_form() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_restore' );

		$backup_manager = new BackupManager( $this->settings, $this->status );
		$repository     = new MediaRepository();
		$restored       = $backup_manager->restore_all( $repository );
		/* translators: %d: number of images restored */
	/* translators: %d: number of images restored */
	$this->status->add_log( sprintf( __( 'Restore triggered from dashboard; %d images reverted.', 'imgsmaller' ), $restored ) );

		$this->status->update(
			[
				'fetched_count'     => 0,
				'sent_count'        => 0,
				'compressed_count'  => 0,
				'queued_count'      => 0,
				'in_progress_count' => 0,
			]
		);
		$this->status->clear_error();

		$this->redirect_to_dashboard(
			[
				'imgsmaller-restore' => 'done',
				'restored'           => $restored,
			]
		);
	}

	public function handle_download_backup() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_download_backup' );

		$backup_manager = new BackupManager( $this->settings, $this->status );
		$result         = $backup_manager->create_backup_archive();

		if ( is_wp_error( $result ) ) {
			/* translators: %s: error message */
			$this->status->add_log( sprintf( __( 'Backup export failed: %s', 'imgsmaller' ), $result->get_error_message() ), 'error' );
			$this->redirect_to_dashboard(
				[
					'imgsmaller-backup-error' => rawurlencode( $result->get_error_message() ),
				]
			);
		}

		$filename = 'imgsmaller-backups-' . gmdate( 'Y-m-d-His' ) . '.zip';
		$filesize = @filesize( $result );

		$this->status->add_log( __( 'Backup archive downloaded from the dashboard.', 'imgsmaller' ), 'info', [ 'file' => $filename ] );

		if ( ob_get_length() ) {
			@ob_end_clean();
		}

		nocache_headers();
		header( 'Content-Type: application/zip' );
		header( 'Content-Disposition: attachment; filename="' . $filename . '"' );
		if ( false !== $filesize ) {
			header( 'Content-Length: ' . (string) $filesize );
		}

	// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_readfile -- Direct streaming is intentional for download response; WP_Filesystem lacks a streaming alternative.
		readfile( $result );
		// Use core helper for safer deletion
		wp_delete_file( $result );
		exit;
	}

	public function handle_import_backup() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_import_backup' );

		if ( empty( $_FILES['imgsmaller_backup_zip'] ) ) {
			$this->redirect_to_dashboard(
				[
					'imgsmaller-backup-error' => rawurlencode( __( 'Please choose a ZIP file to upload.', 'imgsmaller' ) ),
				]
			);
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Accessing PHP upload array; validated via error code and is_uploaded_file(); original filename is not trusted.
		$file = $_FILES['imgsmaller_backup_zip'];
		$error = (int) ( $file['error'] ?? UPLOAD_ERR_OK );

		if ( UPLOAD_ERR_OK !== $error ) {
			$this->redirect_to_dashboard(
				[
					'imgsmaller-backup-error' => rawurlencode( $this->upload_error_message( $error ) ),
				]
			);
		}

		$tmp_name = isset( $file['tmp_name'] ) ? (string) $file['tmp_name'] : '';

		if ( ! $tmp_name || ! is_uploaded_file( $tmp_name ) ) {
			$this->redirect_to_dashboard(
				[
					'imgsmaller-backup-error' => rawurlencode( __( 'Unable to verify the uploaded file. Please try again.', 'imgsmaller' ) ),
				]
			);
		}

		$backup_manager = new BackupManager( $this->settings, $this->status );
		$result         = $backup_manager->import_backup_archive( $tmp_name );

		if ( is_wp_error( $result ) ) {
			/* translators: %s: error message */
			$this->status->add_log( sprintf( __( 'Backup import failed: %s', 'imgsmaller' ), $result->get_error_message() ), 'error' );
			$this->redirect_to_dashboard(
				[
					'imgsmaller-backup-error' => rawurlencode( $result->get_error_message() ),
				]
			);
		}

		$this->status->add_log( __( 'Backup archive imported from the dashboard.', 'imgsmaller' ) );

		$this->redirect_to_dashboard( [ 'imgsmaller-backup' => 'imported' ] );
	}

	public function handle_import_replacements() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_import_replacements' );

		if ( empty( $_FILES['imgsmaller_replace_zip'] ) ) {
			$this->redirect_to_dashboard(
				[
					'imgsmaller-backup-error' => rawurlencode( __( 'Please choose a ZIP file to upload.', 'imgsmaller' ) ),
				]
			);
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Accessing PHP upload array; validated via error code and is_uploaded_file(); original filename is not trusted.
		$file = $_FILES['imgsmaller_replace_zip'];
		$error = (int) ( $file['error'] ?? UPLOAD_ERR_OK );
		if ( UPLOAD_ERR_OK !== $error ) {
			$this->redirect_to_dashboard( [ 'imgsmaller-backup-error' => rawurlencode( $this->upload_error_message( $error ) ) ] );
		}

		$tmp_name = isset( $file['tmp_name'] ) ? (string) $file['tmp_name'] : '';
		if ( ! $tmp_name || ! is_uploaded_file( $tmp_name ) ) {
			$this->redirect_to_dashboard( [ 'imgsmaller-backup-error' => rawurlencode( __( 'Unable to verify the uploaded file. Please try again.', 'imgsmaller' ) ) ] );
		}

		$backup_manager = new \Imgspalat\Services\BackupManager( $this->settings, $this->status );
		$result = $backup_manager->import_replacements_archive( $tmp_name );

		if ( is_wp_error( $result ) ) {
			/* translators: %s: error message */
			$this->status->add_log( sprintf( __( 'Compressed import failed: %s', 'imgsmaller' ), $result->get_error_message() ), 'error' );
			$this->redirect_to_dashboard( [ 'imgsmaller-backup-error' => rawurlencode( $result->get_error_message() ) ] );
		}

		$count = (int) ( $result['replaced'] ?? 0 );
	/* translators: %d: number of images */
	$this->status->add_log( sprintf( __( 'Applied %d compressed images from zip.', 'imgsmaller' ), $count ), 'success' );
		$this->redirect_to_dashboard( [ 'imgsmaller-backup' => 'imported' ] );
	}

	public function handle_delete_backups() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_delete_backups' );

		$backup_manager = new BackupManager( $this->settings, $this->status );
		$result         = $backup_manager->delete_all_backups();

		if ( is_wp_error( $result ) ) {
			/* translators: %s: error message */
			$this->status->add_log( sprintf( __( 'Backup deletion failed: %s', 'imgsmaller' ), $result->get_error_message() ), 'error' );
			$this->redirect_to_dashboard(
				[
					'imgsmaller-backup-error' => rawurlencode( $result->get_error_message() ),
				]
			);
		}

		$files        = isset( $result['files_deleted'] ) ? (int) $result['files_deleted'] : 0;
		$attachments  = isset( $result['attachments_cleared'] ) ? (int) $result['attachments_cleared'] : 0;
		/* translators: 1: number of files, 2: number of attachments */
	/* translators: 1: number of files, 2: number of attachments */
	$log_message  = sprintf( __( 'Deleted %1$d backup files covering %2$d attachments.', 'imgsmaller' ), $files, $attachments );
		$this->status->add_log( $log_message, 'warning', $result );

		$this->redirect_to_dashboard(
			[
				'imgsmaller-backup'        => 'deleted',
				'deleted-files'            => $files,
				'deleted-attachments'      => $attachments,
			]
		);
	}

	private function redirect_to_dashboard( array $args = [] ) : void {
		$target = wp_get_referer();

		if ( ! $target ) {
			// Top-level menu page slug is 'imgsmaller-optimizer'
			$target = admin_url( 'admin.php?page=imgsmaller-optimizer' );
		}

		if ( ! empty( $args ) ) {
			$target = add_query_arg( $args, $target );
		}

		wp_safe_redirect( $target );
		exit;
	}

	private function upload_error_message( int $code ) : string {
		switch ( $code ) {
			case UPLOAD_ERR_INI_SIZE:
			case UPLOAD_ERR_FORM_SIZE:
				return __( 'The uploaded file exceeds the maximum allowed size.', 'imgsmaller' );
			case UPLOAD_ERR_PARTIAL:
				return __( 'The file was only partially uploaded. Please try again.', 'imgsmaller' );
			case UPLOAD_ERR_NO_FILE:
				return __( 'No file was uploaded. Please choose a backup ZIP file.', 'imgsmaller' );
			case UPLOAD_ERR_NO_TMP_DIR:
				return __( 'Missing a temporary folder on the server. Contact your host.', 'imgsmaller' );
			case UPLOAD_ERR_CANT_WRITE:
				return __( 'Unable to write the uploaded file to disk.', 'imgsmaller' );
			case UPLOAD_ERR_EXTENSION:
				return __( 'A server extension stopped the file upload.', 'imgsmaller' );
			default:
				return __( 'An unknown error occurred during upload.', 'imgsmaller' );
		}
	}

	public function handle_reset_plugin() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_reset_plugin' );

		// Clear cron lock and pause notices
		delete_transient( 'imgsmaller_batch_lock' );
		delete_transient( 'imgsmaller_paused_notice' );

		// Unschedule cron (both compression and restore)
		if ( function_exists( 'wp_next_scheduled' ) ) {
			$timestamp = wp_next_scheduled( IMGSMALLER_CRON_HOOK );
			while ( false !== $timestamp ) {
				wp_unschedule_event( $timestamp, IMGSMALLER_CRON_HOOK );
				$timestamp = wp_next_scheduled( IMGSMALLER_CRON_HOOK );
			}
			$rt = wp_next_scheduled( IMGSMALLER_RESTORE_CRON_HOOK );
			while ( false !== $rt ) {
				wp_unschedule_event( $rt, IMGSMALLER_RESTORE_CRON_HOOK );
				$rt = wp_next_scheduled( IMGSMALLER_RESTORE_CRON_HOOK );
			}
		}

		// Optional: also delete backup files if requested
		$delete_backups = ! empty( $_POST['imgsmaller_delete_backups'] );
		if ( $delete_backups ) {
			// Reuse BackupManager's directory path by reading stored option
			$backup_dir = get_option( 'imgsmaller_backup_dir' );
			if ( $backup_dir && is_dir( $backup_dir ) ) {
				// Remove directory recursively
				$this->rrmdir_safe( $backup_dir );
			}
			delete_option( 'imgsmaller_backup_dir' );
		}

		// Clear any pending restore job state
		try {
			( new \Imgspalat\Services\BackupManager( $this->settings, $this->status ) )->clear_restore_job();
		} catch ( \Throwable $e ) {
			// ignore
		}

		// Wipe options and logs
		$this->settings->delete_all();
		$this->status->delete();

		// Clear all attachment meta used by this plugin
		$repository = new MediaRepository();
		$clear_info = $repository->clear_all_plugin_meta();

		// Log a reset entry for transparency (recreate minimal status/log store transiently)
	/* translators: 1: attachments count, 2: rows deleted */
	$this->status->add_log( sprintf( __( 'Plugin reset from dashboard. Cleared meta on %1$d attachments (%2$d rows).', 'imgsmaller' ), (int) $clear_info['attachments'], (int) $clear_info['rows_deleted'] ), 'warning', $clear_info );

		$args = [ 'imgsmaller-reset' => 'done' ];
		if ( $delete_backups ) {
			$args['imgsmaller-backup'] = 'deleted';
		}
		$this->redirect_to_dashboard( $args );
	}

	public function handle_regenerate_cron_token() : void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'imgsmaller' ) );
		}

		check_admin_referer( 'imgsmaller_regenerate_cron_token' );

		$token = $this->settings->regenerate_cron_token();
		$this->status->add_log( __( 'Cron token regenerated from dashboard.', 'imgsmaller' ), 'warning' );

		$this->redirect_to_dashboard( [ 'imgsmaller-cron' => 'regenerated' ] );
	}

	private function rrmdir_safe( string $dir ) : void {
		$items = @scandir( $dir );
		if ( ! $items ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
			@rmdir( $dir );
			return;
		}
		foreach ( array_diff( $items, [ '.', '..' ] ) as $item ) {
			$path = $dir . DIRECTORY_SEPARATOR . $item;
			if ( is_dir( $path ) ) {
				$this->rrmdir_safe( $path );
			} else {
				// Using core delete helper when possible
				wp_delete_file( $path );
			}
		}
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
		@rmdir( $dir );
	}
}
