(function($){
	'use strict';

	var MANUAL_BATCH_COOLDOWN_MS = 15000;
	var manualProcessing = false;
	var manualCooldownUntil = window.ImgSmallerDashboard && ImgSmallerDashboard.manualCooldownUntil ? ImgSmallerDashboard.manualCooldownUntil : 0;
	var restoreRequestInFlight = false;
	var restoreStepInFlight = false;

	function getI18n(){
		return (window.ImgSmallerDashboard && ImgSmallerDashboard.i18n) || {};
	}

	function maskApiKey(key){
		var i18n = getI18n();
		if(!key){
			return i18n.notSet || 'Not set';
		}

		var keep = Math.min(4, key.length);
		var hidden = key.length > keep ? Array(key.length - keep + 1).join('\u2022') : '';
		return hidden + key.slice(-keep);
	}

	function renderLogs(logs){
		var container = $('.js-activity-log');

		if(!container.length){
			return;
		}

		container.empty();

		if(!logs || !logs.length){
			container.append($('<li>').text(getI18n().noLogs || 'No activity logged yet.'));
			return;
		}

		logs.forEach(function(entry){
			var level = entry.level || 'info';
			var item = $('<li>').addClass('log-item').addClass('log-item--' + level.replace(/[^a-z]/gi, '').toLowerCase());
			var timestamp = typeof entry.time === 'number' ? entry.time * 1000 : Date.now();
			var date = new Date(timestamp);
			var formatted = date.toLocaleString();

			item.append($('<strong>').text(formatted));
			item.append($('<span>').text(entry.message || ''));
			container.append(item);
		});
	}
	function getManualCooldownRemaining(){
		var remaining = manualCooldownUntil - Date.now();
		return remaining > 0 ? remaining : 0;
	}

	function updateProcessNowButton(status){
		var button = $('.js-process-now');
		if(!button.length){
			return;
		}

		var i18n = getI18n();
		status = status || {};

		if(!button.data('defaultLabel')){
			var baseLabel = button.attr('data-default-label') || i18n.processNow || button.text() || 'Process Next Batch Now';
			button.data('defaultLabel', baseLabel);
		}

		var paused = !!status.is_paused;
		var restoreActive = restoreRequestInFlight || !!status.restore_in_progress;
		var cooldownRemaining = getManualCooldownRemaining();
		var cooldownActive = cooldownRemaining > 0;
		var label;

		if(manualProcessing){
			label = i18n.processingNow || 'Processing…';
		}else if(cooldownActive){
			var seconds = Math.ceil(cooldownRemaining / 1000);
			var cooldownLabel = i18n.processNowCooldownShort || 'Available in %d s';
			label = cooldownLabel.replace('%d', seconds);
		}else{
			label = button.data('defaultLabel');
		}

		button.text(label);
		button.prop('disabled', paused || restoreActive || manualProcessing || cooldownActive);
	}

	function updateRestoreStepButton(status){
		var btn = $('.js-restore-step');
		if(!btn.length){ return; }
		status = status || {};
		if(!btn.data('defaultLabel')){ btn.data('defaultLabel', btn.text()); }
		var disabled = restoreRequestInFlight || !!status.restore_in_progress || restoreStepInFlight;
		btn.prop('disabled', disabled);
		if(disabled && (restoreRequestInFlight || restoreStepInFlight)){
			btn.text('Running…');
		}else{
			btn.text(btn.data('defaultLabel'));
		}
	}

	function showRestoreStatus(message, variant){
		var container = $('.js-restore-status');
		if(!container.length){
			return;
		}

		if(!message){
			container.attr('hidden', 'hidden').prop('hidden', true).hide();
			container.removeClass('is-success is-warning is-error is-info notice-success notice-warning notice-error notice-info');
			return;
		}

		container.removeAttr('hidden').prop('hidden', false).show();
		container.removeClass('is-success is-warning is-error is-info notice-success notice-warning notice-error notice-info');
		if(variant){
			container.addClass('is-' + variant + ' notice-' + variant);
		}

		var text = container.find('.js-restore-status-text');
		if(!text.length){
			text = $('<p class="js-restore-status-text">').appendTo(container);
		}
		text.text(message);
	}

	function refreshRestoreStatus(status, overrideMessage){
		status = status || {};
		var message = overrideMessage || status.restore_message || '';
		if(!message){
			showRestoreStatus('');
			return;
		}

		var failed = parseInt(status.restore_failed, 10);
		var restored = parseInt(status.restore_restored, 10);
		var variant = 'info';

		if(!isNaN(failed) && failed > 0){
			variant = 'warning';
		}else if(!isNaN(restored) && restored > 0){
			variant = 'success';
		}

		showRestoreStatus(message, variant);

		// Update restore progress bar
		var total = parseInt(status.restore_total, 10) || 0;
		var attempted = parseInt(status.restore_attempted, 10) || 0;
		var percent = total > 0 ? Math.min(100, Math.round((attempted / total) * 100)) : 0;
		var bar = $('.imgsmaller-restore-progress .imgsmaller-progress-bar');
		$('.js-restore-progress-fill').css('width', percent + '%');
		$('.js-restore-progress-text').text('Restore progress: ' + attempted + ' / ' + total);
		if(bar.length){
			bar.attr('aria-valuenow', percent);
			bar.attr('aria-valuetext', percent + '%');
		}
	}

	function updateRestoreButtonState(status){
		var restoreButton = $('.js-restore-submit');
		if(!restoreButton.length){
			return;
		}

		var i18n = getI18n();
		var disabled = restoreRequestInFlight || !!(status && status.restore_in_progress);
		restoreButton.prop('disabled', disabled);

		if(disabled){
			restoreButton.text(i18n.restoreWorking || 'Restoring…');
		}else{
			restoreButton.text(restoreButton.data('defaultLabel') || i18n.restoreButton || 'Restore Originals');
		}
	}

		function bindCancelRestore(){
		$(document).off('click', '.js-cancel-restore').on('click', '.js-cancel-restore', function(){
			var $btn = $(this);
			$btn.prop('disabled', true).text('Cancelling…');
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action: 'imgsmaller_cancel_restore', nonce: ImgSmallerDashboard.nonce }
			).done(function(res){
				if(res && res.success){
					if(res.data && res.data.status){ ImgSmallerDashboard.status = res.data.status; }
					if(res.data && res.data.logs){ ImgSmallerDashboard.logs = res.data.logs; }
					updateDashboard({});
					refreshRestoreStatus(ImgSmallerDashboard.status, ImgSmallerDashboard.status && ImgSmallerDashboard.status.restore_message);
				}else{
					window.alert((res && res.data && res.data.message) || 'Unable to cancel restore.');
				}
			}).fail(function(){
				window.alert('Unable to cancel restore.');
			}).always(function(){
				$btn.prop('disabled', false).text('Cancel Restore');
			});
		});
	}

	function updateActionButtons(status){
		status = status || {};
		var i18n = getI18n();
		var paused = !!status.is_paused;
		var restoreActive = restoreRequestInFlight || !!status.restore_in_progress;
		var quotaBlocked = !!status.quota_blocked || (typeof status.quota_remaining !== 'undefined' && status.quota_remaining !== null && parseInt(status.quota_remaining,10) <= 0 && parseInt(status.quota_reset||0,10) > (Date.now()/1000));

		var startButton = $('.js-start-compression');
		var pauseButton = $('.js-pause-compression');
		var resumeButton = $('.js-resume-compression');

		var queued = parseInt(status.queued_count, 10);
		if(isNaN(queued)){
			queued = 0;
		}

		var inProgress = parseInt(status.in_progress_count, 10);
		if(isNaN(inProgress)){
			inProgress = 0;
		}

		var processing = parseInt(status.processing_count, 10);
		if(isNaN(processing)){
			processing = queued + inProgress;
		}

		var isRunning = queued > 0 || inProgress > 0 || processing > 0;

		if(startButton.length){
			startButton.prop('disabled', paused || isRunning || restoreActive || quotaBlocked);
			if(i18n.startCompression){
				startButton.text(i18n.startCompression);
			}
		}

		if(pauseButton.length){
			pauseButton.prop('disabled', paused || !isRunning || restoreActive);
			if(i18n.pause){
				pauseButton.text(i18n.pause);
			}
		}

		if(resumeButton.length){
			// Allow resuming even if quota is blocked; enqueue will still be gated server-side
			resumeButton.prop('disabled', !paused || restoreActive);
			if(i18n.resume){
				resumeButton.text(i18n.resume);
			}
		}

		updateProcessNowButton(status);
		updateRestoreStepButton(status);
	}

	function handleRestoreFailure(message){
		var i18n = getI18n();
		var finalMessage = message || i18n.restoreFailed || i18n.errorGeneric || 'Something went wrong. Please try again.';
		showRestoreStatus(finalMessage, 'error');
		restoreRequestInFlight = false;
		updateRestoreButtonState(ImgSmallerDashboard.status || {});
		updateActionButtons(ImgSmallerDashboard.status || {});
		window.alert(finalMessage);
	}

	function startRestore(form){
		if(restoreRequestInFlight){
			return;
		}

		var i18n = getI18n();
		var button = form.find('.js-restore-submit');

		restoreRequestInFlight = true;
		button.prop('disabled', true);

		var statusSnapshot = $.extend({}, ImgSmallerDashboard.status || {}, { restore_in_progress: true });
		showRestoreStatus(i18n.restoreStart || 'Preparing restore…', 'info');
		updateRestoreButtonState(statusSnapshot);
		updateActionButtons(statusSnapshot);


		function doRestoreStep(){
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{
					action: 'imgsmaller_restore',
					nonce: ImgSmallerDashboard.nonce,
					time_limit: 8
				}
			).done(function(res){
				if(res && res.success){
					var data = res.data || {};
					if(data.settings){ ImgSmallerDashboard.settings = data.settings; }
					if(data.status){ ImgSmallerDashboard.status = data.status; }
					if(data.logs){ ImgSmallerDashboard.logs = data.logs; }

					updateDashboard(data);
					refreshRestoreStatus(ImgSmallerDashboard.status, data.message);

					if(data.done){
						if(window.wp && wp.a11y){
							wp.a11y.speak(data.message || i18n.restoreComplete || 'Restore complete.');
						}
						restoreRequestInFlight = false;
						updateRestoreButtonState(ImgSmallerDashboard.status || {});
						updateActionButtons(ImgSmallerDashboard.status || {});
					}else{
						// continue steps after a short delay
						setTimeout(doRestoreStep, 250);
					}
				}else{
					handleRestoreFailure(res && res.data && res.data.message ? res.data.message : null);
				}
			}).fail(function(){
				handleRestoreFailure();
			});
		}

		doRestoreStep();
	}

	function updateDashboard(payload){
		payload = payload || {};

		var settings = payload.settings || ImgSmallerDashboard.settings || {};
		var status = payload.status || ImgSmallerDashboard.status || {};
		var logs = payload.logs || ImgSmallerDashboard.logs || [];
		var i18n = getI18n();

		ImgSmallerDashboard.settings = settings;
		ImgSmallerDashboard.status = status;
		ImgSmallerDashboard.logs = logs;

		$('.js-metric').each(function(){
			var key = $(this).data('key');
			if(key && typeof status[key] !== 'undefined'){
				$(this).text(status[key]);
			}
		});

		var progressBar = $('.js-progress-bar');
		var progressFill = $('.js-progress-fill');
		if(progressBar.length && progressFill.length){
			var percentTemplate = i18n.progressPercent || '%d%% complete';
			var percent = parseInt(status.progress_percent, 10);
			if(isNaN(percent)){
				percent = 0;
			}
			percent = Math.max(0, Math.min(100, percent));
			progressFill.css('width', percent + '%');
			var percentText = percentTemplate.replace('%d', percent).replace('%%', '%');
			progressBar.attr('aria-valuenow', percent).attr('aria-valuetext', percentText);
			$('.js-progress-label').text(percentText);

			var remaining = parseInt(status.remaining_count, 10);
			if(isNaN(remaining) || remaining < 0){
				remaining = 0;
			}

			var remainingZero = i18n.imagesRemainingZero || 'No images remaining';
			var remainingSingular = i18n.imagesRemainingSingular || '1 image remaining';
			var remainingPlural = i18n.imagesRemainingPlural || '%d images remaining';

			var remainingText;
			if(remaining === 0){
				remainingText = remainingZero;
			}else if(remaining === 1){
				remainingText = remainingSingular;
			}else{
				remainingText = remainingPlural.replace('%d', remaining);
			}

			$('.js-progress-remaining').text(remainingText);
		}

		var badge = $('.js-connection-status');
		if(badge.length){
			if(settings.connected){
				badge.removeClass('imgsmaller-status-badge--disconnected')
					.addClass('imgsmaller-status-badge--connected')
					.text(i18n.connected || 'Connected');
			}else{
				badge.removeClass('imgsmaller-status-badge--connected')
					.addClass('imgsmaller-status-badge--disconnected')
					.text(i18n.notConnected || 'Not Connected');
			}
		}

		// Quota countdown notice UI
		(function(){
			// Hide quota notice entirely per request; keep logic elsewhere (buttons) intact.
			var container = $('.js-quota-notice');
			if(container.length){ container.remove(); }
		})();

		$('.js-api-key-mask').text(maskApiKey(settings.api_key));

		updateActionButtons(status);
		updateRestoreButtonState(status);
		refreshRestoreStatus(status, payload.message);
		renderLogs(logs);

		// Keep Compression Overview details updated from status mirror
		(function(){
			function formatKB(bytes){ var kb=(bytes||0)/1024; return (Math.round(kb*100)/100) + ' KB'; }
			var t = {
				total: parseInt(status.scan_total_images||0,10)||0,
				opt: parseInt(status.scan_optimized_images||0,10)||0,
				non: parseInt(status.scan_non_optimized_images||0,10)||0,
				bytes: parseInt(status.scan_total_bytes||0,10)||0,
				saved: parseInt(status.scan_saved_bytes||0,10)||0,
				potential: parseInt(status.scan_potential_bytes||0,10)||0
			};
			$('.js-scan-total-images').text(t.total);
			$('.js-scan-optimized').text(t.opt);
			$('.js-scan-nonoptimized').text(t.non);
			$('.js-scan-total-size').text(formatKB(t.bytes));
			$('.js-scan-saved').text(formatKB(t.saved));
			$('.js-scan-potential').text(formatKB(t.potential));
			// Draw pie
			var canvas = document.querySelector('.js-scan-pie');
			var ctx = canvas ? canvas.getContext('2d') : null;
			if(ctx && canvas){
				var total = Math.max(1, t.opt + t.non);
				var start = -Math.PI/2;
				ctx.clearRect(0,0,canvas.width,canvas.height);
				var cx = canvas.width/2, cy = canvas.height/2, r = Math.min(cx, cy) - 10;
				function slice(value, color){ var angle=(value/total)*Math.PI*2; ctx.beginPath(); ctx.moveTo(cx, cy); ctx.arc(cx, cy, r, start, start+angle); ctx.closePath(); ctx.fillStyle=color; ctx.fill(); start += angle; }
				slice(t.opt, '#2ecc71');
				slice(t.non, '#bdc3c7');
				ctx.beginPath(); ctx.arc(cx, cy, r*0.55, 0, Math.PI*2); ctx.fillStyle = '#ffffff'; ctx.fill();
			}
		})();
	}

	function fetchStatus(){
		$.post(
			ImgSmallerDashboard.ajaxUrl,
			{
				action: 'imgsmaller_status',
				nonce: ImgSmallerDashboard.nonce
			}
		).done(function(res){
			if(res && res.success){
				updateDashboard(res.data);
			}
		});
	}

	function testConnection(button){
		var i18n = getI18n();
		button.prop('disabled', true).text(i18n.testing || 'Testing…');

		$.post(
			ImgSmallerDashboard.ajaxUrl,
			{
				action: 'imgsmaller_test_connection',
				nonce: ImgSmallerDashboard.nonce
			}
		).done(function(res){
			if(res && res.success){
				fetchStatus();
				if(window.wp && wp.a11y){
					wp.a11y.speak(res.data && res.data.connected ? (i18n.connected || 'Connected') : (i18n.notConnected || 'Not Connected'));
				}
			}
		}).fail(function(){
			window.alert(i18n.errorGeneric || 'Something went wrong. Please try again.');
		}).always(function(){
			button.prop('disabled', false).text(i18n.testConnection || 'Test Connection');
		});
	}

	function startCompression(button){
		var i18n = getI18n();
		var startingLabel = i18n.starting || i18n.startCompression || 'Starting…';
		var startLabel = i18n.startCompression || 'Start Compression';
		var errorMessage = i18n.errorGeneric || 'Something went wrong. Please try again.';
		button.prop('disabled', true).text(startingLabel);

		$.post(
			ImgSmallerDashboard.ajaxUrl,
			{
				action: 'imgsmaller_start',
				nonce: ImgSmallerDashboard.nonce
			}
		).done(function(res){
			if(res && res.success){
				fetchStatus();
				if(window.wp && wp.a11y && i18n.batchQueued){
					wp.a11y.speak(i18n.batchQueued);
				}
				// Show background processing notice
				var popupMsg = i18n.startedBackground || 'Your images are being optimized in the background. You can now safely close the admin dashboard.';
				try { window.alert(popupMsg); } catch (e) {}
			}else{
				var msg = (res && res.data && res.data.message) ? res.data.message : errorMessage;
				window.alert(msg);
				button.prop('disabled', false);
			}
		}).fail(function(){
			window.alert(errorMessage);
			button.prop('disabled', false);
		}).always(function(){
			button.text(startLabel);
		});
	}

	function pauseCompression(button){
		var i18n = getI18n();
		button.prop('disabled', true).text(i18n.pausing || i18n.pause || 'Pausing…');

		$.post(
			ImgSmallerDashboard.ajaxUrl,
			{
				action: 'imgsmaller_pause',
				nonce: ImgSmallerDashboard.nonce
			}
		).done(function(res){
			if(res && res.success){
				updateDashboard(res.data);
				if(window.wp && wp.a11y && i18n.pausedMessage){
					wp.a11y.speak(i18n.pausedMessage);
				}
			}else{
				window.alert(i18n.errorGeneric || 'Something went wrong. Please try again.');
			}
		}).fail(function(){
			window.alert(i18n.errorGeneric || 'Something went wrong. Please try again.');
		}).always(function(){
			updateActionButtons(ImgSmallerDashboard.status || {});
		});
	}

	function resumeCompression(button){
		var i18n = getI18n();
		button.prop('disabled', true).text(i18n.resuming || i18n.resume || 'Resuming…');

		$.post(
			ImgSmallerDashboard.ajaxUrl,
			{
				action: 'imgsmaller_resume',
				nonce: ImgSmallerDashboard.nonce
			}
		).done(function(res){
			if(res && res.success){
				updateDashboard(res.data);
				if(window.wp && wp.a11y && i18n.resumedMessage){
					wp.a11y.speak(i18n.resumedMessage);
				}
			}else{
				window.alert(i18n.errorGeneric || 'Something went wrong. Please try again.');
			}
		}).fail(function(){
			window.alert(i18n.errorGeneric || 'Something went wrong. Please try again.');
		}).always(function(){
			updateActionButtons(ImgSmallerDashboard.status || {});
		});
	}

	function triggerManualProcess(button){
		var i18n = getI18n();
		if(manualProcessing){
			return;
		}

		var cooldownRemaining = getManualCooldownRemaining();
		if(cooldownRemaining > 0){
			var seconds = Math.ceil(cooldownRemaining / 1000);
			var cooldownMessage = i18n.processNowCooldown || 'Please wait %d seconds before triggering another manual batch.';
			window.alert(cooldownMessage.replace('%d', seconds));
			updateProcessNowButton(ImgSmallerDashboard.status || {});
			return;
		}

		manualProcessing = true;
		ImgSmallerDashboard.manualProcessing = true;
		button.prop('disabled', true);
		updateProcessNowButton(ImgSmallerDashboard.status || {});

		$.post(
			ImgSmallerDashboard.ajaxUrl,
			{
				action: 'imgsmaller_process_now',
				nonce: ImgSmallerDashboard.nonce,
				cycles: 3
			}
		).done(function(res){
			if(res && res.success){
				var data = res.data || {};
				manualCooldownUntil = Date.now() + MANUAL_BATCH_COOLDOWN_MS;
				ImgSmallerDashboard.manualCooldownUntil = manualCooldownUntil;

				if(data.status){
					ImgSmallerDashboard.status = data.status;
				}
				if(data.settings){
					ImgSmallerDashboard.settings = data.settings;
				}
				if(data.logs){
					ImgSmallerDashboard.logs = data.logs;
				}

				updateDashboard(data);

				if(window.wp && wp.a11y && i18n.processNowDone){
					wp.a11y.speak(i18n.processNowDone);
				}

				if(data.errors && data.errors.length){
					var partialTemplate = i18n.processNowPartial || 'Manual batch completed with %d errors. Check the activity log for details.';
					window.alert(partialTemplate.replace('%d', data.errors.length));
				}
			}else{
				var message = res && res.data && res.data.message ? res.data.message : null;
				window.alert(message || i18n.processNowFailed || i18n.errorGeneric || 'Something went wrong. Please try again.');
			}
		}).fail(function(){
			window.alert(i18n.processNowFailed || i18n.errorGeneric || 'Something went wrong. Please try again.');
		}).always(function(){
			manualProcessing = false;
			ImgSmallerDashboard.manualProcessing = false;
			updateProcessNowButton(ImgSmallerDashboard.status || {});
			updateActionButtons(ImgSmallerDashboard.status || {});
		});
	}

	(function(){
		var container = $('.js-delete-backups');
		if(!container.length){
			return;
		}

		var trigger = container.find('.js-delete-backups-trigger');
		var confirmBox = container.find('.js-delete-backups-confirm');
		var input = container.find('.js-delete-backups-input');
		var submit = container.find('.js-delete-backups-submit');
		var cancel = container.find('.js-delete-backups-cancel');
		var i18n = getI18n();
		var required = (i18n.deleteBackupsText || 'DELETE').toUpperCase();

		function resetConfirm(){
			confirmBox.attr('hidden', 'hidden').hide();
			input.val('');
			submit.prop('disabled', true);
		}

		trigger.on('click', function(){
			confirmBox.removeAttr('hidden').show();
			input.trigger('focus');
		});

		cancel.on('click', function(){
			resetConfirm();
		});

		input.on('input', function(){
			submit.prop('disabled', input.val().trim().toUpperCase() !== required);
		});

		container.find('form').on('submit', function(event){
			if(input.val().trim().toUpperCase() !== required){
				event.preventDefault();
				return;
			}

			var message = i18n.deleteBackupsConfirm || 'This will permanently delete all stored backup originals. Continue?';
			if(!window.confirm(message)){
				event.preventDefault();
				return;
			}

			resetConfirm();
		});
	})();

	// Danger Zone: Reset plugin confirmation
	(function(){
		var container = $('.js-reset-plugin');
		if(!container.length){
			return;
		}

		var trigger = container.find('.js-reset-plugin-trigger');
		var confirmBox = container.find('.js-reset-plugin-confirm');
		var input = container.find('.js-reset-plugin-input');
		var submit = container.find('.js-reset-plugin-submit');
		var required = 'RESET';

		function resetConfirm(){
			confirmBox.attr('hidden', 'hidden').hide();
			input.val('');
			submit.prop('disabled', true);
		}

		trigger.on('click', function(){
			confirmBox.removeAttr('hidden').show();
			input.trigger('focus');
		});

		container.find('.js-reset-plugin-cancel').on('click', function(){
			resetConfirm();
		});

		input.on('input', function(){
			submit.prop('disabled', input.val().trim().toUpperCase() !== required);
		});

		container.find('form').on('submit', function(event){
			if(input.val().trim().toUpperCase() !== required){
				event.preventDefault();
				return;
			}

			// If user opted to delete backups, enforce DELETE confirmation
			var deleteBackupsChecked = container.find('.js-reset-delete-backups-toggle').is(':checked');
			if(deleteBackupsChecked){
				var delInput = container.find('.js-reset-delete-backups-input');
				if(!delInput.val() || delInput.val().trim().toUpperCase() !== 'DELETE'){
					event.preventDefault();
					window.alert('Please type DELETE to confirm backup deletion.');
					return;
				}
			}

			var message = 'This will reset all plugin data and settings. Continue?';
			if(!window.confirm(message)){
				event.preventDefault();
				return;
			}

			resetConfirm();
		});
	})();

	// Copy Cron URL button
	(function(){
		var btn = $('.js-copy-cron-url');
		if(!btn.length){ return; }
		btn.on('click', function(){
			var textEl = $('.js-cron-url-text');
			var url = textEl.text();
			if(!url){ return; }
			if(navigator.clipboard && navigator.clipboard.writeText){
				navigator.clipboard.writeText(url).then(function(){
					btn.text('Copied!');
					setTimeout(function(){ btn.text('Copy URL'); }, 1500);
				});
			}else{
				// Fallback
				var temp = $('<input type="text" style="position:absolute;left:-9999px;">').val(url).appendTo(document.body);
				temp[0].select();
				try{ document.execCommand('copy'); }catch(e){}
				temp.remove();
				btn.text('Copied!');
				setTimeout(function(){ btn.text('Copy URL'); }, 1500);
			}
		});
	})();

	$(function(){
		// Failed images browser
		(function(){
			var grid = $('.js-failed-grid');
			if(!grid.length){ return; }
			var loading = $('.js-failed-loading');
			var refreshBtn = $('.js-failed-refresh');
			var loadMoreBtn = $('.js-failed-load-more');
			var page = 1;
			var per = 20;
			var totalPages = 1;
			var isLoading = false;

			function render(items, append){
				if(!append){ grid.empty(); }
				items.forEach(function(it){
					var cell = $('<div class="imgsmaller-exclude-item">');
					var img = $('<img>').attr('src', it.url || '').attr('alt', it.title || '').css({ width:'100%', height:'auto', display:'block', aspectRatio:'1/1', objectFit:'cover', background:'#f7f7f7' });
					var meta = $('<div class="imgsmaller-exclude-meta" style="margin-top:4px; font-size:12px; line-height:1.3; color:#444;">');
					var title = $('<div class="imgsmaller-exclude-title">').text(it.title || ('#' + it.id));
					meta.append(title);
					cell.append(img).append(meta);
					grid.append(cell);
				});
			}

			function load(reset){
				if(isLoading){ return; }
				isLoading = true;
				loading.show();
				if(reset){ page = 1; totalPages = 1; }
				$.post(
					ImgSmallerDashboard.ajaxUrl,
					{ action:'imgsmaller_failed_list', nonce: ImgSmallerDashboard.nonce, page: page, per: per }
				).done(function(res){
					if(res && res.success){
						totalPages = res.data.totalPages || 1;
						render(res.data.items || [], !reset && page > 1);
					}
				}).always(function(){
					isLoading = false;
					loading.hide();
				});
			}

			refreshBtn.on('click', function(){ load(true); });
			loadMoreBtn.on('click', function(){ if(page < totalPages){ page++; load(false); } });
			load(true);
		})();
		if(!window.ImgSmallerDashboard){
			return;
		}

		// Quick Tour (first-time onboarding)
		function openTour(force){
			var settings = ImgSmallerDashboard.settings || {};
			var i18n = getI18n();
			var dismissed = !!settings.onboarding_dismissed;
			var completed = !!settings.onboarding_completed;
			if(!force && (dismissed || completed)){ return; }

			var steps = [
				{ title: i18n.tourWelcomeTitle || 'Welcome to ImgSmaller', body: i18n.tourWelcomeBody || 'Let’s quickly set up your image optimization. This takes less than a minute.' },
				{ title: i18n.tourStepApiTitle || 'Add your API Key', body: i18n.tourStepApiBody || 'Paste your ImgSmaller API key and click Connect.' },
				{ title: i18n.tourStepDomainTitle || 'Allow your domain', body: i18n.tourStepDomainBody || 'Add your site’s domain to the allowed list so the service can fetch your images.' },
				{ title: i18n.tourStepStartTitle || 'Start optimizing', body: i18n.tourStepStartBody || 'Click Start Compression. It runs safely in the background.' }
			];

			var $overlay = $('<div class="imgsmaller-tour-overlay" role="dialog" aria-modal="true" aria-labelledby="imgsmaller-tour-title" style="position:fixed;inset:0;background:rgba(0,0,0,0.45);z-index:100000;display:flex;align-items:center;justify-content:center;">');
			var $card = $('<div class="imgsmaller-tour-card" style="background:#fff;border-radius:8px;box-shadow:0 10px 30px rgba(0,0,0,0.2);max-width:520px;width:92%;padding:20px;">');
			var $title = $('<h2 id="imgsmaller-tour-title" style="margin:0 0 8px 0;">');
			var $body = $('<p style="margin:0 0 16px 0;">');
			var $actions = $('<div style="display:flex;align-items:center;justify-content:space-between;gap:8px;">');
			var $left = $('<div>');
			var $right = $('<div>');
			var $back = $('<button type="button" class="button">').text(i18n.tourBack || 'Back');
			var $continue = $('<button type="button" class="button button-primary">').text(i18n.tourContinue || 'Continue');
			var $skip = $('<button type="button" class="button">').text(i18n.tourNoThanks || 'No thanks');
			var $done = $('<button type="button" class="button button-primary">').text(i18n.tourDone || 'Done');

			$left.append($back);
			$right.append($skip).append($continue).append($done.hide());
			$actions.append($left).append($right);
			$card.append($title).append($body).append($actions);
			$overlay.append($card);

			var step = 0;
			function render(){
				var s = steps[step];
				$title.text(s.title);
				$body.text(s.body);
				$back.prop('disabled', step===0);
				$continue.toggle(step < steps.length - 1);
				$done.toggle(step === steps.length - 1);
			}

			function close(){ $overlay.remove(); }

			$back.on('click', function(){ if(step>0){ step--; render(); } });
			$continue.on('click', function(){ if(step < steps.length-1){ step++; render(); } });
			$skip.on('click', function(){ $.post(ImgSmallerDashboard.ajaxUrl, { action:'imgsmaller_tour_dismiss', nonce: ImgSmallerDashboard.nonce }).always(close); });
			$done.on('click', function(){ $.post(ImgSmallerDashboard.ajaxUrl, { action:'imgsmaller_tour_complete', nonce: ImgSmallerDashboard.nonce }).always(close); });

			$('body').append($overlay);
			render();
		}

		// Auto-show condition: no API key OR current domain not allowed
		(function(){
			var settings = ImgSmallerDashboard.settings || {};
			var needsApi = !settings.api_key;
			var dismissed = !!settings.onboarding_dismissed;
			var completed = !!settings.onboarding_completed;
			if(dismissed || completed){ /* allow manual */ } else {
				if(needsApi){ openTour(false); return; }
				// If API present, fetch plan info to check allowed domains
				$.post(ImgSmallerDashboard.ajaxUrl, { action:'imgsmaller_plan_info', nonce: ImgSmallerDashboard.nonce }).done(function(res){
					if(res && res.success){
						var d = res.data || {};
						var allows = d.allows || {};
						var site = d.site || {};
						var domains = (allows.domains && Array.isArray(allows.domains)) ? allows.domains : [];
						var current = (site && site.domain) ? site.domain : '';
						var allowed = current && domains.map(function(x){ return (x||'').toString().toLowerCase(); }).indexOf(current.toString().toLowerCase()) !== -1;
						if(!allowed){ openTour(false); }
					}
				});
			}
		})();

		// Floating button to rerun the tour
		(function(){
			var i18n = getI18n();
			var $btn = $('<button type="button" class="button imgsmaller-tour-fab" aria-label="'+(i18n.tourButton || 'Show quick tour')+'" title="'+(i18n.tourButton || 'Show quick tour')+'">?</button>');
			$btn.css({ position:'fixed', right:'18px', bottom:'18px', zIndex:100001, borderRadius:'50%', width:'40px', height:'40px', lineHeight:'40px', textAlign:'center', padding:0 });
			$btn.on('click', function(){ openTour(true); });
			$('body').append($btn);
		})();

		updateDashboard({
			settings: ImgSmallerDashboard.settings || {},
			status: ImgSmallerDashboard.status || {},
			logs: ImgSmallerDashboard.logs || []
		});

		$('.js-restore-submit').each(function(){
			var $btn = $(this);
			if(!$btn.data('defaultLabel')){
				$btn.data('defaultLabel', $btn.text());
			}
		});

		$('.js-process-now').each(function(){
			var $btn = $(this);
			if(!$btn.data('defaultLabel')){
				var label = $btn.attr('data-default-label') || getI18n().processNow || $btn.text();
				$btn.data('defaultLabel', label);
				$btn.text(label);
			}
		});

		$('.js-test-connection').on('click', function(){
			testConnection($(this));
		});

		$('.js-start-compression').on('click', function(){
			startCompression($(this));
		});

		$('.js-pause-compression').on('click', function(){
			pauseCompression($(this));
		});

		$('.js-resume-compression').on('click', function(){
			resumeCompression($(this));
		});

		$('.js-restore-step').each(function(){
			var $btn = $(this);
			if(!$btn.data('defaultLabel')){
				$btn.data('defaultLabel', $btn.text());
			}
		}).on('click', function(){
			var $btn = $(this);
			var defaultLabel = $btn.data('defaultLabel') || 'Run one restore step now';
			restoreStepInFlight = true;
			$btn.prop('disabled', true).text('Running…');
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action: 'imgsmaller_restore_step', nonce: ImgSmallerDashboard.nonce }
			).done(function(res){
				if(res && res.success){
					var data = res.data || {};
					if(data.status){ ImgSmallerDashboard.status = data.status; }
					if(data.logs){ ImgSmallerDashboard.logs = data.logs; }
					updateDashboard(data);
					refreshRestoreStatus(ImgSmallerDashboard.status, data.message);
					if(window.wp && wp.a11y){ wp.a11y.speak(data.message || 'Restore step completed.'); }
				}else{
					window.alert((res && res.data && res.data.message) || 'Restore step failed.');
				}
			}).fail(function(){
				window.alert('Restore step failed.');
			}).always(function(){
				restoreStepInFlight = false;
				$btn.prop('disabled', false).text(defaultLabel);
				updateActionButtons(ImgSmallerDashboard.status || {});
				updateRestoreButtonState(ImgSmallerDashboard.status || {});
			});
		});

		$('.js-restore-form').on('submit', function(event){
			event.preventDefault();
			startRestore($(this));
		});

		$('.js-process-now').on('click', function(){
			triggerManualProcess($(this));
		});

		$('<button type="button" class="button" style="margin-left: 10px;">Debug Media</button>')
			.insertAfter('.js-start-compression')
			.on('click', function(){
				var $self = $(this);
				$self.prop('disabled', true).text('Checking...');

				$.post(
					ImgSmallerDashboard.ajaxUrl,
					{
						action: 'imgsmaller_debug_media',
						nonce: ImgSmallerDashboard.nonce
					}
				).done(function(res){
					if(res && res.success){
						console.log('Debug Info:', res.data);
						window.alert('Debug info logged to console. Check browser developer tools.');
						fetchStatus();
					}
				}).always(function(){
					$self.prop('disabled', false).text('Debug Media');
				});
			});

		bindCancelRestore();

		// Fetch current plan & allowed domains
		(function(){
			var box = $('.js-plan-box');
			if(!box.length){ return; }
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action:'imgsmaller_plan_info', nonce: ImgSmallerDashboard.nonce }
			).done(function(res){
				var loading = box.find('.js-plan-loading');
				var content = box.find('.js-plan-content');
				loading.hide();
				if(res && res.success){
					var d = res.data || {};
					var plan = d.plan || {};
					var info = d.info || {};
					var allows = d.allows || {};
					var plans = (d.plans && d.plans.plans) ? d.plans.plans : null;
					var site = d.site || {};
					content.show();
					box.find('.js-plan-name').text(plan.plan_name || plan.plan_key || '—');
					box.find('.js-plan-usage').text(typeof plan.usage !== 'undefined' ? plan.usage : (info.used_today || 0));
					box.find('.js-plan-limit').text(typeof plan.limit !== 'undefined' ? (plan.limit === 0 ? 'Unlimited' : plan.limit) : (typeof info.limit !== 'undefined' ? info.limit : '—'));
					box.find('.js-plan-renews').text(plan.renews_at || '—');
					box.find('.js-plan-remaining').text(typeof info.remaining_today !== 'undefined' ? info.remaining_today : '—');

					// Hide Billing Cycle and Images Remaining for Free plan only
					var planKey = (plan.plan_key || plan.plan_name || '').toString().toLowerCase();
					var isFree = planKey.indexOf('free') !== -1;
					var renewsRow = box.find('.js-plan-renews').closest('p');
					var remainingRow = box.find('.js-plan-remaining').closest('p');
					if(isFree){
						renewsRow.hide();
						remainingRow.hide();
					} else {
						renewsRow.show();
						remainingRow.show();
					}
					// Add generic Upgrade button near content
					if(!box.find('.js-upgrade-generic').length){
						$('<p><a class="button button-primary js-upgrade-generic" target="_blank" rel="noopener noreferrer" href="https://imgsmaller.com/dashboard/billing">Upgrade Plan</a></p>').appendTo(content);
					}
					var domainInput = $('.js-domain-input');
					var currentDomain = (site && site.domain) ? site.domain : '';
					if(currentDomain){ domainInput.val(currentDomain); }
					var domains = (allows && allows.domains) ? allows.domains : [];
					box.find('.js-domain-list').text(domains && domains.length ? domains.join(', ') : '—');

					// Render Available Plans
					var plansWrap = $('.js-plans-list');
					if(plansWrap.length){
						plansWrap.find('.js-plans-loading').remove();
						if(plans){
							var currentKey = plan.plan_key || '';
							var entries = Object.keys(plans).map(function(key){ return { key:key, data:plans[key] }; });
							entries.sort(function(a,b){
								var pa = a.data.price || 0, pb = b.data.price || 0;
								return pa - pb;
							});
							if(!entries.length){ plansWrap.append($('<p>').text('No plans data.')); }
							entries.forEach(function(entry){
								var p = entry.data || {};
								var card = $('<div class="imgsmaller-plan-item" style="border:1px solid #e5e5e5;border-radius:4px;padding:10px;margin:8px 0;background:#fff;">');
								var title = $('<div style="display:flex;align-items:center;justify-content:space-between;gap:8px;">');
								var name = $('<strong>').text(p.name || entry.key || 'Plan');
								var price = typeof p.price !== 'undefined' ? ('$' + p.price + '/mo') : '';
								var priceEl = $('<span>').text(price).css({opacity:0.9});
								title.append(name).append(priceEl);
								var limit = (typeof p.limit !== 'undefined') ? (p.limit === 0 ? 'Unlimited' : p.limit) : '—';
								var domainsLimit = (typeof p.allowed_domains_limit !== 'undefined') ? p.allowed_domains_limit : '—';
								var details = $('<div style="margin-top:6px;">')
									.append($('<div>').text('Number of Images: ' + limit))
									.append($('<div>').text('Allowed domains: ' + domainsLimit));
								var actions = $('<div style="margin-top:8px;">');
								var upgradeUrl = 'https://imgsmaller.com/dashboard/billing';
								var btnLabel = (entry.key === currentKey) ? 'Current Plan' : 'Upgrade';
								var btnClass = (entry.key === currentKey) ? 'button' : 'button button-primary';
								var btn = $('<a target="_blank" rel="noopener noreferrer">').attr('href', upgradeUrl).addClass(btnClass).text(btnLabel);
								if(entry.key === currentKey){ btn.prop('disabled', true); }
								actions.append(btn);
								card.append(title).append(details).append(actions);
								plansWrap.append(card);
							});
						}else{
							plansWrap.append($('<p>').text('Unable to load plans.'));
						}
					}
				}else{
					box.find('.js-plan-loading').text((res && res.data && res.data.message) || 'Unable to load plan.');
				}
			});
		})();

		// Save domain
		$(document).on('click', '.js-save-domain', function(){
			var input = $('.js-domain-input');
			var status = $('.js-domain-status');
			var btn = $(this);
			var raw = (input.val() || '').toString().trim();
			if(!raw){ status.text('Enter a domain like example.com'); input.trigger('focus'); return; }
			btn.prop('disabled', true).text('Saving…');
			status.text('');
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action:'imgsmaller_set_domain', nonce: ImgSmallerDashboard.nonce, domain: raw }
			).done(function(res){
				if(res && res.success){
					var d = res.data || {};
					status.text(d.message || 'Saved.').css('color', '#2c7');
					var list = d.domains || [];
					$('.js-domain-list').text(list.length ? list.join(', ') : raw);
				}else{
					status.text((res && res.data && res.data.message) || 'Failed to save domain.').css('color', '#c22');
				}
			}).fail(function(){
				status.text('Failed to save domain.').css('color', '#c22');
			}).always(function(){
				btn.prop('disabled', false).text('Save Domain');
			});
		});
		setInterval(fetchStatus, 10000);
	});

	// Exclude Images browser (search + infinite scroll + selection)
	$(function(){
		var grid = $('.js-exclude-grid');
		if(!grid.length){ return; }

		var input = $('.js-exclude-ids-input');
		var searchField = $('.js-exclude-search');
		var sizeFilter = $('.js-exclude-size-filter');
		var typeFilter = $('.js-exclude-type-filter');
		var refreshBtn = $('.js-exclude-refresh');
		var loadMoreBtn = $('.js-exclude-load-more');
		var loading = $('.js-exclude-loading');
		var page = 1;
		var per = 10;
		var totalPages = 1;
		var query = '';
		var isLoading = false;
		var queuedReload = false;
		var debounceTimer = null;
		var reqToken = 0;
		var selected = new Set();

		function parseInput(){
			selected.clear();
			var val = (input.val() || '').toString();
			val.split(/[\s,]+/).forEach(function(p){
				var id = parseInt(p, 10);
				if(!isNaN(id) && id > 0){ selected.add(id); }
			});
		}

		function syncInput(){
			var arr = Array.from(selected);
			arr.sort(function(a,b){ return a-b; });
			input.val(arr.join(','));
		}

		function renderItems(items, append){
			if(!append){ grid.empty(); }
			items.forEach(function(it){
				var cell = $('<label class="imgsmaller-exclude-item" style="display:block; border:1px solid #eee; padding:6px; border-radius:3px; background:#fff;">');
				var img = $('<img>').attr('src', it.url || '').attr('alt', it.title || '').css({ width:'100%', height:'auto', display:'block', aspectRatio:'1/1', objectFit:'cover', background:'#f7f7f7' });
				var chk = $('<input type="checkbox" class="js-exclude-chk">').css({ marginTop:'4px' }).prop('checked', !!it.selected || selected.has(it.id)).data('id', it.id);
				var meta = $('<div class="imgsmaller-exclude-meta" style="margin-top:4px; font-size:12px; line-height:1.3; color:#444; word-break:break-word;">');
				var title = $('<div class="imgsmaller-exclude-title">').text(it.title || ('#' + it.id));
				var size = $('<div class="imgsmaller-exclude-size" style="opacity:0.8;">').text(it.size_h || '');
				meta.append(title).append(size);
				cell.append(img);
				cell.append(chk);
				cell.append(meta);
				grid.append(cell);
			});
		}

		function loadPage(reset){
			if(isLoading){
				queuedReload = reset || queuedReload; // ensure we reload with reset once current finishes
				return;
			}
			isLoading = true;
			loading.show();
			if(reset){ page = 1; totalPages = 1; grid.empty(); }
			var myToken = ++reqToken;
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action:'imgsmaller_media_search', nonce: ImgSmallerDashboard.nonce, q: query, page: page, per: per, size: (sizeFilter.val() || 'all'), type: (typeFilter.val() || 'all') }
			).done(function(res){
				// Ignore out-of-date responses
				if(myToken !== reqToken){ return; }
				if(res && res.success){
					totalPages = res.data.totalPages || 1;
					var items = res.data.items || [];
					if(reset && (!items || items.length === 0)){
						grid.empty().append($('<div class="imgsmaller-exclude-empty">').text('Not Found'));
					}else{
						renderItems(items, page > 1);
					}
				}
			}).always(function(){
				isLoading = false;
				loading.hide();
				if(queuedReload){ queuedReload = false; loadPage(true); }
			});
		}

		parseInput();
		loadPage(true);

		refreshBtn.on('click', function(){
			query = (searchField.val() || '').toString();
			loadPage(true);
		});

		searchField.on('keydown', function(e){ if(e.key === 'Enter'){ e.preventDefault(); refreshBtn.click(); } });

		// Active (debounced) search
		searchField.on('input', function(){
			query = (searchField.val() || '').toString();
			if(debounceTimer){ clearTimeout(debounceTimer); }
			debounceTimer = setTimeout(function(){ loadPage(true); }, 350);
		});

		// Size filter change
		sizeFilter.on('change', function(){
			loadPage(true);
		});

		// Type filter change
		typeFilter.on('change', function(){
			loadPage(true);
		});

		grid.on('change', '.js-exclude-chk', function(){
			var id = parseInt($(this).data('id'), 10);
			if(isNaN(id) || id <= 0){ return; }
			if($(this).is(':checked')){ selected.add(id); } else { selected.delete(id); }
			syncInput();
		});

		grid.on('scroll', function(){
			var el = grid[0];
			if(!el){ return; }
			if((el.scrollTop + el.clientHeight + 50) >= el.scrollHeight){
				if(!isLoading && page < totalPages){ page++; loadPage(false); }
			}
		});

		// Load 50 more on demand
		if(loadMoreBtn.length){
			loadMoreBtn.on('click', function(){
				if(isLoading){ return; }
				per = 50; // fetch 50 per request for this manual load
				if(page < totalPages){ page++; }
				loadPage(false);
			});
		}

		// Select all on current page
		$('.js-exclude-select-all').on('click', function(){
			grid.find('.js-exclude-chk').each(function(){
				var $chk = $(this);
				var id = parseInt($chk.data('id'), 10);
				if(isNaN(id) || id <= 0){ return; }
				$chk.prop('checked', true);
				selected.add(id);
			});
			syncInput();
		});

		// Clear selection on current page
		$('.js-exclude-clear').on('click', function(){
			grid.find('.js-exclude-chk').each(function(){
				var $chk = $(this);
				var id = parseInt($chk.data('id'), 10);
				if(isNaN(id) || id <= 0){ return; }
				$chk.prop('checked', false);
				selected.delete(id);
			});
			syncInput();
		});
	});

	// Restore (Individual) browser: search, load-more, select, restore-selected
	$(function(){
		var grid = $('.js-restore-grid');
		if(!grid.length){ return; }
		if(!window.ImgSmallerDashboard){
			grid.empty().append($('<div class="imgsmaller-exclude-empty">').text('Configuration not loaded (no global). Reload page.'));
			return;
		}

		var searchField = $('.js-restore-search');
		var refreshBtn = $('.js-restore-refresh');
		var loadMoreBtn = $('.js-restore-load-more');
		var loading = $('.js-restore-loading');
		var restoreBtn = $('.js-restore-selected');
		var selectedCountEl = $('.js-restore-selected-count');

		var page = 1;
		var per = 10;
		var totalPages = 1;
		var query = '';
		var isLoading = false;
		var debounceTimer = null;
		var reqToken = 0;
		var selected = new Set();

		function updateSelectedCount(){
			var count = selected.size;
			selectedCountEl.text(count + ' selected');
			restoreBtn.prop('disabled', count === 0);
		}

		function renderItems(items, append){
			if(!append){ grid.empty(); }
			items.forEach(function(it){
				var cell = $('<label class="imgsmaller-exclude-item" style="display:block; border:1px solid #eee; padding:6px; border-radius:3px; background:#fff;">');
				var img = $('<img>').attr('src', it.url || '').attr('alt', it.title || '').css({ width:'100%', height:'auto', display:'block', aspectRatio:'1/1', objectFit:'cover', background:'#f7f7f7' });
				var chk = $('<input type="checkbox" class="js-restore-chk">').css({ marginTop:'4px' }).prop('checked', selected.has(it.id)).data('id', it.id);
				var meta = $('<div class="imgsmaller-exclude-meta" style="margin-top:4px; font-size:12px; line-height:1.3; color:#444; word-break:break-word;">');
				var title = $('<div class="imgsmaller-exclude-title">').text(it.title || ('#' + it.id));
				meta.append(title);
				cell.append(img);
				cell.append(chk);
				cell.append(meta);
				grid.append(cell);
			});
		}

		function loadPage(reset){
			if(isLoading){ return; }
			isLoading = true;
			loading.show();
			if(reset){ page = 1; totalPages = 1; grid.empty(); }
			var myToken = ++reqToken;
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action:'imgsmaller_restore_search', nonce: ImgSmallerDashboard.nonce, q: query, page: page, per: per }
			).done(function(res){
				if(myToken !== reqToken){ return; }
				if(res && res.success){
					totalPages = res.data.totalPages || 1;
					var items = res.data.items || [];
					if(reset && (!items || items.length === 0)){
						grid.empty().append($('<div class="imgsmaller-exclude-empty">').text('No backed up images found.')); // fallback message
					}else{
						renderItems(items, page > 1);
					}
				}else{
					grid.empty().append($('<div class="imgsmaller-exclude-empty">').text('Error loading images.'));
				}
			}).fail(function(){
				grid.empty().append($('<div class="imgsmaller-exclude-empty">').text('Request failed. Check console/logs.'));
			}).always(function(){
				isLoading = false;
				loading.hide();
				updateSelectedCount();
			});
		}

		refreshBtn.on('click', function(){
			query = (searchField.val() || '').toString();
			loadPage(true);
		});

		searchField.on('keydown', function(e){ if(e.key === 'Enter'){ e.preventDefault(); refreshBtn.click(); } });

		searchField.on('input', function(){
			query = (searchField.val() || '').toString();
			if(debounceTimer){ clearTimeout(debounceTimer); }
			debounceTimer = setTimeout(function(){ loadPage(true); }, 350);
		});

		grid.on('change', '.js-restore-chk', function(){
			var id = parseInt($(this).data('id'), 10);
			if(isNaN(id) || id <= 0){ return; }
			if($(this).is(':checked')){ selected.add(id); } else { selected.delete(id); }
			updateSelectedCount();
		});

		grid.on('scroll', function(){
			var el = grid[0];
			if(!el){ return; }
			if((el.scrollTop + el.clientHeight + 50) >= el.scrollHeight){
				if(!isLoading && page < totalPages){ page++; loadPage(false); }
			}
		});

		if(loadMoreBtn.length){
			loadMoreBtn.on('click', function(){
				if(isLoading){ return; }
				per = 50;
				if(page < totalPages){ page++; }
				loadPage(false);
			});
		}

		// Select all on current page
		$('.js-restore-select-all').on('click', function(){
			grid.find('.js-restore-chk').each(function(){
				var $chk = $(this);
				var id = parseInt($chk.data('id'), 10);
				if(isNaN(id) || id <= 0){ return; }
				$chk.prop('checked', true);
				selected.add(id);
			});
			updateSelectedCount();
		});

		// Clear selection on current page
		$('.js-restore-clear').on('click', function(){
			grid.find('.js-restore-chk').each(function(){
				var $chk = $(this);
				var id = parseInt($chk.data('id'), 10);
				if(isNaN(id) || id <= 0){ return; }
				$chk.prop('checked', false);
				selected.delete(id);
			});
			updateSelectedCount();
		});

		restoreBtn.on('click', function(){
			if(selected.size === 0){ window.alert('Please select at least one image.'); return; }
			var ids = Array.from(selected);
			restoreBtn.prop('disabled', true).text('Restoring…');
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action:'imgsmaller_restore_selected', nonce: ImgSmallerDashboard.nonce, ids: ids }
			).done(function(res){
				if(res && res.success){
					var data = res.data || {};
					if(data.status){ ImgSmallerDashboard.status = data.status; }
					if(data.logs){ ImgSmallerDashboard.logs = data.logs; }
					updateDashboard(data);
					// If job started and not done, start incremental loop (reuse main restore loop)
					if(data.started && !data.done){
						function pollStep(){
							$.post(
								ImgSmallerDashboard.ajaxUrl,
								{ action: 'imgsmaller_restore', nonce: ImgSmallerDashboard.nonce, time_limit: 8 }
							).done(function(resp){
								if(resp && resp.success){
									var d = resp.data || {};
									if(d.settings){ ImgSmallerDashboard.settings = d.settings; }
									if(d.status){ ImgSmallerDashboard.status = d.status; }
									if(d.logs){ ImgSmallerDashboard.logs = d.logs; }
									updateDashboard(d);
									refreshRestoreStatus(ImgSmallerDashboard.status, d.message);
									if(!d.done){ setTimeout(pollStep, 300); }
								}
							});
						}
						pollStep();
					}
					// Clear selection after action and refresh UI
					selected.clear();
					updateSelectedCount();
					loadPage(true);
				}else{
					window.alert((res && res.data && res.data.message) || 'Restore failed.');
				}
			}).fail(function(){
				window.alert('Restore failed.');
			}).always(function(){
				restoreBtn.prop('disabled', selected.size === 0).text('Restore Selected');
			});
		});

		// initial load
		loadPage(true);
		updateSelectedCount();
	});

	// Media library scan with pie chart
	$(function(){
		var btn = $('.js-scan-media');
		var canvas = document.querySelector('.js-scan-pie');
		var ctx = canvas ? canvas.getContext('2d') : null;
		var running = false;
		var firstCall = true;

		function formatKB(bytes){
			var kb = (bytes || 0) / 1024;
			return (Math.round(kb * 100) / 100) + ' KB';
		}

		function drawPie(optimized, nonOpt){
			if(!ctx || !canvas){ return; }
			var total = Math.max(1, optimized + nonOpt);
			var start = -Math.PI/2;
			ctx.clearRect(0,0,canvas.width,canvas.height);
			var cx = canvas.width/2, cy = canvas.height/2, r = Math.min(cx, cy) - 10;
			function slice(value, color){
				var angle = (value/total) * Math.PI * 2;
				ctx.beginPath();
				ctx.moveTo(cx, cy);
				ctx.arc(cx, cy, r, start, start + angle);
				ctx.closePath();
				ctx.fillStyle = color;
				ctx.fill();
				start += angle;
			}
			slice(optimized, '#2ecc71'); // green
			slice(nonOpt, '#bdc3c7'); // gray
			// inner circle for donut look
			ctx.beginPath();
			ctx.arc(cx, cy, r*0.55, 0, Math.PI*2);
			ctx.fillStyle = '#ffffff';
			ctx.fill();
		}

		function updateScanUI(t){
			t = t || {};
			var total = t.total_images || 0;
			var opt = t.optimized_images || 0;
			var non = t.non_optimized_images || 0;
			$('.js-scan-total-images').text(total);
			$('.js-scan-optimized').text(opt);
			$('.js-scan-nonoptimized').text(non);
			$('.js-scan-total-size').text(formatKB(t.total_bytes||0));
			$('.js-scan-saved').text(formatKB(t.saved_bytes||0));
			$('.js-scan-potential').text(formatKB(t.estimated_potential_bytes||0));
			drawPie(opt, non);
		}

		function doStep(){
			$.post(
				ImgSmallerDashboard.ajaxUrl,
				{ action: 'imgsmaller_scan', nonce: ImgSmallerDashboard.nonce, time_limit: 8, reset: firstCall ? 1 : 0 }
			).done(function(res){
				if(res && res.success){
					var data = res.data || {};
					updateScanUI(data.totals||{});
					if(!data.done){
						firstCall = false;
						setTimeout(doStep, 200);
					}else{
						running = false;
						btn.prop('disabled', false).text(btn.data('defaultLabel') || 'Check Now');
					}
				}else{
					running = false;
					btn.prop('disabled', false).text('Scan Images Now');
					window.alert((res && res.data && res.data.message) || 'Scan failed.');
				}
			}).fail(function(){
				running = false;
				btn.prop('disabled', false).text(btn.data('defaultLabel') || 'Check Now');
				window.alert('Scan failed.');
			});
		}

		if(btn.length){
			if(!btn.data('defaultLabel')){ btn.data('defaultLabel', btn.attr('data-default-label') || btn.text() || 'Check Now'); }
			if(!btn.data('workingLabel')){ btn.data('workingLabel', btn.attr('data-working-label') || 'Checking…'); }
			btn.on('click', function(){
				if(running){ return; }
				running = true;
				firstCall = true;
				btn.prop('disabled', true).text(btn.data('workingLabel'));
				doStep();
			});
		}

	});
})(jQuery);

