<?php

/*
Plugin Name: Image Alt Sync
Description: Replace <img> alt attributes in post content with the media library alt text. Batch processing with status/date filters, post ID ranges, skip & exclude options, dry run logs (browser & WP-CLI).
Version: 1.4.11
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
Author: Laurent DUFOUR
Author URI: https://www.laurentdufour.eu/
Text Domain: image-alt-sync
*/

/*

Copyright (C) 2025  Laurent DUFOUR (hotmail.com ID: dufour_l)

*/

if ( ! defined( 'ABSPATH' ) ) { exit; }
if(!defined('IASPLUGIN_VERSION')) define('IASPLUGIN_VERSION','1.4.11');


class Image_Alt_Sync {
    const VERSION = '1.4.11';
    const SLUG = 'image-alt-sync';
    const NONCE = 'iasplugin_nonce';

    public function __construct() {
        add_action('admin_menu', [$this, 'iasplugin_add_admin_page']);
        add_action('admin_enqueue_scripts', [$this, 'iasplugin_enqueue_assets']);
        add_action('wp_ajax_iasplugin_prepare', [$this, 'iasplugin_ajax_prepare']);
        add_action('wp_ajax_iasplugin_process_batch', [$this, 'iasplugin_ajax_process_batch']);
        add_action('wp_ajax_iasplugin_get_lowest_id', [$this, 'iasplugin_ajax_get_lowest_id']);
        add_action('wp_ajax_iasplugin_get_highest_id', [$this, 'iasplugin_ajax_get_highest_id']);
        if ( defined( 'WP_CLI' ) && WP_CLI ) {
            WP_CLI::add_command('image-alt-sync', [$this, 'iasplugin_cli_command']);
        }
    }

    public function iasplugin_add_admin_page() {
        add_management_page(
            __('Image Alt Sync', 'image-alt-sync'),
            __('Image Alt Sync', 'image-alt-sync'),
            'manage_options',
            self::SLUG,
            [$this, 'iasplugin_render_admin_page']
        );
    }

    public function iasplugin_enqueue_assets($hook) {
        if ($hook !== 'tools_page_' . self::SLUG) { return; }
        wp_enqueue_script(self::SLUG . '-admin', plugins_url('assets/admin.js', __FILE__), ['jquery'], self::VERSION, true);
        wp_localize_script(self::SLUG . '-admin', 'IASPLUGIN', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce(self::NONCE),
            'i18n'    => [
                'starting' => __('Starting…', 'image-alt-sync'),
                'processing' => __('Processing…', 'image-alt-sync'),
                'done' => __('Done.', 'image-alt-sync'),
                'error' => __('Error', 'image-alt-sync'),
            ],
        ]);
        wp_enqueue_style(self::SLUG . '-admin-style', plugins_url('assets/admin.css', __FILE__), [], self::VERSION);
        $css = '.ias-card{background:#fff;border:1px solid #ddd;border-radius:12px;padding:16px;max-width:980px}
        .ias-controls{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:12px}
        .ias-controls .full{grid-column:1 / -1}
        .ias-row{display:flex;gap:12px;align-items:center}
        .ias-btn{background:#2271b1;color:#fff;border:none;border-radius:8px;padding:8px 12px;cursor:pointer}
        .ias-btn.secondary{background:#555}
        .ias-btn:disabled{opacity:.6;cursor:not-allowed}
        .ias-badges button{margin-right:6px;margin-bottom:6px}
        .ias-log{max-height:420px;overflow:auto;background:#0b1021;color:#e5e7eb;border-radius:8px;padding:12px;font-family:ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace}
        .ias-log a{color:#7dd3fc;text-decoration:underline}
        .ias-grid-3{display:grid;grid-template-columns:repeat(3,minmax(0,1fr));gap:12px}
        .ias-grid-2{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:12px}
        .ias-muted{color:#666}
        .ias-table{width:100%;border-collapse:collapse;margin-top:8px}
        .ias-table th,.ias-table td{border-bottom:1px solid #344; padding:6px;vertical-align:top}
        ';
        wp_add_inline_style(self::SLUG . '-admin-style', $css);
    }

    public function iasplugin_render_admin_page() {
        ?>
        <div class="wrap">
					 
            <h1><?php esc_html_e('Image Alt Sync', 'image-alt-sync'); ?> <span class="ias-muted">v<?php echo esc_html(self::VERSION); ?></span></h1>
													
			<h2><?php esc_html_e('The Image Alt Sync plugin is installed and working correctly.', 'image-alt-sync'); ?></h2>
			<p><?php esc_html_e('This plugin will replace <img> tag alt attributes in posts with the alt stored in the media library.', 'image-alt-sync'); ?></p>
			<p><?php esc_html_e('This is a major revision in the way the plugin works.', 'image-alt-sync');?> <a href="mailto:dufour_l@hotmail.com"> Please report all bugs</a> <?php esc_html_e('as soon as possible.', 'image-alt-sync'); ?></p>
			<p><br></p>

													
            <div class="ias-card">
                <div class="ias-controls">
                    <div>
											
                        <label><strong><?php esc_html_e('Post Status', 'image-alt-sync'); ?></strong></label><br/>
                        <select id="ias-status">
														 
                            <option value="publish"><?php esc_html_e('publish', 'image-alt-sync'); ?></option>
														 
                            <option value="pending"><?php esc_html_e('pending', 'image-alt-sync'); ?></option>
													   
                            <option value="draft"><?php esc_html_e('draft', 'image-alt-sync'); ?></option>
							
							<option value="private"><?php esc_html_e('private', 'image-alt-sync'); ?></option>
													 
                            <option value="any"><?php esc_html_e('any', 'image-alt-sync'); ?></option>
                        </select>
                    </div>
                    <div>
											
                        <label><strong><?php esc_html_e('Batch Size', 'image-alt-sync'); ?></strong></label><br/>
                        <input type="number" id="ias-batch-size" value="100" min="1" step="1" />
                    </div>
                    <div>
											
                        <label><strong><?php esc_html_e('Delay Between Batches (seconds)', 'image-alt-sync'); ?></strong></label><br/>
                        <input type="number" id="ias-delay" value="1" min="0" step="0.5" />
                    </div>
                    <div>
											
                        <label><strong><?php esc_html_e('Dry Run (no changes)', 'image-alt-sync'); ?></strong></label><br/>
                        <select id="ias-dry-run">
												   
                            <option value="1"><?php esc_html_e('Yes', 'image-alt-sync'); ?></option>
												   
                            <option value="0"><?php esc_html_e('No', 'image-alt-sync'); ?></option>
                        </select>
                    </div>
                    <div class="full">
											
                        <label><strong><?php esc_html_e('Date Range', 'image-alt-sync'); ?></strong></label>
                        <div class="ias-row">
                            <input type="date" id="ias-after" /> <span>→</span> <input type="date" id="ias-before" />
                        </div>
                        <div class="ias-badges" style="margin-top:8px">
                            <button class="ias-btn secondary" data-range="today"><?php esc_html_e('Today', 'image-alt-sync'); ?></button>
																										   
                            <button class="ias-btn secondary" data-range="yesterday"><?php esc_html_e('Yesterday', 'image-alt-sync'); ?></button>
																											   
                            <button class="ias-btn secondary" data-range="last_week"><?php esc_html_e('Last week', 'image-alt-sync'); ?></button>
							
							<button class="ias-btn secondary" data-range="last_4_weeks"><?php esc_html_e('Last 4 weeks', 'image-alt-sync'); ?></button>
																											   
                            <button class="ias-btn secondary" data-range="last_month"><?php esc_html_e('Last month', 'image-alt-sync'); ?></button>
																					
                            <button class="ias-btn secondary" data-range="fom"><?php esc_html_e('Since first of this month', 'image-alt-sync'); ?></button>
							
							<button class="ias-btn secondary" data-range="ytd"><?php esc_html_e('Since beginning of the year', 'image-alt-sync'); ?></button>
							
                            <button class="ias-btn secondary" data-range="all"><?php esc_html_e('All dates', 'image-alt-sync'); ?></button>
																											   
                        </div>
                    </div>

                    <div>
											
                        <label><strong><?php esc_html_e('Lowest Post ID', 'image-alt-sync'); ?></strong></label><br/>
                        <input type="number" id="ias-lowest-id" placeholder="<?php esc_attr_e('e.g. 1', 'image-alt-sync'); ?>"/>
																									  
                        <button class="ias-btn secondary" id="ias-fill-lowest"><?php esc_html_e('Lowest Post ID', 'image-alt-sync'); ?></button>
																													
                    </div>
                    <div>
											
                        <label><strong><?php esc_html_e('Highest Post ID', 'image-alt-sync'); ?></strong></label><br/>
                        <input type="number" id="ias-highest-id" placeholder="<?php esc_attr_e('e.g. 9999', 'image-alt-sync'); ?>"/>
																										 
                        <button class="ias-btn secondary" id="ias-fill-highest"><?php esc_html_e('Highest Post ID', 'image-alt-sync'); ?></button>
																													 
                    </div>

                    <div class="full">
																						 
                        <label><input type="checkbox" id="ias-skip-noimg" checked/> <?php esc_html_e('Skip posts without <img> tags', 'image-alt-sync'); ?></label>
                    </div>
                    <div class="full">
											
                        <label><strong><?php esc_html_e('Exclude image extensions (comma-separated)', 'image-alt-sync'); ?></strong></label><br/>
                        <textarea id="ias-exclude-ext" rows="2" placeholder=".svg,.gif"></textarea>
                    </div>

                    <div class="full">
																	
                        <button id="ias-start" class="ias-btn"><?php esc_html_e('Start Processing', 'image-alt-sync'); ?></button>
                        <button id="ias-stop" class="ias-btn secondary" disabled><?php esc_html_e('Stop', 'image-alt-sync'); ?></button>
																										  
                    </div>
                </div>
            </div>

											  
            <h2 style="margin-top:24px;"><?php esc_html_e('Logs', 'image-alt-sync'); ?></h2>
            <div id="ias-log" class="ias-log"></div>
        </div>
        <?php
														
    }

    private function iasplugin_sanitize_date($date) {
        if ( empty($date) ) return '';
        $t = strtotime( $date );
        if ( $t === false ) return '';
        return date('Y-m-d', $t);
    }

    public function iasplugin_ajax_get_lowest_id() {
        global $wpdb;
        $id = $wpdb->get_var("SELECT MIN(ID) FROM {$wpdb->posts} WHERE post_type='post'");
        wp_send_json_success(['id' => intval($id)]);
    }

    public function iasplugin_ajax_get_highest_id() {
        global $wpdb;
        $id = $wpdb->get_var("SELECT MAX(ID) FROM {$wpdb->posts} WHERE post_type='post'");
        wp_send_json_success(['id' => intval($id)]);
    }

    public function iasplugin_ajax_prepare() {
        check_ajax_referer(self::NONCE, 'nonce');
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['message' => 'Permission denied.'], 403);
        }

        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'publish';
        $after  = $this->iasplugin_sanitize_date( preg_replace("([^0-9/])", "", $_POST['after']) ?? '' );
        $before = $this->iasplugin_sanitize_date( preg_replace("([^0-9/])", "", $_POST['before']) ?? '' );
        $low_id = intval($_POST['low_id'] ?? 0);
        $high_id= intval($_POST['high_id'] ?? 0);
        $skip_noimg = !empty($_POST['skip_noimg']);
        $exclude_ext = array_filter(array_map('trim', explode(',', map_deep( $_POST['exclude_ext'], 'sanitize_text_field' ) ?? '')));
		
		

        // Build query
        $args = [
            'post_type'      => 'post',
            'post_status'    => $status === 'any' ? 'any' : $status,
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'orderby'        => 'ID',
            'order'          => 'ASC',
        ];

        if ( $after || $before ) {
            $dq = ['inclusive' => true];
            if ($after)  { $dq['after']  = $after; }
            if ($before) { $dq['before'] = $before; }
            $args['date_query'] = [$dq];
        }

        $ids = get_posts($args);

        // Filter by ID range, <img> presence, excluded extensions, and at least one image attachment
        $filtered = [];
        foreach ($ids as $id) {
            if ($low_id && $id < $low_id) continue;
            if ($high_id && $id > $high_id) continue;
            $post = get_post($id);
            if ( ! $post ) continue;

            // Must include at least one image attachment
            $attached = get_children([
                'post_parent' => $id,
                'post_type'   => 'attachment',
                'post_mime_type' => 'image',
                'fields'      => 'ids',
                'numberposts' => 1,
            ]);
            if ( empty($attached) ) continue;

            $has_img = stripos($post->post_content, '<img') !== false;
            if ($skip_noimg && ! $has_img) continue;

            if (!empty($exclude_ext)) {
                $skip = false;
                foreach ($exclude_ext as $ext) {
                    $ext = trim($ext);
                    if ($ext && stripos($post->post_content, $ext) !== false) { $skip = true; break; }
                }
                if ($skip) continue;
            }
            $filtered[] = $id;
        }

        wp_send_json_success(['ids' => array_map('intval', $filtered)]);
    }

    public function iasplugin_ajax_process_batch() {
        check_ajax_referer(self::NONCE, 'nonce');
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error(['message' => 'Permission denied.'], 403);
        }

        $ids      = isset($_POST['ids']) ? array_map('intval', (array) $_POST['ids']) : [];
        $dry_run  = isset($_POST['dry_run']) ? (bool) intval($_POST['dry_run']) : true;

        $batch_log = [];
        foreach ( $ids as $post_id ) {
            $result = $this->iasplugin_process_post_content_alts($post_id, $dry_run);
            $batch_log[] = $result;
        }

        wp_send_json_success(['log' => $batch_log]);
    }

    private function iasplugin_get_filename_from_attachment($aid){
        $url = wp_get_attachment_url($aid);
        if (!$url) return '';
        $parts = wp_parse_url($url);
        if (!empty($parts['path'])) {
            return basename($parts['path']);
        }
        return '';
    }

    private function iasplugin_process_post_content_alts( $post_id, $dry_run = true ) {
        $post = get_post( $post_id );
        if ( ! $post ) {
            return [
                'post_id' => $post_id,
                'post_title' => '(missing)',
                'changes' => [],
                'error' => 'Post not found',
                'edit_link' => admin_url( 'post.php?post=' . intval($post_id) . '&action=edit' ),
            ];
        }

        $content = $post->post_content;
        $changes = [];

        // Find <img ... wp-image-XXX ...> and pull attachment ID
        if (preg_match_all('/<img[^>]*class=["\'][^"\']*wp-image-(\d+)[^"\']*[^>]*>/i', $content, $m, PREG_OFFSET_CAPTURE)) {
            foreach ($m[0] as $i => $match) {
                $full_tag = $match[0];
                $aid = intval($m[1][$i][0]);
                $media_alt = get_post_meta($aid, '_wp_attachment_image_alt', true);
                $old_alt = '';

                // Extract current alt attr
                if (preg_match('/alt=["\']([^"\']*)["\']/i', $full_tag, $am)) {
                    $old_alt = $am[1];
                }

                if ($media_alt !== $old_alt) {
                    $filename = $this->iasplugin_get_filename_from_attachment($aid);
                    $changes[] = [
                        'attachment_id' => $aid,
                        'filename' => $filename,
                        'old' => $old_alt,
                        'new' => $media_alt,
                        'attachment_edit' => admin_url( 'post.php?post=' . intval($aid) . '&action=edit' ),
                    ];

                    if (! $dry_run) {
                        // Replace or inject alt attr in this tag only
                        $new_tag = $full_tag;
                        if (preg_match('/alt=["\'][^"\']*["\']/i', $new_tag)) {
                            $new_tag = preg_replace('/alt=["\'][^"\']*["\']/i', 'alt="' . esc_attr($media_alt) . '"', $new_tag, 1);
                        } else {
                            // inject after <img
                            $new_tag = preg_replace('/<img/i', '<img alt="' . esc_attr($media_alt) . '"', $new_tag, 1);
                        }
                        // Replace in content
                        $content = str_replace($full_tag, $new_tag, $content);
                    }
                }
            }
        }

        if (! $dry_run && !empty($changes)) {
            wp_update_post(['ID' => $post_id, 'post_content' => $content]);
        }

        return [
            'post_id' => $post_id,
            'post_title' => get_the_title( $post_id ),
            'edit_link' => admin_url( 'post.php?post=' . intval($post_id) . '&action=edit' ),
            'changes' => $changes,
            'changed_count' => count($changes),
        ];
    }

    /**
     * WP-CLI usage:
     * wp image-alt-sync run --status=publish --after=2025-01-01 --before=2025-08-13 --batch-size=100 --delay=1 --dry-run --low-id=1 --high-id=9999 --skip-noimg=1 --exclude-ext=".svg,.gif"
     */
    public function iasplugin_cli_command( $args, $assoc_args ) {
        $status = $assoc_args['status'] ?? 'publish';
        $after  = $this->iasplugin_sanitize_date( preg_replace("([^0-9/])", "", $assoc_args['after']) ?? '' );
		$before = $this->iasplugin_sanitize_date( preg_replace("([^0-9/])", "", $assoc_args['before']) ?? '' );
        $batch_size = isset($assoc_args['batch-size']) ? max(1, intval($assoc_args['batch-size'])) : 100;
        $delay = isset($assoc_args['delay']) ? max(0, floatval($assoc_args['delay'])) : 1.0;
        $dry_run = isset($assoc_args['dry-run']) ? (bool)$assoc_args['dry-run'] : false;
        $low_id = isset($assoc_args['low-id']) ? intval($assoc_args['low-id']) : 0;
        $high_id = isset($assoc_args['high-id']) ? intval($assoc_args['high-id']) : 0;
        $skip_noimg = isset($assoc_args['skip-noimg']) ? (bool)$assoc_args['skip-noimg'] : false;
        $exclude_ext = array_filter(array_map('trim', explode(',', map_deep( $assoc_args['exclude-ext'], 'sanitize_text_field' ) ?? '')));
		

        if ( $after && $before && strtotime($after) > strtotime($before) ) {
            WP_CLI::error("'after' must be earlier than or equal to 'before'.");
            return;
        }

        $args = [
            'post_type'      => 'post',
            'post_status'    => $status === 'any' ? 'any' : $status,
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'orderby'        => 'ID',
            'order'          => 'ASC',
        ];
        if ( $after || $before ) {
            $dq = ['inclusive' => true];
            if ($after)  { $dq['after']  = $after; }
            if ($before) { $dq['before'] = $before; }
            $args['date_query'] = [$dq];
        }
        $ids = get_posts($args);

        // Filter like in AJAX
        $ids = array_values(array_filter($ids, function($id) use($low_id,$high_id,$skip_noimg,$exclude_ext){
            if ($low_id && $id < $low_id) return false;
            if ($high_id && $id > $high_id) return false;
            $p = get_post($id);
            if (!$p) return false;

            $attached = get_children([
                'post_parent' => $id,
                'post_type'   => 'attachment',
                'post_mime_type' => 'image',
                'fields'      => 'ids',
                'numberposts' => 1,
            ]);
            if ( empty($attached) ) return false;

            $has_img = stripos($p->post_content, '<img') !== false;
            if ($skip_noimg && ! $has_img) return false;

            if (!empty($exclude_ext)) {
                foreach ($exclude_ext as $ext) {
                    $ext = trim($ext);
                    if ($ext && stripos($p->post_content, $ext) !== false) return false;
                }
            }
            return true;
        }));

        $batches = array_chunk($ids, $batch_size);
        $total_changes = 0;
        foreach ($batches as $i => $batch) {
            WP_CLI::log(sprintf("Batch %d/%d (size %d)%s", $i+1, count($batches), count($batch), $dry_run ? " [DRY RUN]" : ""));
            foreach ($batch as $post_id) {
                $res = $this->iasplugin_process_post_content_alts($post_id, $dry_run);
                $post_line = sprintf("#%d %s (%d change%s)", $res['post_id'], $res['post_title'], $res['changed_count'], $res['changed_count'] === 1 ? '' : 's');
                WP_CLI::log($post_line);
                if (!empty($res['changes'])) {
                    foreach ($res['changes'] as $ch) {
                        $fname = $ch['filename'] ?? '';
                        WP_CLI::log(sprintf("  - Attachment #%d%s: \"%s\"  =>  \"%s\"",
                            $ch['attachment_id'],
                            $fname ? " [{$fname}]" : '',
                            $ch['old'],
                            $ch['new']
                        ));
                    }
                }
                $total_changes += $res['changed_count'];
            }
            if ( $delay > 0 && $i < count($batches) - 1 ) {
                usleep((int)($delay * 1000000));
            }
        }
        WP_CLI::success(sprintf("Completed. Total <img> alt changes: %d%s", $total_changes, $dry_run ? " (dry run, no changes applied)" : ""));
    }
}

new Image_Alt_Sync();
