(function($){

    // Robust error -> text helper to avoid "[object Object]"
    function iaspluginToText(e){
        try {
            if (e && e instanceof Error) return (e.message || e.toString()) + (e.stack ? "\n" + e.stack : "");
            if (typeof e === 'object') return JSON.stringify(e, null, 2);
            return String(e);
        } catch(_){ return String(e); }
    }
    function appendLogSafe(msgHtml){
        try { $log.append(msgHtml); $log.scrollTop($log[0].scrollHeight); }
        catch(_) { /* no-op */ }
    }
    window.addEventListener('error', function(ev){
        appendLogSafe('<div style="color:#fca5a5">⚠️ JS Error: <pre style="white-space:pre-wrap;margin:0">'+ esc(iaspluginToText(ev.error || ev.message)) +'</pre></div>');
    });
    window.addEventListener('unhandledrejection', function(ev){
        appendLogSafe('<div style="color:#fca5a5">⚠️ Promise rejection: <pre style="white-space:pre-wrap;margin:0">'+ esc(iaspluginToText(ev.reason)) +'</pre></div>');
    });

    const $start = $('#ias-start');
    const $stop  = $('#ias-stop');
    const $log   = $('#ias-log');
    const $status = $('#ias-status');
    const $batchSize = $('#ias-batch-size');
    const $delay = $('#ias-delay');
    const $dryRun = $('#ias-dry-run');
    const $after = $('#ias-after');
    const $before = $('#ias-before');
    const $low = $('#ias-lowest-id');
    const $high = $('#ias-highest-id');
    const $skipNoImg = $('#ias-skip-noimg');
    const $excludeExt = $('#ias-exclude-ext');

    let state = { ids: [], cursor: 0, stopping: false };

    function log(html){
        $log.append(html + "\n");
        $log.scrollTop($log[0].scrollHeight);
    }
    function esc(s){
        return $('<div/>').text(String(s ?? '')).html();
    }

    function setRange(kind){
        const today = new Date();
        let a = '', b = '';
        function fmt(d){ return d.toISOString().slice(0,10); }
        switch(kind){
            case 'today':
                a = fmt(new Date(today.getFullYear(), today.getMonth(), today.getDate()));
                b = a;
                break;
            case 'yesterday': {
                const y = new Date(today.getFullYear(), today.getMonth(), today.getDate()-1);
                a = fmt(y); b = fmt(y);
                break;
            }
            case 'last_week': {
                const end = new Date(today.getFullYear(), today.getMonth(), today.getDate());
                const start = new Date(end); start.setDate(start.getDate()-7);
                a = fmt(start); b = fmt(end);
                break;
            }
            case 'last_4_weeks': {
                const end = new Date(today.getFullYear(), today.getMonth(), today.getDate());
                const start = new Date(end); start.setMonth(start.getMonth()-1);
                a = fmt(start); b = fmt(end);
                break;
            }
            case 'last_month': {
                const end = new Date(today.getFullYear(), today.getMonth()-1, today.getDate());
                const start = new Date(new Date().getFullYear(), new Date().getMonth()-1, 1, 0, 0, 0, 0);
                a = fmt(start); b = fmt(end);
                break;
            }
			case 'fom': {
                const end = new Date(today.getFullYear(), today.getMonth(), today.getDate());
                const start = new Date(new Date().getFullYear(), new Date().getMonth(), 1, 0, 0, 0, 0);
                a = fmt(start); b = fmt(end);
                break;
            }
            case 'ytd': {
                const start = new Date(today.getFullYear(), 0, 1);
                a = fmt(start); b = fmt(new Date(today.getFullYear(), today.getMonth(), today.getDate()));
                break;
            }
            case 'all':
                a = ''; b = '';
				b = fmt(new Date(today.getFullYear(), today.getMonth(), today.getDate()));
				a = fmt(new Date(0));
                break;
        }
        $after.val(a); $before.val(b);
    }

    $('.ias-badges').on('click', 'button[data-range]', function(e){
        e.preventDefault();
        const r = $(this).data('range');
        setRange(r);
    });

    $('#ias-fill-lowest').on('click', function(e){
        e.preventDefault();
        $.post(IASPLUGIN.ajaxurl,{action:'iasplugin_get_lowest_id',nonce:IASPLUGIN.nonce},function(res){
            }).fail(function(err){ appendLogSafe('<div style="color:#fca5a5">⚠️ Lowest ID error: <pre style="white-space:pre-wrap;margin:0">'+esc(iaspluginToText(err))+'</pre></div>');
            if(res && res.success) $low.val(res.data.id);
        });
    });
    $('#ias-fill-highest').on('click', function(e){
        e.preventDefault();
        $.post(IASPLUGIN.ajaxurl,{action:'iasplugin_get_highest_id',nonce:IASPLUGIN.nonce},function(res){
            }).fail(function(err){ appendLogSafe('<div style="color:#fca5a5">⚠️ Highest ID error: <pre style="white-space:pre-wrap;margin:0">'+esc(iaspluginToText(err))+'</pre></div>');
            if(res && res.success) $high.val(res.data.id);
        });
    });

    function validateDates(){
        const a = $after.val();
        const b = $before.val();
        if (a && b) {
            const ad = new Date(a);
            const bd = new Date(b);
            if (ad.getTime() > bd.getTime()) {
                alert('"Date after" must be earlier than or equal to "date before".');
                return false;
            }
        }
        return true;
    }

    async function prepare(){
        const postData = {
            action: 'iasplugin_prepare',
            nonce: IASPLUGIN.nonce,
            status: $status.val(),
            after: $after.val(),
            before: $before.val(),
            low_id: $low.val(),
            high_id: $high.val(),
            skip_noimg: $skipNoImg.is(':checked')?1:0,
            exclude_ext: $excludeExt.val()
        };
        const res = await $.post(IASPLUGIN.ajaxurl, postData);
        if (!res || !res.success) throw new Error(res?.data?.message || 'Failed to prepare');
        return res.data.ids || [];
    }

    async function processBatch(ids){
        const postData = {
            action: 'iasplugin_process_batch',
            nonce: IASPLUGIN.nonce,
            ids: ids,
            dry_run: $dryRun.val()
        };
        const res = await $.post(IASPLUGIN.ajaxurl, postData);
        if (!res || !res.success) throw new Error(res?.data?.message || 'Failed to process batch');
        return res.data.log || [];
    }

    async function run(){
        if (!validateDates()) return;
        $start.prop('disabled', true);
        $stop.prop('disabled', false);
        state.stopping = false;
        log('<div>▶️ <strong>Starting…</strong></div>');

        try{
            state.ids = await prepare();
            state.cursor = 0;
            log('<div>Found <strong>' + esc(state.ids.length) + '</strong> posts to process.</div>');

            const size = Math.max(1, parseInt($batchSize.val(), 10) || 100);
            const delaySec = Math.max(0, parseFloat($delay.val()) || 0);

            while (state.cursor < state.ids.length){
                if (state.stopping) break;
                const chunk = state.ids.slice(state.cursor, state.cursor + size);
                log('<div class="ias-muted">Processing IDs ' + esc(state.cursor+1) + '–' + esc(state.cursor+chunk.length) + '…</div>');
                let results = [];
                try {
                    results = await processBatch(chunk);
                } catch(batchErr){
                    appendLogSafe('<div style="color:#fca5a5">⚠️ Batch failed: <pre style="white-space:pre-wrap;margin:0">'+esc(iaspluginToText(batchErr))+'</pre></div>');
                }

                results.forEach(r => {
                    const header = '<div>Post <a target="_blank" href="'+esc(r.edit_link)+'">#'+esc(r.post_id)+'</a> — '+esc(r.post_title)+' ('+esc(r.changed_count)+' change'+(r.changed_count===1?'':'s')+')</div>';
                    let table = '';
                    if (r.changes && r.changes.length){
                        table += '<table class="ias-table"><thead><tr><th>Attachment</th><th>Filename</th><th>Old alt</th><th>New alt</th></tr></thead><tbody>';
                        r.changes.forEach(ch => {
                            table += '<tr><td><a target="_blank" href="'+esc(ch.attachment_edit)+'">#'+esc(ch.attachment_id)+'</a></td><td>'+esc(ch.filename||"")+'</td><td>'+esc(ch.old||'')+'</td><td>'+esc(ch.new||'')+'</td></tr>';
                        });
                        table += '</tbody></table>';
                    } else {
                        table = '<div class="ias-muted">No <code>&lt;img&gt;</code> alt changes.</div>';
                    }
                    log('<div style="margin:8px 0 14px 0">'+header+table+'</div>');
                });

                state.cursor += chunk.length;

                if (delaySec > 0 && state.cursor < state.ids.length){
                    await new Promise(res => setTimeout(res, delaySec*1000));
                }
            }

            if (state.stopping){
                log('<div>⏹ Stopped by user.</div>');
            } else {
                log('<div>✅ <strong>Done.</strong></div>');
            }
        } catch(err){ appendLogSafe('<div style="color:#fca5a5">⚠️ '+esc(iaspluginToText(err))+'</div>'); } finally {
            $start.prop('disabled', false);
            $stop.prop('disabled', true);
        }
    }

    $start.on('click', function(e){
        e.preventDefault();
        run();
    });

    $stop.on('click', function(e){
        e.preventDefault();
        state.stopping = true;
        $stop.prop('disabled', true);
    });

})(jQuery);
