<?php
/**
 * Plugin Name:       IASM - AI Search Visibility Monitor
 * Plugin URI:        https://iaseogenerator.com/producto/iaseomonitor/
 * Description:       Check whether your site’s AI-readable files are present and reachable (llms.txt, humans.txt, metadata.json, sitemap-ia.xml, ai-plugin.json).
 * Version:           1.0.0
 * Requires at least: 6.0
 * Requires PHP:      7.4
 * Author:            Mediterranea Services
 * Author URI:        https://www.mediterraneaservices.com
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       iasm-ai-search-visibility-monitor
 * Domain Path:       /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'IASM_VERSION', '1.0.0' ); 
/**
 * Add admin menu entry.
 */
function iasm_add_admin_menu() {
	add_menu_page(
		esc_html__( 'IASM - AI Search Visibility Monitor', 'iasm-ai-search-visibility-monitor' ),
		esc_html__( 'IASM - AI Search Visibility Monitor', 'iasm-ai-search-visibility-monitor' ),
		'manage_options',
		'iasm-ai-search-visibility-monitor',
		'iasm_render_admin_page',
		'dashicons-visibility',
		3
	);
}
add_action( 'admin_menu', 'iasm_add_admin_menu' );

/**
 * Enqueue admin assets only on our screen.
 */
function iasm_admin_assets( $hook ) {
	if ( 'toplevel_page_iasm-ai-search-visibility-monitor' !== $hook ) {
		return;
	}

	$css = '
	.iasm-wrap{max-width:1100px}
	.iasm-hero{background:#fff;border:1px solid #e5e7eb;border-radius:14px;padding:18px 18px 14px;margin:14px 0 16px;box-shadow:0 1px 1px rgba(0,0,0,.03)}
	.iasm-hero h1{margin:0 0 6px;font-size:22px;line-height:1.2}
	.iasm-hero p{margin:0;color:#4b5563}
	.iasm-badges{display:flex;gap:8px;flex-wrap:wrap;margin-top:10px}
	.iasm-badge{display:inline-flex;align-items:center;gap:6px;border:1px solid #e5e7eb;border-radius:999px;padding:4px 10px;font-size:12px;background:#f9fafb}
	.iasm-grid{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:14px;margin-top:14px}
	@media (max-width:820px){.iasm-grid{grid-template-columns:1fr}}
	.iasm-card{background:#fff;border:1px solid #e5e7eb;border-radius:14px;padding:14px 14px 12px;box-shadow:0 1px 1px rgba(0,0,0,.03)}
	.iasm-card h3{margin:0 0 8px;font-size:15px;display:flex;align-items:center;gap:8px}
	.iasm-meta{font-size:12px;color:#6b7280;word-break:break-all;margin:0 0 10px}
	.iasm-pill{display:inline-flex;align-items:center;gap:6px;border-radius:999px;padding:4px 10px;font-size:12px;border:1px solid #e5e7eb;background:#f9fafb}
	.iasm-pill.ok{background:#ecfdf5;border-color:#a7f3d0;color:#065f46}
	.iasm-pill.warn{background:#fffbeb;border-color:#fde68a;color:#92400e}
	.iasm-pill.bad{background:#fef2f2;border-color:#fecaca;color:#991b1b}
	.iasm-row{display:flex;gap:10px;align-items:center;justify-content:space-between;flex-wrap:wrap}
	.iasm-small{font-size:12px;color:#6b7280;margin:8px 0 0}
	.iasm-score{margin-top:14px;background:#fff;border:1px solid #e5e7eb;border-radius:14px;padding:14px;box-shadow:0 1px 1px rgba(0,0,0,.03)}
	.iasm-bar{height:10px;border-radius:999px;background:#e5e7eb;overflow:hidden;margin-top:8px}
	.iasm-bar > span{display:block;height:10px;border-radius:999px;background:#2563eb;width:0%}
	.iasm-legend{margin-top:14px;background:#fff;border:1px solid #e5e7eb;border-radius:14px;padding:14px;box-shadow:0 1px 1px rgba(0,0,0,.03)}
	.iasm-legend ul{margin:10px 0 0;padding-left:18px;color:#374151}
	.iasm-legend li{margin:6px 0;line-height:1.4}
	.iasm-legend .iasm-pill{margin-right:8px}
		.iasm-pro{margin-top:14px;background:#fff;border:1px solid #e5e7eb;border-radius:14px;padding:14px;box-shadow:0 1px 1px rgba(0,0,0,.03)}
	.iasm-pro h2{margin:0 0 8px;font-size:16px}
	.iasm-pro ul{margin:0 0 12px;padding-left:18px;color:#374151}
	.iasm-pro a.button-primary{background:#2563eb;border-color:#1d4ed8}
	.iasm-footnote{margin-top:8px;color:#6b7280;font-size:12px}
	';

	wp_register_style( 'iasm-admin', false, array(), IASM_VERSION );
	wp_enqueue_style( 'iasm-admin' );
	wp_add_inline_style( 'iasm-admin', $css );
}
add_action( 'admin_enqueue_scripts', 'iasm_admin_assets' );

/**
 * Check file status (HEAD first, then GET).
 *
 * @param string $url
 * @return array {code:int|string, ok:bool, level:string, error:string}
 */
function iasm_check_url( $url, $expect = 'text' ) {
	$timeout = 5;

	$args = array(
		'timeout'     => $timeout,
		'redirection' => 3,
		'headers'     => array(
			'Cache-Control' => 'no-cache',
			'Pragma'        => 'no-cache',
		),
	);

	$response = wp_remote_head( $url, $args );
	$code     = is_wp_error( $response ) ? 0 : (int) wp_remote_retrieve_response_code( $response );
	$headers  = is_wp_error( $response ) ? array() : (array) wp_remote_retrieve_headers( $response );

	// Fallback to GET if HEAD is blocked or unreliable.
	if ( is_wp_error( $response ) || 0 === $code ) {
		$response = wp_remote_get( $url, $args + array( 'limit_response_size' => 12000 ) );
		$code     = is_wp_error( $response ) ? 0 : (int) wp_remote_retrieve_response_code( $response );
		$headers  = is_wp_error( $response ) ? array() : (array) wp_remote_retrieve_headers( $response );
	}

	if ( is_wp_error( $response ) ) {
		return array(
			'code'  => '-',
			'ok'    => false,
			'level' => 'bad',
			'error' => $response->get_error_message(),
		);
	}

	$body = '';
	if ( 200 === $code ) {
		$get = wp_remote_get( $url, $args + array( 'limit_response_size' => 12000 ) );
		if ( ! is_wp_error( $get ) ) {
			$body    = (string) wp_remote_retrieve_body( $get );
			$headers = (array) wp_remote_retrieve_headers( $get );
		}
	}

	// Basic soft-404 / HTML detection: 200 but HTML page or theme output.
	$content_type = '';
	if ( isset( $headers['content-type'] ) ) {
		$content_type = is_array( $headers['content-type'] ) ? (string) reset( $headers['content-type'] ) : (string) $headers['content-type'];
	}
	$body_lc = strtolower( substr( trim( $body ), 0, 300 ) );
	$is_html = ( false !== strpos( $content_type, 'text/html' ) ) || ( false !== strpos( $body_lc, '<!doctype html' ) ) || ( false !== strpos( $body_lc, '<html' ) );

	// Format validation (lightweight, Free-friendly).
	$valid_format = true;
	if ( 200 === $code && '' !== $body ) {
		if ( 'json' === $expect ) {
			$decoded      = json_decode( $body, true );
			$valid_format = is_array( $decoded ) && ! empty( $decoded );
		} elseif ( 'xml' === $expect ) {
			$valid_format = ( false !== strpos( $body, '<urlset' ) ) || ( false !== strpos( $body, '<sitemapindex' ) );
		} elseif ( 'llms' === $expect ) {
			// Heuristic: generated llms.txt typically contains "llms" and/or "IA SEO Generator".
			$valid_format = ( false !== stripos( $body, 'llms' ) ) || ( false !== stripos( $body, 'ia seo generator' ) );
		} elseif ( 'humans' === $expect ) {
			$valid_format = ( false !== stripos( $body, '/team/' ) ) || ( false !== stripos( $body, 'team:' ) ) || ( false !== stripos( $body, '/* team */' ) );
		}
	}

	// Scoring levels:
	// - OK: 200 + not HTML + valid format (when applicable)
	// - WARN: 200 but looks like HTML / invalid format OR 2xx/3xx not 200
	// - BAD: other codes
	if ( 200 === $code ) {
		if ( $is_html || ! $valid_format ) {
			return array( 'code' => $code, 'ok' => false, 'level' => 'warn', 'error' => '' );
		}
		return array( 'code' => $code, 'ok' => true, 'level' => 'ok', 'error' => '' );
	}
	if ( $code >= 200 && $code < 400 ) {
		return array( 'code' => $code, 'ok' => false, 'level' => 'warn', 'error' => '' );
	}

	return array( 'code' => $code, 'ok' => false, 'level' => 'bad', 'error' => '' );
}

/**
 * Render admin page.
 */
function iasm_render_admin_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	// Optional ecosystem hint: detect IA SEO Generator (Free or Premium). Purely informational.
	$iasg_detected = false;
	if ( is_admin() ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		$iasg_detected = ( is_plugin_active( 'ia-seo-generator/ia-seo-generator.php' ) || is_plugin_active( 'ia-seo-generator-premium/ia-seo-generator.php' ) );
	}

	$files = array(
		'/llms.txt'       => esc_html__( 'llms.txt', 'iasm-ai-search-visibility-monitor' ),
		'/humans.txt'     => esc_html__( 'humans.txt', 'iasm-ai-search-visibility-monitor' ),
		'/metadata.json'  => esc_html__( 'metadata.json', 'iasm-ai-search-visibility-monitor' ),
		'/sitemap-ia.xml' => esc_html__( 'sitemap-ia.xml', 'iasm-ai-search-visibility-monitor' ),
		'/ai-plugin.json' => esc_html__( 'ai-plugin.json', 'iasm-ai-search-visibility-monitor' ),
	);

	$results   = array();
	$ok_count  = 0;
	$base_url  = home_url( '/' );

	foreach ( $files as $path => $label ) {
		$url    = home_url( $path );
		$expect = 'text';
		if ( false !== strpos( $path, '.json' ) ) {
			$expect = 'json';
		} elseif ( false !== strpos( $path, '.xml' ) ) {
			$expect = 'xml';
		} elseif ( '/llms.txt' === $path ) {
			$expect = 'llms';
		} elseif ( '/humans.txt' === $path ) {
			$expect = 'humans';
		}
		$check  = iasm_check_url( $url, $expect );
		$code   = $check['code'];
		$level  = $check['level'];
		$error  = $check['error'];

		if ( true === $check['ok'] ) {
			$ok_count++;
		}

		$results[] = array(
			'label' => $label,
			'path'  => $path,
			'url'   => $url,
			'code'  => $code,
			'level' => $level,
			'error' => $error,
		);
	}

	$score = min( 100, max( 0, $ok_count * 20 ) );
	?>
	<div class="wrap iasm-wrap">
		<div class="iasm-hero">
			<h1><?php echo esc_html__( 'IASM - AI Search Visibility Monitor', 'iasm-ai-search-visibility-monitor' ); ?></h1>
			<p><?php echo esc_html__( 'Quickly check whether your AI-readable files exist and return a valid HTTP response.', 'iasm-ai-search-visibility-monitor' ); ?></p>

			<div class="iasm-badges">
				<span class="iasm-badge"><?php echo esc_html__( 'Mode: Free', 'iasm-ai-search-visibility-monitor' ); ?></span>
				<span class="iasm-badge"><?php echo esc_html__( 'Checks: Manual (this page)', 'iasm-ai-search-visibility-monitor' ); ?></span>
				<span class="iasm-badge"><?php echo esc_html__( 'Scope: Presence + HTTP status', 'iasm-ai-search-visibility-monitor' ); ?></span>
			</div>
		</div>

		<div class="iasm-score">
			<div class="iasm-row">
				<div>
					<strong><?php echo esc_html__( 'AI Readiness Snapshot', 'iasm-ai-search-visibility-monitor' ); ?></strong>
					<div class="iasm-small"><?php echo esc_html__( 'Score is based on files returning HTTP 200 (no content analysis in Free).', 'iasm-ai-search-visibility-monitor' ); ?></div>
					<?php if ( $iasg_detected ) : ?>
						<div class="iasm-small" style="margin-top:6px;">
							<?php echo esc_html__( 'IA SEO Generator detected on this site.', 'iasm-ai-search-visibility-monitor' ); ?>
						</div>
					<?php else : ?>
						<div class="iasm-small" style="margin-top:6px;">
							<?php echo esc_html__( 'IA SEO Generator not detected. This site may be using another method to generate AI-readable files.', 'iasm-ai-search-visibility-monitor' ); ?>
						</div>
					<?php endif; ?>
				</div>
				<div class="iasm-pill <?php echo ( $score >= 80 ? 'ok' : ( $score >= 40 ? 'warn' : 'bad' ) ); ?>">
					<?php
					/* translators: %d: readiness score value from 0 to 100 */
					echo esc_html( sprintf( esc_html__( 'Score: %d/100', 'iasm-ai-search-visibility-monitor' ), $score ) );
				?>
				</div>
			</div>
			<div class="iasm-bar" aria-hidden="true"><span style="width: <?php echo esc_attr( (string) $score ); ?>%"></span></div>
		</div>

		<div class="iasm-grid">
			<?php foreach ( $results as $r ) : ?>
				<div class="iasm-card">
					<h3>
						<span class="dashicons dashicons-media-text" aria-hidden="true"></span>
						<?php echo esc_html( $r['label'] ); ?>
					</h3>
					<p class="iasm-meta"><?php echo esc_html( $r['url'] ); ?></p>

					<div class="iasm-row">
						<?php
						$status_label = '';
						if ( 'ok' === $r['level'] ) {
							$status_label = esc_html__( 'OK (200)', 'iasm-ai-search-visibility-monitor' );
						} elseif ( 'warn' === $r['level'] ) {
							$status_label = esc_html__( 'Found (review recommended)', 'iasm-ai-search-visibility-monitor' );
						} else {
							$status_label = esc_html__( 'Missing or not reachable', 'iasm-ai-search-visibility-monitor' );
						}
						?>
						<span class="iasm-pill <?php echo esc_attr( $r['level'] ); ?>">
							<?php echo esc_html( $status_label ); ?>
							<?php if ( '-' !== $r['code'] ) : ?>
								<?php echo esc_html( ' • ' . $r['code'] ); ?>
							<?php endif; ?>
						</span>

						<a class="button" href="<?php echo esc_url( $r['url'] ); ?>" target="_blank" rel="noopener noreferrer">
							<?php echo esc_html__( 'Open file', 'iasm-ai-search-visibility-monitor' ); ?>
						</a>
					</div>

					<?php if ( ! empty( $r['error'] ) ) : ?>
						<p class="iasm-small"><?php echo esc_html( $r['error'] ); ?></p>
					<?php endif; ?>
				</div>
			<?php endforeach; ?>
		</div>


		<div class="iasm-legend">
			<strong><?php echo esc_html__( 'Status legend', 'iasm-ai-search-visibility-monitor' ); ?></strong>
			<ul>
				<li><span class="iasm-pill ok"><?php echo esc_html__( 'OK (200)', 'iasm-ai-search-visibility-monitor' ); ?></span> <?php echo esc_html__( 'File exists and returns a valid HTTP 200 response.', 'iasm-ai-search-visibility-monitor' ); ?></li>
				<li><span class="iasm-pill warn"><?php echo esc_html__( 'Found (review recommended)', 'iasm-ai-search-visibility-monitor' ); ?></span> <?php echo esc_html__( 'File returns HTTP 200, but content may not match the expected format (possible soft-404 or fallback).', 'iasm-ai-search-visibility-monitor' ); ?></li>
				<li><span class="iasm-pill bad"><?php echo esc_html__( 'Missing or not reachable', 'iasm-ai-search-visibility-monitor' ); ?></span> <?php echo esc_html__( 'File does not exist or returns an error response (404/5xx).', 'iasm-ai-search-visibility-monitor' ); ?></li>
			</ul>
		</div>

		<div class="iasm-pro">
			<h2><?php echo esc_html__( 'Unlock continuous monitoring (Pro)', 'iasm-ai-search-visibility-monitor' ); ?></h2>
			<ul>
				<li><?php echo esc_html__( 'Scheduled checks (daily/weekly)', 'iasm-ai-search-visibility-monitor' ); ?></li>
				<li><?php echo esc_html__( 'Alerts when a file breaks or disappears', 'iasm-ai-search-visibility-monitor' ); ?></li>
				<li><?php echo esc_html__( 'History and change tracking', 'iasm-ai-search-visibility-monitor' ); ?></li>
				<li><?php echo esc_html__( 'Advanced readiness scoring', 'iasm-ai-search-visibility-monitor' ); ?></li>
				<li><?php echo esc_html__( 'Exportable reports for clients', 'iasm-ai-search-visibility-monitor' ); ?></li>
			</ul>

			<a class="button button-primary" href="<?php echo esc_url( 'https://iaseogenerator.com/producto/iaseomonitor/?utm_source=wp-admin&utm_medium=plugin&utm_campaign=iasm-free' ); ?>" target="_blank" rel="noopener noreferrer">
				<?php echo esc_html__( 'Get Pro', 'iasm-ai-search-visibility-monitor' ); ?>
			</a>

			<div class="iasm-footnote">
				<?php echo esc_html__( 'Free version includes no ads and no tracking. Pro features are optional.', 'iasm-ai-search-visibility-monitor' ); ?>
			</div>
		</div>

		<p class="iasm-footnote">
			<?php
			/* translators: %s: plugin version number */
			echo esc_html( sprintf( esc_html__( 'Plugin version: %s', 'iasm-ai-search-visibility-monitor' ), IASM_VERSION ) );
		?>
			<?php echo esc_html__( '• Tip: If you use IA SEO Generator, generate the missing files first.', 'iasm-ai-search-visibility-monitor' ); ?>
		</p>
	</div>
	<?php
}
