<?php
/**
 * Helper Functions
 *
 * @package WPResourceMonitor
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Create database tables on activation
 */
function nandresmon_create_database_tables()
{
    global $wpdb;

    $table_name = $wpdb->prefix . NANDRESMON_LOGS_TABLE;
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE {$table_name} (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        source_type VARCHAR(20) NOT NULL DEFAULT 'request',
        source_name VARCHAR(255) NOT NULL DEFAULT '',
        memory_peak BIGINT(20) NOT NULL DEFAULT 0,
        memory_usage BIGINT(20) NOT NULL DEFAULT 0,
        execution_time FLOAT NOT NULL DEFAULT 0,
        query_count INT NOT NULL DEFAULT 0,
        query_time FLOAT NOT NULL DEFAULT 0,
        page_url VARCHAR(500) DEFAULT '',
        recorded_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY idx_source (source_type, source_name),
        KEY idx_recorded (recorded_at)
    ) {$charset_collate};";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
}

/**
 * Format bytes to human readable string
 *
 * @param int $bytes Number of bytes.
 * @param int $precision Decimal precision.
 * @return string Formatted string.
 */
function nandresmon_format_bytes($bytes, $precision = 2)
{
    $units = array('B', 'KB', 'MB', 'GB', 'TB');

    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);

    $bytes /= pow(1024, $pow);

    return round($bytes, $precision) . ' ' . $units[$pow];
}

/**
 * Format time in milliseconds to human readable
 *
 * @param float $seconds Time in seconds.
 * @return string Formatted time.
 */
function nandresmon_format_time($seconds)
{
    if ($seconds < 0.001) {
        return round($seconds * 1000000, 2) . ' µs';
    } elseif ($seconds < 1) {
        return round($seconds * 1000, 2) . ' ms';
    } else {
        return round($seconds, 2) . ' s';
    }
}

/**
 * Get PHP ini value with proper parsing
 *
 * @param string $setting PHP setting name.
 * @return int Value in bytes.
 */
function nandresmon_parse_ini_size($setting)
{
    $value = ini_get($setting);

    if (is_numeric($value)) {
        return (int) $value;
    }

    $suffix = strtoupper(substr($value, -1));
    $number = (int) substr($value, 0, -1);

    switch ($suffix) {
        case 'G':
            return $number * 1024 * 1024 * 1024;
        case 'M':
            return $number * 1024 * 1024;
        case 'K':
            return $number * 1024;
        default:
            return (int) $value;
    }
}

/**
 * Get status color class based on percentage
 *
 * @param float $percentage Usage percentage.
 * @return string CSS class name.
 */
function nandresmon_get_status_class($percentage)
{
    if ($percentage >= 90) {
        return 'status-danger';
    } elseif ($percentage >= 70) {
        return 'status-warning';
    } else {
        return 'status-success';
    }
}

/**
 * Get WordPress username for plugin author info
 *
 * @return string Current admin username.
 */
function nandresmon_get_admin_username()
{
    $current_user = wp_get_current_user();
    return $current_user->user_login;
}

/**
 * Enqueue toolbar assets for frontend
 */
function nandresmon_enqueue_toolbar_assets()
{
    $options = get_option(NANDRESMON_OPTIONS_KEY);

    if (empty($options['enable_toolbar']) || !is_admin_bar_showing()) {
        return;
    }

    wp_enqueue_style(
        'nandresmon-toolbar',
        NANDRESMON_PLUGIN_URL . 'assets/css/admin-styles.css',
        array(),
        NANDRESMON_VERSION
    );

    wp_enqueue_script(
        'nandresmon-toolbar',
        NANDRESMON_PLUGIN_URL . 'assets/js/toolbar-widget.js',
        array('jquery'),
        NANDRESMON_VERSION,
        true
    );
}

/**
 * Add toolbar node to admin bar
 *
 * @param WP_Admin_Bar $wp_admin_bar Admin bar object.
 */
function nandresmon_add_toolbar_node($wp_admin_bar)
{
    $options = get_option(NANDRESMON_OPTIONS_KEY);

    if (empty($options['enable_toolbar']) || !is_admin_bar_showing()) {
        return;
    }

    $memory_usage = memory_get_usage(true);
    $memory_peak = memory_get_peak_usage(true);
    $memory_limit = nandresmon_parse_ini_size('memory_limit');
    $percentage = ($memory_peak / $memory_limit) * 100;

    global $wpdb;
    $query_count = $wpdb->num_queries;

    $status_class = nandresmon_get_status_class($percentage);

    $wp_admin_bar->add_node(array(
        'id' => 'nandresmon-stats',
        'title' => sprintf(
            '<span class="nandresmon-toolbar-item %s">📊 %s / %d queries</span>',
            esc_attr($status_class),
            esc_html(nandresmon_format_bytes($memory_peak)),
            (int) $query_count
        ),
        'href' => admin_url('admin.php?page=hungry-resource-monitor'),
        'meta' => array(
            'class' => 'nandresmon-toolbar-wrapper',
            'title' => __('Hungry Resource Monitor', 'hungry-resource-monitor'),
        ),
    ));

    // Add submenu items
    $wp_admin_bar->add_node(array(
        'id' => 'nandresmon-memory',
        'parent' => 'nandresmon-stats',
        'title' => sprintf(
            /* translators: 1: current memory usage, 2: memory limit, 3: percentage used */
            __('Memory: %1$s / %2$s (%3$.1f%%)', 'hungry-resource-monitor'),
            nandresmon_format_bytes($memory_peak),
            nandresmon_format_bytes($memory_limit),
            $percentage
        ),
    ));

    $wp_admin_bar->add_node(array(
        'id' => 'nandresmon-queries',
        'parent' => 'nandresmon-stats',
        'title' => sprintf(
            /* translators: %d: number of database queries */
            __('DB Queries: %d', 'hungry-resource-monitor'),
            $query_count
        ),
    ));
}
