<?php
/**
 * Server Limits Class
 *
 * @package WPResourceMonitor
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class NANDRESMON_Server_Limits
 * Displays server configuration limits
 */
class NANDRESMON_Server_Limits
{

    /**
     * Get all server limits
     *
     * @return array Server limits data.
     */
    public static function get_all_limits()
    {
        global $wpdb;

        return array(
            'php' => self::get_php_limits(),
            'wordpress' => self::get_wordpress_limits(),
            'database' => self::get_database_info(),
            'server' => self::get_server_info(),
        );
    }

    /**
     * Get PHP limits
     *
     * @return array PHP limits.
     */
    public static function get_php_limits()
    {
        return array(
            array(
                'label' => __('PHP Version', 'hungry-resource-monitor'),
                'value' => phpversion(),
                'type' => 'text',
            ),
            array(
                'label' => __('Memory Limit', 'hungry-resource-monitor'),
                'value' => ini_get('memory_limit'),
                'current' => memory_get_peak_usage(true),
                'max' => nandresmon_parse_ini_size('memory_limit'),
                'type' => 'progress',
            ),
            array(
                'label' => __('Max Execution Time', 'hungry-resource-monitor'),
                'value' => ini_get('max_execution_time') . 's',
                'type' => 'text',
            ),
            array(
                'label' => __('Upload Max Filesize', 'hungry-resource-monitor'),
                'value' => ini_get('upload_max_filesize'),
                'type' => 'text',
            ),
            array(
                'label' => __('Post Max Size', 'hungry-resource-monitor'),
                'value' => ini_get('post_max_size'),
                'type' => 'text',
            ),
            array(
                'label' => __('Max Input Vars', 'hungry-resource-monitor'),
                'value' => ini_get('max_input_vars'),
                'type' => 'text',
            ),
        );
    }

    /**
     * Get WordPress limits
     *
     * @return array WordPress limits.
     */
    public static function get_wordpress_limits()
    {
        return array(
            array(
                'label' => __('WordPress Version', 'hungry-resource-monitor'),
                'value' => get_bloginfo('version'),
                'type' => 'text',
            ),
            array(
                'label' => __('WP Memory Limit', 'hungry-resource-monitor'),
                'value' => WP_MEMORY_LIMIT,
                'type' => 'text',
            ),
            array(
                'label' => __('WP Max Memory Limit', 'hungry-resource-monitor'),
                'value' => defined('WP_MAX_MEMORY_LIMIT') ? WP_MAX_MEMORY_LIMIT : 'Not set',
                'type' => 'text',
            ),
            array(
                'label' => __('Debug Mode', 'hungry-resource-monitor'),
                'value' => WP_DEBUG ? __('Enabled', 'hungry-resource-monitor') : __('Disabled', 'hungry-resource-monitor'),
                'type' => 'badge',
                'status' => WP_DEBUG ? 'warning' : 'success',
            ),
        );
    }

    /**
     * Get database info
     *
     * @return array Database info.
     */
    public static function get_database_info()
    {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for MySQL version; one-time read, no caching needed
        $mysql_version = $wpdb->get_var("SELECT VERSION()");

        return array(
            array(
                'label' => __('MySQL Version', 'hungry-resource-monitor'),
                'value' => $mysql_version,
                'type' => 'text',
            ),
            array(
                'label' => __('Database Prefix', 'hungry-resource-monitor'),
                'value' => $wpdb->prefix,
                'type' => 'text',
            ),
        );
    }

    /**
     * Get server info
     *
     * @return array Server info.
     */
    public static function get_server_info()
    {
        return array(
            array(
                'label' => __('Server Software', 'hungry-resource-monitor'),
                'value' => isset($_SERVER['SERVER_SOFTWARE']) ? sanitize_text_field(wp_unslash($_SERVER['SERVER_SOFTWARE'])) : 'Unknown',
                'type' => 'text',
            ),
            array(
                'label' => __('Operating System', 'hungry-resource-monitor'),
                'value' => PHP_OS,
                'type' => 'text',
            ),
        );
    }
}
