<?php
/**
 * Resource Tracker Class
 *
 * @package WPResourceMonitor
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class NANDRESMON_Resource_Tracker
 * Tracks memory and execution time at different WordPress lifecycle points
 */
class NANDRESMON_Resource_Tracker
{

    /**
     * Start time of request
     *
     * @var float
     */
    private static $start_time = 0;

    /**
     * Initialize tracking
     */
    public static function init()
    {
        self::$start_time = microtime(true);

        // Track on shutdown
        add_action('shutdown', array(__CLASS__, 'record_request_stats'), 9999);
    }

    /**
     * Record stats at end of request
     */
    public static function record_request_stats()
    {
        // Skip if start time not set
        if (empty(self::$start_time)) {
            return;
        }

        $options = get_option(NANDRESMON_OPTIONS_KEY);

        if (empty($options['enable_monitoring'])) {
            return;
        }

        // Skip AJAX and cron for now
        if (wp_doing_ajax() || wp_doing_cron()) {
            return;
        }

        global $wpdb;

        $data = array(
            'source_type' => 'request',
            'source_name' => self::get_current_page_type(),
            'memory_peak' => memory_get_peak_usage(true),
            'memory_usage' => memory_get_usage(true),
            'execution_time' => microtime(true) - self::$start_time,
            'query_count' => $wpdb->num_queries,
            'query_time' => 0,
            'page_url' => self::get_current_url(),
        );

        // Calculate query time if available
        if (defined('SAVEQUERIES') && SAVEQUERIES && !empty($wpdb->queries)) {
            foreach ($wpdb->queries as $query) {
                $data['query_time'] += $query[1];
            }
        }

        NANDRESMON_Database_Handler::insert_log($data);
    }

    /**
     * Get current page type
     *
     * @return string Page type identifier.
     */
    private static function get_current_page_type()
    {
        if (is_admin()) {
            return 'admin';
        } elseif (is_front_page()) {
            return 'front_page';
        } elseif (is_single()) {
            return 'single';
        } elseif (is_page()) {
            return 'page';
        } elseif (is_archive()) {
            return 'archive';
        } else {
            return 'other';
        }
    }

    /**
     * Get current URL
     *
     * @return string Current URL.
     */
    private static function get_current_url()
    {
        if (isset($_SERVER['REQUEST_URI'])) {
            return esc_url_raw(home_url(sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI']))));
        }
        return '';
    }
}

// Initialize tracker
add_action('plugins_loaded', array('NANDRESMON_Resource_Tracker', 'init'), 1);
