<?php
/**
 * Email Reports Class
 *
 * @package WPResourceMonitor
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class NANDRESMON_Email_Reports
 * Handles email report generation and sending
 */
class NANDRESMON_Email_Reports
{

    /**
     * Send email report
     *
     * @return bool Success status.
     */
    public static function send_report()
    {
        $options = get_option(NANDRESMON_OPTIONS_KEY);
        $recipient = !empty($options['email_recipient']) ? $options['email_recipient'] : get_option('admin_email');

        $subject = sprintf(
            /* translators: %s: site name */
            __('[%s] Hungry Resource Monitor Report', 'hungry-resource-monitor'),
            get_bloginfo('name')
        );

        $body = self::generate_report_html();

        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>',
        );

        return wp_mail($recipient, $subject, $body, $headers);
    }

    /**
     * Generate HTML report
     *
     * @return string HTML content.
     */
    public static function generate_report_html()
    {
        $stats = NANDRESMON_Database_Handler::get_overall_stats(7);
        $top_memory_pages = NANDRESMON_Database_Handler::get_top_consumers('page', 5, 7);
        $bloat = NANDRESMON_Cleanup_Handler::get_bloat_stats();
        $orphaned_tables = NANDRESMON_Cleanup_Handler::get_orphaned_tables();
        $inactive_plugins = NANDRESMON_Plugin_Profiler::get_inactive_plugins();
        $unused_themes = NANDRESMON_Plugin_Profiler::get_unused_themes();

        // Count orphaned crons
        $crons = _get_cron_array();
        $orphaned_cron_count = 0;
        $total_cron_count = 0;
        $wp_core_crons = array('wp_scheduled_delete', 'wp_scheduled_auto_draft_delete', 'wp_update_plugins', 'wp_update_themes', 'wp_version_check', 'delete_expired_transients', 'wp_privacy_delete_old_export_files', 'wp_site_health_scheduled_check', 'recovery_mode_clean_expired_keys', 'wp_https_detection', 'wp_update_user_counts');
        if (!empty($crons)) {
            foreach ($crons as $timestamp => $cron_hooks) {
                foreach ($cron_hooks as $hook => $cron_data) {
                    ++$total_cron_count;
                    if (!has_action($hook) && !in_array($hook, $wp_core_crons, true)) {
                        ++$orphaned_cron_count;
                    }
                }
            }
        }

        $total_bloat = array_sum($bloat);

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>

        <head>
            <style>
                body {
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                    line-height: 1.6;
                    color: #334155;
                    margin: 0;
                    padding: 0;
                    background: #f1f5f9;
                }

                .container {
                    max-width: 650px;
                    margin: 0 auto;
                    padding: 20px;
                }

                .header {
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white;
                    padding: 30px;
                    text-align: center;
                    border-radius: 12px 12px 0 0;
                }

                .header h1 {
                    margin: 0 0 5px 0;
                    font-size: 24px;
                }

                .header p {
                    margin: 0;
                    opacity: 0.9;
                    font-size: 14px;
                }

                .content {
                    background: #ffffff;
                    padding: 30px;
                    border-radius: 0 0 12px 12px;
                }

                .card {
                    background: #f8f9fa;
                    border-radius: 8px;
                    padding: 20px;
                    margin-bottom: 20px;
                    border: 1px solid #e2e8f0;
                }

                .card h3 {
                    margin: 0 0 15px 0;
                    color: #1e293b;
                    font-size: 16px;
                    border-bottom: 2px solid #667eea;
                    padding-bottom: 10px;
                }

                .stats-grid {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 15px;
                }

                .stat-box {
                    flex: 1;
                    min-width: 100px;
                    text-align: center;
                    background: white;
                    padding: 15px;
                    border-radius: 8px;
                    border: 1px solid #e2e8f0;
                }

                .stat-value {
                    font-size: 24px;
                    font-weight: 700;
                    color: #667eea;
                    display: block;
                }

                .stat-value.warning {
                    color: #f59e0b;
                }

                .stat-value.danger {
                    color: #ef4444;
                }

                .stat-value.success {
                    color: #10b981;
                }

                .stat-label {
                    font-size: 11px;
                    color: #64748b;
                    text-transform: uppercase;
                }

                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 10px;
                }

                th,
                td {
                    padding: 10px;
                    text-align: left;
                    border-bottom: 1px solid #e2e8f0;
                    font-size: 13px;
                }

                th {
                    background: #f1f5f9;
                    font-weight: 600;
                    color: #475569;
                }

                .alert {
                    padding: 15px;
                    border-radius: 8px;
                    margin-bottom: 15px;
                }

                .alert-warning {
                    background: #fef3c7;
                    border: 1px solid #f59e0b;
                    color: #92400e;
                }

                .alert-success {
                    background: #d1fae5;
                    border: 1px solid #10b981;
                    color: #065f46;
                }

                .footer {
                    text-align: center;
                    padding: 20px;
                    font-size: 12px;
                    color: #64748b;
                }

                .btn {
                    display: inline-block;
                    padding: 12px 24px;
                    background: #667eea;
                    color: white;
                    text-decoration: none;
                    border-radius: 6px;
                    margin-top: 15px;
                }
            </style>
        </head>

        <body>
            <div class="container">
                <div class="header">
                    <h1><?php esc_html_e('Hungry Resource Monitor', 'hungry-resource-monitor'); ?></h1>
                    <p><?php echo esc_html(get_bloginfo('name')); ?> -
                        <?php esc_html_e('Weekly Performance Report', 'hungry-resource-monitor'); ?>
                    </p>
                </div>
                <div class="content">

                    <!-- Summary Stats -->
                    <div class="card">
                        <h3>📊 <?php esc_html_e('Performance Summary (Last 7 Days)', 'hungry-resource-monitor'); ?></h3>
                        <div class="stats-grid">
                            <div class="stat-box">
                                <span
                                    class="stat-value"><?php echo esc_html(nandresmon_format_bytes($stats->avg_memory_peak ?? 0)); ?></span>
                                <span class="stat-label"><?php esc_html_e('Avg Memory', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span
                                    class="stat-value"><?php echo esc_html(nandresmon_format_bytes($stats->max_memory_peak ?? 0)); ?></span>
                                <span class="stat-label"><?php esc_html_e('Peak Memory', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span class="stat-value"><?php echo esc_html(round($stats->avg_query_count ?? 0)); ?></span>
                                <span class="stat-label"><?php esc_html_e('Avg Queries', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span class="stat-value"><?php echo esc_html($stats->total_samples ?? 0); ?></span>
                                <span class="stat-label"><?php esc_html_e('Samples', 'hungry-resource-monitor'); ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Cleanup Summary -->
                    <div class="card">
                        <h3>🧹 <?php esc_html_e('Cleanup Status', 'hungry-resource-monitor'); ?></h3>
                        <div class="stats-grid">
                            <div class="stat-box">
                                <span
                                    class="stat-value <?php echo $total_bloat > 100 ? 'warning' : 'success'; ?>"><?php echo esc_html(number_format($total_bloat)); ?></span>
                                <span
                                    class="stat-label"><?php esc_html_e('DB Bloat Items', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span
                                    class="stat-value <?php echo count($orphaned_tables) > 0 ? 'warning' : 'success'; ?>"><?php echo esc_html(count($orphaned_tables)); ?></span>
                                <span class="stat-label"><?php esc_html_e('Orphan Tables', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span
                                    class="stat-value <?php echo $orphaned_cron_count > 0 ? 'warning' : 'success'; ?>"><?php echo esc_html($orphaned_cron_count); ?></span>
                                <span class="stat-label"><?php esc_html_e('Orphan Crons', 'hungry-resource-monitor'); ?></span>
                            </div>
                        </div>
                        <div class="stats-grid" style="margin-top: 15px;">
                            <div class="stat-box">
                                <span
                                    class="stat-value <?php echo count($inactive_plugins) > 0 ? 'warning' : 'success'; ?>"><?php echo esc_html(count($inactive_plugins)); ?></span>
                                <span
                                    class="stat-label"><?php esc_html_e('Inactive Plugins', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span
                                    class="stat-value <?php echo count($unused_themes) > 0 ? 'warning' : 'success'; ?>"><?php echo esc_html(count($unused_themes)); ?></span>
                                <span class="stat-label"><?php esc_html_e('Unused Themes', 'hungry-resource-monitor'); ?></span>
                            </div>
                        </div>
                    </div>

                    <!-- Database Bloat Details -->
                    <div class="card">
                        <h3>🗄️ <?php esc_html_e('Database Bloat Details', 'hungry-resource-monitor'); ?></h3>
                        <table>
                            <tr>
                                <th><?php esc_html_e('Type', 'hungry-resource-monitor'); ?></th>
                                <th><?php esc_html_e('Count', 'hungry-resource-monitor'); ?></th>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Post Revisions', 'hungry-resource-monitor'); ?></td>
                                <td><?php echo esc_html($bloat['revisions']); ?></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Auto-Drafts', 'hungry-resource-monitor'); ?></td>
                                <td><?php echo esc_html($bloat['auto_drafts']); ?></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Trashed Posts', 'hungry-resource-monitor'); ?></td>
                                <td><?php echo esc_html($bloat['trashed_posts']); ?></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Spam Comments', 'hungry-resource-monitor'); ?></td>
                                <td><?php echo esc_html($bloat['spam_comments']); ?></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Trashed Comments', 'hungry-resource-monitor'); ?></td>
                                <td><?php echo esc_html($bloat['trashed_comments']); ?></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Expired Transients', 'hungry-resource-monitor'); ?></td>
                                <td><?php echo esc_html($bloat['expired_transients']); ?></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Orphaned Postmeta', 'hungry-resource-monitor'); ?></td>
                                <td><?php echo esc_html($bloat['orphaned_postmeta']); ?></td>
                            </tr>
                        </table>
                    </div>

                    <?php if (!empty($top_memory_pages)): ?>
                        <!-- Top Memory Pages -->
                        <div class="card">
                            <h3>📈 <?php esc_html_e('Top Memory Consuming Pages', 'hungry-resource-monitor'); ?></h3>
                            <table>
                                <tr>
                                    <th><?php esc_html_e('Page', 'hungry-resource-monitor'); ?></th>
                                    <th><?php esc_html_e('Avg Memory', 'hungry-resource-monitor'); ?></th>
                                </tr>
                                <?php foreach ($top_memory_pages as $page): ?>
                                    <tr>
                                        <td><?php echo esc_html(substr($page->source_name, 0, 50)); ?><?php echo strlen($page->source_name) > 50 ? '...' : ''; ?>
                                        </td>
                                        <td><?php echo esc_html(nandresmon_format_bytes($page->avg_memory)); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </table>
                        </div>
                    <?php endif; ?>

                    <!-- Server Info -->
                    <div class="card">
                        <h3>⚙️ <?php esc_html_e('Server Information', 'hungry-resource-monitor'); ?></h3>
                        <table>
                            <tr>
                                <td><?php esc_html_e('PHP Version', 'hungry-resource-monitor'); ?></td>
                                <td><strong><?php echo esc_html(phpversion()); ?></strong></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Memory Limit', 'hungry-resource-monitor'); ?></td>
                                <td><strong><?php echo esc_html(ini_get('memory_limit')); ?></strong></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Max Execution Time', 'hungry-resource-monitor'); ?></td>
                                <td><strong><?php echo esc_html(ini_get('max_execution_time')); ?>s</strong></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('WordPress Version', 'hungry-resource-monitor'); ?></td>
                                <td><strong><?php echo esc_html(get_bloginfo('version')); ?></strong></td>
                            </tr>
                        </table>
                    </div>

                    <!-- Cron Jobs -->
                    <div class="card">
                        <h3>⏰ <?php esc_html_e('Cron Jobs Status', 'hungry-resource-monitor'); ?></h3>
                        <div class="stats-grid">
                            <div class="stat-box">
                                <span class="stat-value"><?php echo esc_html($total_cron_count); ?></span>
                                <span class="stat-label"><?php esc_html_e('Total Crons', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span
                                    class="stat-value success"><?php echo esc_html($total_cron_count - $orphaned_cron_count); ?></span>
                                <span class="stat-label"><?php esc_html_e('Active Crons', 'hungry-resource-monitor'); ?></span>
                            </div>
                            <div class="stat-box">
                                <span
                                    class="stat-value <?php echo $orphaned_cron_count > 0 ? 'danger' : 'success'; ?>"><?php echo esc_html($orphaned_cron_count); ?></span>
                                <span
                                    class="stat-label"><?php esc_html_e('Orphaned Crons', 'hungry-resource-monitor'); ?></span>
                            </div>
                        </div>
                        <?php
                        // List orphaned crons if any
                        if ($orphaned_cron_count > 0):
                            $orphaned_cron_hooks = array();
                            foreach ($crons as $timestamp => $cron_hooks_inner) {
                                foreach ($cron_hooks_inner as $hook => $cron_data) {
                                    if (!has_action($hook) && !in_array($hook, $wp_core_crons, true) && !isset($orphaned_cron_hooks[$hook])) {
                                        $orphaned_cron_hooks[$hook] = true;
                                    }
                                }
                            }
                            ?>
                            <table style="margin-top: 15px;">
                                <tr>
                                    <th><?php esc_html_e('Orphaned Cron Hooks', 'hungry-resource-monitor'); ?></th>
                                </tr>
                                <?php foreach (array_slice(array_keys($orphaned_cron_hooks), 0, 10) as $hook): ?>
                                    <tr>
                                        <td><code
                                                style="background: #fee2e2; padding: 3px 8px; border-radius: 4px; font-size: 12px;"><?php echo esc_html($hook); ?></code>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (count($orphaned_cron_hooks) > 10): ?>
                                    <tr>
                                        <td><em><?php
                                        /* translators: %d: number of additional items */
                                        printf(esc_html__('...and %d more', 'hungry-resource-monitor'), count($orphaned_cron_hooks) - 10); ?></em>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </table>
                        <?php endif; ?>
                    </div>

                    <!-- Recommendations -->
                    <?php
                    $has_issues = ($total_bloat > 50) || (count($orphaned_tables) > 0) || ($orphaned_cron_count > 0) || (count($inactive_plugins) > 2);
                    if ($has_issues):
                        ?>
                        <div class="alert alert-warning">
                            <strong>💡 <?php esc_html_e('Recommendations:', 'hungry-resource-monitor'); ?></strong>
                            <ul style="margin: 10px 0 0 20px; padding: 0;">
                                <?php if ($total_bloat > 50): ?>
                                    <li><?php esc_html_e('Clean up database bloat to improve site performance.', 'hungry-resource-monitor'); ?>
                                    </li>
                                <?php endif; ?>
                                <?php if (count($orphaned_tables) > 0): ?>
                                    <li><?php
                                    /* translators: %d: number of orphaned tables */
                                    printf(esc_html__('Review %d orphaned database tables from deleted plugins.', 'hungry-resource-monitor'), count($orphaned_tables)); ?>
                                    </li>
                                <?php endif; ?>
                                <?php if ($orphaned_cron_count > 0): ?>
                                    <li><?php
                                    /* translators: %d: number of orphaned cron jobs */
                                    printf(esc_html__('Remove %d orphaned cron jobs.', 'hungry-resource-monitor'), intval($orphaned_cron_count)); ?>
                                    </li>
                                <?php endif; ?>
                                <?php if (count($inactive_plugins) > 2): ?>
                                    <li><?php
                                    /* translators: %d: number of inactive plugins */
                                    printf(esc_html__('Consider deleting %d inactive plugins for security.', 'hungry-resource-monitor'), count($inactive_plugins)); ?>
                                    </li>
                                <?php endif; ?>
                            </ul>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-success">
                            <strong>✅ <?php esc_html_e('Great job!', 'hungry-resource-monitor'); ?></strong>
                            <?php esc_html_e('Your site is well optimized. No immediate actions needed.', 'hungry-resource-monitor'); ?>
                        </div>
                    <?php endif; ?>

                    <!-- CTA Button -->
                    <div style="text-align: center;">
                        <a href="<?php echo esc_url(admin_url('admin.php?page=hungry-resource-monitor')); ?>" class="btn">
                            <?php esc_html_e('View Full Dashboard →', 'hungry-resource-monitor'); ?>
                        </a>
                    </div>

                </div>
                <div class="footer">
                    <p><?php
                    /* translators: %s: date and time */
                    printf(esc_html__('Generated by Hungry Resource Monitor on %s', 'hungry-resource-monitor'), esc_html(wp_date('F j, Y \a\t g:i a'))); ?>
                    </p>
                    <p style="font-size: 11px; color: #94a3b8;">
                        <?php esc_html_e('You are receiving this email because email reports are enabled in Hungry Resource Monitor settings.', 'hungry-resource-monitor'); ?>
                    </p>
                </div>
            </div>
        </body>

        </html>
        <?php
        return ob_get_clean();
    }
}

/**
 * Global function to send email report
 */
function nandresmon_send_email_report()
{
    return NANDRESMON_Email_Reports::send_report();
}
