<?php
/**
 * Database Handler Class
 *
 * @package WPResourceMonitor
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class NANDRESMON_Database_Handler
 * Handles all database operations with proper prepared statements
 */
class NANDRESMON_Database_Handler
{

    /**
     * Get table name with prefix
     *
     * @return string Table name.
     */
    public static function get_table_name()
    {
        global $wpdb;
        return $wpdb->prefix . NANDRESMON_LOGS_TABLE;
    }

    /**
     * Insert a log entry
     *
     * @param array $data Log data to insert.
     * @return int|false Insert ID or false on failure.
     */
    public static function insert_log($data)
    {
        global $wpdb;

        $table_name = self::get_table_name();

        // Check if table exists, create if not
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query to check table existence
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name));
        if (!$table_exists) {
            nandresmon_create_database_tables();
        }

        $defaults = array(
            'source_type' => 'request',
            'source_name' => '',
            'memory_peak' => 0,
            'memory_usage' => 0,
            'execution_time' => 0,
            'query_count' => 0,
            'query_time' => 0,
            'page_url' => '',
            'recorded_at' => current_time('mysql', true),
        );

        $data = wp_parse_args($data, $defaults);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct insert required for efficient logging; caching not needed for write operation
        $result = $wpdb->insert(
            self::get_table_name(),
            array(
                'source_type' => sanitize_text_field($data['source_type']),
                'source_name' => sanitize_text_field($data['source_name']),
                'memory_peak' => absint($data['memory_peak']),
                'memory_usage' => absint($data['memory_usage']),
                'execution_time' => floatval($data['execution_time']),
                'query_count' => absint($data['query_count']),
                'query_time' => floatval($data['query_time']),
                'page_url' => esc_url_raw($data['page_url']),
                'recorded_at' => sanitize_text_field($data['recorded_at']),
            ),
            array('%s', '%s', '%d', '%d', '%f', '%d', '%f', '%s', '%s')
        );

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Get top resource consumers by source type
     *
     * @param string $source_type Type of source (plugin, theme, etc).
     * @param int    $limit Number of results.
     * @param int    $days Number of days to look back.
     * @return array Results.
     */
    public static function get_top_consumers($source_type = 'plugin', $limit = 5, $days = 7)
    {
        global $wpdb;

        $table_name = self::get_table_name();
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for aggregated statistics; caching would prevent real-time data
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT source_name, 
                        AVG(memory_peak) as avg_memory,
                        AVG(execution_time) as avg_time,
                        COUNT(*) as sample_count
                 FROM %i 
                 WHERE source_type = %s 
                   AND recorded_at > %s
                   AND source_name != ''
                 GROUP BY source_name 
                 ORDER BY avg_memory DESC 
                 LIMIT %d",
                $table_name,
                $source_type,
                $cutoff_date,
                $limit
            )
        );

        return $results ? $results : array();
    }

    /**
     * Get top pages by memory usage (grouped by URL)
     *
     * @param int $limit Number of results.
     * @param int $days Number of days to look back.
     * @return array Results with page URLs.
     */
    public static function get_top_pages_by_url($limit = 10, $days = 7)
    {
        global $wpdb;

        $table_name = self::get_table_name();
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for aggregated statistics; caching would prevent real-time data
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT page_url, 
                        AVG(memory_peak) as avg_memory,
                        MAX(memory_peak) as max_memory,
                        AVG(query_count) as avg_queries,
                        COUNT(*) as sample_count
                 FROM %i 
                 WHERE recorded_at > %s
                   AND page_url != ''
                   AND page_url NOT LIKE %s
                 GROUP BY page_url 
                 ORDER BY avg_memory DESC 
                 LIMIT %d",
                $table_name,
                $cutoff_date,
                '%/wp-admin%',
                $limit
            )
        );

        return $results ? $results : array();
    }

    /**
     * Get overall statistics
     *
     * @param int $days Number of days to look back.
     * @return object Statistics object.
     */
    public static function get_overall_stats($days = 7)
    {
        global $wpdb;

        $table_name = self::get_table_name();
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for aggregated statistics; caching would prevent real-time data
        $stats = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT 
                    AVG(memory_peak) as avg_memory_peak,
                    MAX(memory_peak) as max_memory_peak,
                    AVG(execution_time) as avg_execution_time,
                    AVG(query_count) as avg_query_count,
                    AVG(query_time) as avg_query_time,
                    COUNT(*) as total_samples
                 FROM %i 
                 WHERE recorded_at > %s",
                $table_name,
                $cutoff_date
            )
        );

        return $stats;
    }

    /**
     * Get memory usage over time for charts
     *
     * @param int $days Number of days.
     * @return array Chart data.
     */
    public static function get_memory_chart_data($days = 7)
    {
        global $wpdb;

        $table_name = self::get_table_name();
        $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for chart data aggregation; caching would prevent real-time updates
        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT 
                    DATE(recorded_at) as date,
                    AVG(memory_peak) as avg_memory,
                    MAX(memory_peak) as max_memory,
                    AVG(query_count) as avg_queries
                 FROM %i 
                 WHERE recorded_at > %s
                 GROUP BY DATE(recorded_at)
                 ORDER BY date ASC",
                $table_name,
                $cutoff_date
            )
        );

        return $results ? $results : array();
    }

    /**
     * Clear all logs
     *
     * @return int|false Number of rows deleted or false on error.
     */
    public static function clear_all_logs()
    {
        global $wpdb;

        $table_name = self::get_table_name();

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for table truncation; no caching needed
        return $wpdb->query(
            $wpdb->prepare("TRUNCATE TABLE %i", $table_name)
        );
    }
}
