<?php
/**
 * Cleanup Handler Class
 *
 * @package WPResourceMonitor
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class NANDRESMON_Cleanup_Handler
 * Handles bloat detection and cleanup operations
 */
class NANDRESMON_Cleanup_Handler
{

    /**
     * Get all bloat statistics
     *
     * @return array Bloat data.
     */
    public static function get_bloat_stats()
    {
        return array(
            'revisions' => self::count_revisions(),
            'auto_drafts' => self::count_auto_drafts(),
            'trashed_posts' => self::count_trashed_posts(),
            'spam_comments' => self::count_spam_comments(),
            'trashed_comments' => self::count_trashed_comments(),
            'expired_transients' => self::count_expired_transients(),
            'orphaned_postmeta' => self::count_orphaned_postmeta(),
        );
    }

    /**
     * Count post revisions
     *
     * @return int Count.
     */
    public static function count_revisions()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct count for bloat detection
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'revision'");
    }

    /**
     * Count auto-drafts
     *
     * @return int Count.
     */
    public static function count_auto_drafts()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct count for bloat detection
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'auto-draft'");
    }

    /**
     * Count trashed posts
     *
     * @return int Count.
     */
    public static function count_trashed_posts()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct count for bloat detection
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'trash'");
    }

    /**
     * Count spam comments
     *
     * @return int Count.
     */
    public static function count_spam_comments()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct count for bloat detection
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = 'spam'");
    }

    /**
     * Count trashed comments
     *
     * @return int Count.
     */
    public static function count_trashed_comments()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct count for bloat detection
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved = 'trash'");
    }

    /**
     * Count expired transients
     *
     * @return int Count.
     */
    public static function count_expired_transients()
    {
        global $wpdb;
        $time = time();
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct count for bloat detection
        return (int) $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->options} 
                 WHERE option_name LIKE %s 
                 AND option_value < %d",
                '_transient_timeout_%',
                $time
            )
        );
    }

    /**
     * Count orphaned postmeta
     *
     * @return int Count.
     */
    public static function count_orphaned_postmeta()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct count for bloat detection
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} pm 
             LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
             WHERE p.ID IS NULL"
        );
    }

    /**
     * Delete revisions
     *
     * @return int Deleted count.
     */
    public static function delete_revisions()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct delete for cleanup
        return $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_type = 'revision'");
    }

    /**
     * Delete auto-drafts
     *
     * @return int Deleted count.
     */
    public static function delete_auto_drafts()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct delete for cleanup
        return $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_status = 'auto-draft'");
    }

    /**
     * Delete trashed posts
     *
     * @return int Deleted count.
     */
    public static function delete_trashed_posts()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct delete for cleanup
        return $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_status = 'trash'");
    }

    /**
     * Delete spam comments
     *
     * @return int Deleted count.
     */
    public static function delete_spam_comments()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct delete for cleanup
        return $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_approved = 'spam'");
    }

    /**
     * Delete trashed comments
     *
     * @return int Deleted count.
     */
    public static function delete_trashed_comments()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct delete for cleanup
        return $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_approved = 'trash'");
    }

    /**
     * Delete expired transients
     *
     * @return int Deleted count.
     */
    public static function delete_expired_transients()
    {
        global $wpdb;
        $time = time();

        // Get expired transient names
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query for cleanup
        $expired = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} 
                 WHERE option_name LIKE %s AND option_value < %d",
                '_transient_timeout_%',
                $time
            )
        );

        $count = 0;
        foreach ($expired as $transient) {
            $name = str_replace('_transient_timeout_', '', $transient);
            if (delete_transient($name)) {
                ++$count;
            }
        }

        return $count;
    }

    /**
     * Delete orphaned postmeta
     *
     * @return int Deleted count.
     */
    public static function delete_orphaned_postmeta()
    {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct delete for cleanup
        return $wpdb->query(
            "DELETE pm FROM {$wpdb->postmeta} pm 
             LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID 
             WHERE p.ID IS NULL"
        );
    }

    /**
     * Scan a plugin directory for table names it uses
     *
     * @param string $plugin_dir Plugin directory path.
     * @return array List of table names (without wp prefix).
     */
    public static function scan_plugin_for_tables($plugin_dir)
    {
        $tables = array();

        if (!is_dir($plugin_dir)) {
            return $tables;
        }

        // Scan PHP files for table patterns
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($plugin_dir, RecursiveDirectoryIterator::SKIP_DOTS)
        );

        foreach ($iterator as $file) {
            if ($file->getExtension() !== 'php') {
                continue;
            }

            // Read file contents
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Filesystem operation only
            $content = file_get_contents($file->getPathname());
            if ($content === false) {
                continue;
            }

            // Pattern 1: $wpdb->prefix . 'tablename' or "tablename"
            if (preg_match_all('/\$wpdb->prefix\s*\.\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 2: {$wpdb->prefix}tablename in strings
            if (preg_match_all('/\{\$wpdb->prefix\}([a-zA-Z0-9_]+)/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 3: CREATE TABLE ... prefix_tablename or `prefix_tablename`
            if (preg_match_all('/CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?(?:`)?(?:\{?\$\w+->prefix\}?)?([a-zA-Z0-9_]+)(?:`)?/i', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 4: networkTable('tablename') - used by Wordfence
            if (preg_match_all('/networkTable\s*\(\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 5: table_name constants/variables like TABLE_NAME = 'yoast_indexable'
            if (preg_match_all('/(?:TABLE_NAME|table_name|tableName)\s*[=:]\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 6: $table = $prefix . 'tablename'
            if (preg_match_all('/\$(?:table|table_name)\s*=\s*\$\w+\s*\.\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 7: 'table' => 'tablename' in arrays
            if (preg_match_all('/[\'"]table[\'"]\s*=>\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 8: define() with TABLE in constant name - e.g. define('XXX_TABLE_NAME', 'tablename')
            if (preg_match_all('/define\s*\(\s*[\'"][A-Z0-9_]*TABLE[A-Z0-9_]*[\'"]\s*,\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 9: const with TABLE in name = 'tablename'
            if (preg_match_all('/const\s+[A-Z0-9_]*TABLE[A-Z0-9_]*\s*=\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 10: private/protected/public $table_name = 'tablename'
            if (preg_match_all('/(?:private|protected|public)\s+\$\w*table\w*\s*=\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }

            // Pattern 11: define() with LOG or DATA in constant name that might be tables
            if (preg_match_all('/define\s*\(\s*[\'"][A-Z0-9_]*(?:LOG|LOGS|DATA|STATS|METRICS|ALERTS)[A-Z0-9_]*[\'"]\s*,\s*[\'"]([a-zA-Z0-9_]+)[\'"]/', $content, $matches)) {
                foreach ($matches[1] as $table_name) {
                    $tables[strtolower($table_name)] = true;
                }
            }
        }

        return array_keys($tables);
    }

    /**
     * Get all database tables with their owners
     *
     * @return array List of tables with owner info.
     */
    public static function get_all_tables_with_owners()
    {
        global $wpdb;

        // Get all tables with wp prefix
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query to check database tables
        $all_tables = $wpdb->get_col(
            $wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $wpdb->prefix . '%'
            )
        );

        // Core WordPress tables
        $core_table_names = array(
            'commentmeta',
            'comments',
            'links',
            'options',
            'postmeta',
            'posts',
            'term_relationships',
            'term_taxonomy',
            'termmeta',
            'terms',
            'usermeta',
            'users'
        );

        // Get ALL installed plugins
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $all_plugins = get_plugins();
        $active_plugins = get_option('active_plugins', array());

        // Build table -> plugin mapping by scanning plugin files
        $table_to_plugin = array();
        $plugins_dir = WP_PLUGIN_DIR;

        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $parts = explode('/', $plugin_file);
            $folder = $parts[0];
            $plugin_dir = $plugins_dir . '/' . $folder;
            $is_active = in_array($plugin_file, $active_plugins, true);

            // Scan plugin for tables it uses
            $plugin_tables = self::scan_plugin_for_tables($plugin_dir);

            foreach ($plugin_tables as $table_name) {
                // Only update if not set or if this plugin is active
                if (!isset($table_to_plugin[$table_name]) || $is_active) {
                    $table_to_plugin[$table_name] = array(
                        'name' => $plugin_data['Name'],
                        'active' => $is_active,
                    );
                }
            }
        }

        // Also build prefix mapping from plugin folder names as fallback
        $plugin_prefixes = array();
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $parts = explode('/', $plugin_file);
            $folder = strtolower($parts[0]);
            $is_active = in_array($plugin_file, $active_plugins, true);

            // Add folder name and variations as prefixes
            $prefixes = array(
                $folder,
                str_replace('-', '_', $folder),
                str_replace('-', '', $folder),
                str_replace('_', '', $folder),
            );

            foreach ($prefixes as $prefix) {
                if (!isset($plugin_prefixes[$prefix]) || $is_active) {
                    $plugin_prefixes[$prefix] = array(
                        'name' => $plugin_data['Name'],
                        'active' => $is_active,
                    );
                }
            }

            // Also add prefixes from tables this plugin defines (first part before _)
            if (isset($table_to_plugin)) {
                foreach ($table_to_plugin as $table_name => $plugin_info) {
                    $table_prefix = explode('_', $table_name)[0];
                    if (strlen($table_prefix) >= 2 && !isset($plugin_prefixes[$table_prefix])) {
                        $plugin_prefixes[$table_prefix] = $plugin_info;
                    }
                }
            }
        }

        $tables = array();

        foreach ($all_tables as $table) {
            // Get table name without wp prefix
            $table_without_prefix = strtolower(substr($table, strlen($wpdb->prefix)));

            // Get table size
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query for table info
            $size_result = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT 
                        ROUND((DATA_LENGTH + INDEX_LENGTH) / 1024, 2) as size_kb,
                        TABLE_ROWS as row_count
                     FROM information_schema.TABLES 
                     WHERE TABLE_SCHEMA = %s AND TABLE_NAME = %s",
                    DB_NAME,
                    $table
                )
            );

            $table_info = array(
                'name' => $table,
                'size' => $size_result ? $size_result->size_kb : 0,
                'rows' => $size_result ? $size_result->row_count : 0,
                'owner' => 'Unknown',
                'owner_type' => 'unknown',
                'is_active' => false,
            );

            // Check if core table
            if (in_array($table_without_prefix, $core_table_names, true)) {
                $table_info['owner'] = 'WordPress Core';
                $table_info['owner_type'] = 'core';
                $table_info['is_active'] = true;
            } elseif (isset($table_to_plugin[$table_without_prefix])) {
                // Found in plugin scan (exact match)
                $plugin_info = $table_to_plugin[$table_without_prefix];
                $table_info['owner'] = $plugin_info['name'];
                $table_info['owner_type'] = 'plugin';
                $table_info['is_active'] = $plugin_info['active'];
            } else {
                // Fallback: try prefix-based matching
                foreach ($plugin_prefixes as $prefix => $plugin_info) {
                    if (strlen($prefix) >= 3 && strpos($table_without_prefix, $prefix) === 0) {
                        $table_info['owner'] = $plugin_info['name'];
                        $table_info['owner_type'] = 'plugin';
                        $table_info['is_active'] = $plugin_info['active'];
                        break;
                    }
                }
            }

            $tables[] = $table_info;
        }

        // Sort: Core first, then active plugins, then inactive, then unknown
        usort($tables, function ($a, $b) {
            $order = array('core' => 0, 'plugin' => 1, 'unknown' => 2);
            $type_diff = $order[$a['owner_type']] - $order[$b['owner_type']];
            if ($type_diff !== 0)
                return $type_diff;
            if ($a['is_active'] !== $b['is_active'])
                return $b['is_active'] - $a['is_active'];
            return strcmp($a['name'], $b['name']);
        });

        return $tables;
    }

    /**
     * Get only orphaned tables (Unknown owner)
     *
     * @return array List of orphaned tables.
     */
    public static function get_orphaned_tables()
    {
        $all_tables = self::get_all_tables_with_owners();
        return array_filter($all_tables, function ($table) {
            return $table['owner_type'] === 'unknown';
        });
    }

    /**
     * Delete a specific table
     *
     * @param string $table_name Table name to delete.
     * @return bool Success.
     */
    public static function delete_table($table_name)
    {
        global $wpdb;

        // Only allow deleting tables with our prefix
        if (strpos($table_name, $wpdb->prefix) !== 0) {
            return false;
        }

        // Never delete core tables
        $core_tables = array('posts', 'postmeta', 'options', 'users', 'usermeta', 'terms', 'termmeta', 'term_taxonomy', 'term_relationships', 'comments', 'commentmeta', 'links');
        foreach ($core_tables as $core) {
            if ($table_name === $wpdb->prefix . $core) {
                return false;
            }
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Direct DROP required for table cleanup
        $result = $wpdb->query($wpdb->prepare("DROP TABLE IF EXISTS %i", $table_name));

        return $result !== false;
    }
}

