<?php
/**
 * AJAX Handlers
 *
 * @package WPResourceMonitor
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handle cleanup AJAX action
 */
function nandresmon_ajax_cleanup_action()
{
    check_ajax_referer('nandresmon_admin_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized', 'hungry-resource-monitor')));
    }

    $action = isset($_POST['cleanup_action']) ? sanitize_text_field(wp_unslash($_POST['cleanup_action'])) : '';

    $actions_map = array(
        'revisions' => 'delete_revisions',
        'auto_drafts' => 'delete_auto_drafts',
        'trashed_posts' => 'delete_trashed_posts',
        'spam_comments' => 'delete_spam_comments',
        'trashed_comments' => 'delete_trashed_comments',
        'expired_transients' => 'delete_expired_transients',
        'orphaned_postmeta' => 'delete_orphaned_postmeta',
    );

    if (!isset($actions_map[$action])) {
        wp_send_json_error(array('message' => __('Invalid action', 'hungry-resource-monitor')));
    }

    $method = $actions_map[$action];
    $deleted = NANDRESMON_Cleanup_Handler::$method();

    wp_send_json_success(array(
        'message' => sprintf(
            /* translators: %d: number of items deleted */
            __('%d items cleaned up.', 'hungry-resource-monitor'),
            $deleted
        ),
        'deleted' => $deleted,
    ));
}

/**
 * Handle contact form AJAX
 */
function nandresmon_ajax_send_contact()
{
    check_ajax_referer('nandresmon_contact_nonce', 'nandresmon_contact_nonce');

    $name = isset($_POST['contact_name']) ? sanitize_text_field(wp_unslash($_POST['contact_name'])) : '';
    $email = isset($_POST['contact_email']) ? sanitize_email(wp_unslash($_POST['contact_email'])) : '';
    $message = isset($_POST['contact_message']) ? sanitize_textarea_field(wp_unslash($_POST['contact_message'])) : '';

    if (empty($name) || empty($email) || empty($message)) {
        wp_send_json_error(array('message' => __('All fields are required.', 'hungry-resource-monitor')));
    }

    $to = 'prakhar@nandann.com';
    $subject = sprintf('[Hungry Resource Monitor] Message from %s', $name);
    $body = sprintf("Name: %s\nEmail: %s\nSite: %s\n\nMessage:\n%s", $name, $email, home_url(), $message);
    $headers = array('Reply-To: ' . $name . ' <' . $email . '>');

    $sent = wp_mail($to, $subject, $body, $headers);

    if ($sent) {
        wp_send_json_success(array('message' => __('Message sent successfully!', 'hungry-resource-monitor')));
    } else {
        wp_send_json_error(array('message' => __('Failed to send message.', 'hungry-resource-monitor')));
    }
}

/**
 * Handle test email AJAX
 */
function nandresmon_ajax_send_test_email()
{
    check_ajax_referer('nandresmon_admin_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized', 'hungry-resource-monitor')));
    }

    $sent = NANDRESMON_Email_Reports::send_report();

    if ($sent) {
        wp_send_json_success(array('message' => __('Test email sent!', 'hungry-resource-monitor')));
    } else {
        wp_send_json_error(array('message' => __('Failed to send email.', 'hungry-resource-monitor')));
    }
}

/**
 * Handle single cron delete AJAX
 */
function nandresmon_ajax_delete_cron()
{
    check_ajax_referer('nandresmon_admin_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized', 'hungry-resource-monitor')));
    }

    $hook = isset($_POST['hook']) ? sanitize_text_field(wp_unslash($_POST['hook'])) : '';
    $timestamp = isset($_POST['timestamp']) ? absint($_POST['timestamp']) : 0;

    if (empty($hook) || empty($timestamp)) {
        wp_send_json_error(array('message' => __('Invalid cron data', 'hungry-resource-monitor')));
    }

    $result = wp_unschedule_event($timestamp, $hook);

    if ($result) {
        wp_send_json_success(array('message' => __('Cron removed successfully.', 'hungry-resource-monitor')));
    } else {
        wp_send_json_error(array('message' => __('Failed to remove cron.', 'hungry-resource-monitor')));
    }
}

/**
 * Handle clear all orphaned crons AJAX
 */
function nandresmon_ajax_clear_orphaned_crons()
{
    check_ajax_referer('nandresmon_admin_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized', 'hungry-resource-monitor')));
    }

    $crons = _get_cron_array();
    $wp_core_crons = array(
        'wp_scheduled_delete',
        'wp_scheduled_auto_draft_delete',
        'wp_update_plugins',
        'wp_update_themes',
        'wp_version_check',
        'delete_expired_transients',
        'wp_privacy_delete_old_export_files',
        'wp_site_health_scheduled_check',
        'recovery_mode_clean_expired_keys',
        'wp_https_detection',
        'wp_update_user_counts',
    );

    $removed = 0;

    if (!empty($crons)) {
        foreach ($crons as $timestamp => $cron_hooks) {
            foreach ($cron_hooks as $hook => $cron_data) {
                $is_orphaned = !has_action($hook);
                $is_core = in_array($hook, $wp_core_crons, true);

                if ($is_orphaned && !$is_core) {
                    wp_unschedule_event($timestamp, $hook);
                    ++$removed;
                }
            }
        }
    }

    wp_send_json_success(array(
        'message' => sprintf(
            /* translators: %d: number of crons removed */
            __('%d orphaned crons removed.', 'hungry-resource-monitor'),
            $removed
        ),
        'removed' => $removed,
    ));
}

/**
 * Handle delete table AJAX
 */
function nandresmon_ajax_delete_table()
{
    check_ajax_referer('nandresmon_admin_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => __('Unauthorized', 'hungry-resource-monitor')));
    }

    $table_name = isset($_POST['table_name']) ? sanitize_text_field(wp_unslash($_POST['table_name'])) : '';

    if (empty($table_name)) {
        wp_send_json_error(array('message' => __('Invalid table name', 'hungry-resource-monitor')));
    }

    $result = NANDRESMON_Cleanup_Handler::delete_table($table_name);

    if ($result) {
        wp_send_json_success(array('message' => __('Table deleted successfully.', 'hungry-resource-monitor')));
    } else {
        wp_send_json_error(array('message' => __('Failed to delete table or table is protected.', 'hungry-resource-monitor')));
    }
}
