<?php
/**
 * Plugin Name: Hungry Resource Monitor
 * Plugin URI: https://nandann.com/contact
 * Description: Monitor memory, CPU, and resource usage by plugins, themes, and scripts. Includes bloat detection, cleanup tools, and weekly reports.
 * Version: 1.0.1
 * Author: Prakhar Bhatia
 * Author URI: https://nandann.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: hungry-resource-monitor
 * Domain Path: /languages
 * Requires at least: 6.2
 * Tested up to: 6.9
 * Requires PHP: 7.4
 *
 * @package HungryResourceMonitor
 * @author Prakhar Bhatia <prakhar@nandann.com>
 * @copyright 2026 Nandann Creative Agency
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants with unique 10-char prefix: nandresmon
define('NANDRESMON_VERSION', '1.0.1');
define('NANDRESMON_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('NANDRESMON_PLUGIN_URL', plugin_dir_url(__FILE__));
define('NANDRESMON_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('NANDRESMON_OPTIONS_KEY', 'nandresmon_options');
define('NANDRESMON_LOGS_TABLE', 'nandresmon_logs');

// Include required files
require_once NANDRESMON_PLUGIN_DIR . 'includes/functions.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-database-handler.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-resource-tracker.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-plugin-profiler.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-server-limits.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-cleanup-handler.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-email-reports.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-toolbar-widget.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-support-page.php';
require_once NANDRESMON_PLUGIN_DIR . 'includes/class-admin-page.php';

// Activation and deactivation hooks
register_activation_hook(__FILE__, 'nandresmon_activate_plugin');
register_deactivation_hook(__FILE__, 'nandresmon_deactivate_plugin');

/**
 * Plugin activation
 */
function nandresmon_activate_plugin()
{
    nandresmon_create_database_tables();
    nandresmon_initialize_options();
    nandresmon_schedule_events();
    flush_rewrite_rules();
}

/**
 * Plugin deactivation
 */
function nandresmon_deactivate_plugin()
{
    nandresmon_clear_scheduled_events();
    flush_rewrite_rules();
}

/**
 * Initialize plugin
 */
function nandresmon_init()
{
    // Initialize admin
    if (is_admin()) {
        nandresmon_init_admin();
    }

    // Initialize frontend toolbar for admins
    if (current_user_can('manage_options')) {
        add_action('wp_enqueue_scripts', 'nandresmon_enqueue_toolbar_assets');
        add_action('admin_bar_menu', 'nandresmon_add_toolbar_node', 100);
    }
}

add_action('init', 'nandresmon_init');

/**
 * Initialize admin functionality
 */
function nandresmon_init_admin()
{
    // Add admin menu
    add_action('admin_menu', 'nandresmon_add_admin_menu');

    // Enqueue admin scripts and styles
    add_action('admin_enqueue_scripts', 'nandresmon_admin_enqueue_scripts');

    // Register AJAX handlers
    add_action('wp_ajax_nandresmon_cleanup_action', 'nandresmon_ajax_cleanup_action');
    add_action('wp_ajax_nandresmon_send_contact', 'nandresmon_ajax_send_contact');
    add_action('wp_ajax_nandresmon_send_test_email', 'nandresmon_ajax_send_test_email');
    add_action('wp_ajax_nandresmon_delete_cron', 'nandresmon_ajax_delete_cron');
    add_action('wp_ajax_nandresmon_clear_orphaned_crons', 'nandresmon_ajax_clear_orphaned_crons');
    add_action('wp_ajax_nandresmon_delete_table', 'nandresmon_ajax_delete_table');
}

/**
 * Add admin menu
 */
function nandresmon_add_admin_menu()
{
    add_menu_page(
        __('Resource Monitor', 'hungry-resource-monitor'),
        __('Resource Monitor', 'hungry-resource-monitor'),
        'manage_options',
        'hungry-resource-monitor',
        'nandresmon_admin_page',
        'dashicons-chart-area',
        75
    );
}

/**
 * Enqueue admin scripts and styles
 *
 * @param string $hook Current admin page hook.
 */
function nandresmon_admin_enqueue_scripts($hook)
{
    // Only load on our plugin page
    if (strpos($hook, 'hungry-resource-monitor') === false) {
        return;
    }

    // Enqueue Chart.js (bundled locally)
    wp_enqueue_script(
        'chartjs',
        NANDRESMON_PLUGIN_URL . 'assets/js/chart.min.js',
        array(),
        '4.5.1',
        true
    );

    wp_enqueue_script(
        'nandresmon-admin',
        NANDRESMON_PLUGIN_URL . 'assets/js/admin-dashboard.js',
        array('jquery', 'chartjs'),
        NANDRESMON_VERSION,
        true
    );

    wp_enqueue_style(
        'nandresmon-admin',
        NANDRESMON_PLUGIN_URL . 'assets/css/admin-styles.css',
        array(),
        NANDRESMON_VERSION,
        'all'
    );

    // Localize script
    wp_localize_script('nandresmon-admin', 'nandresmonAdmin', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('nandresmon_admin_nonce'),
        'strings' => array(
            'confirm_cleanup' => __('Are you sure you want to perform this cleanup action? This cannot be undone.', 'hungry-resource-monitor'),
            'confirm_delete' => __('Are you sure you want to delete this item?', 'hungry-resource-monitor'),
            'processing' => __('Processing...', 'hungry-resource-monitor'),
            'success' => __('Success!', 'hungry-resource-monitor'),
            'error' => __('An error occurred. Please try again.', 'hungry-resource-monitor'),
        ),
    ));
}

/**
 * Initialize default options
 */
function nandresmon_initialize_options()
{
    $defaults = array(
        'enable_monitoring' => 1,
        'enable_toolbar' => 1,
        'enable_email_reports' => 1,
        'email_frequency' => 'weekly',
        'email_recipient' => get_option('admin_email'),
        'data_retention_days' => 30,
        'track_plugins' => 1,
        'track_themes' => 1,
        'track_queries' => 1,
    );

    $existing = get_option(NANDRESMON_OPTIONS_KEY);
    if (!$existing) {
        add_option(NANDRESMON_OPTIONS_KEY, $defaults);
    } else {
        // Merge with defaults to add any new options
        update_option(NANDRESMON_OPTIONS_KEY, array_merge($defaults, $existing));
    }
}

/**
 * Schedule cron events
 */
function nandresmon_schedule_events()
{
    // Schedule daily cleanup
    if (!wp_next_scheduled('nandresmon_daily_cleanup')) {
        wp_schedule_event(time(), 'daily', 'nandresmon_daily_cleanup');
    }

    // Schedule email reports based on settings
    $options = get_option(NANDRESMON_OPTIONS_KEY);
    if (!empty($options['enable_email_reports'])) {
        $frequency = !empty($options['email_frequency']) ? $options['email_frequency'] : 'weekly';
        if (!wp_next_scheduled('nandresmon_send_report')) {
            wp_schedule_event(time(), $frequency, 'nandresmon_send_report');
        }
    }
}

/**
 * Clear scheduled events
 */
function nandresmon_clear_scheduled_events()
{
    wp_clear_scheduled_hook('nandresmon_daily_cleanup');
    wp_clear_scheduled_hook('nandresmon_send_report');
}

// Hook for daily cleanup
add_action('nandresmon_daily_cleanup', 'nandresmon_run_daily_cleanup');

/**
 * Run daily cleanup of old data
 */
function nandresmon_run_daily_cleanup()
{
    global $wpdb;

    $options = get_option(NANDRESMON_OPTIONS_KEY);
    $retention_days = !empty($options['data_retention_days']) ? absint($options['data_retention_days']) : 30;

    $table_name = $wpdb->prefix . NANDRESMON_LOGS_TABLE;
    $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$retention_days} days"));

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query required for efficient cleanup of old monitoring data; no caching needed for delete operation
    $wpdb->query(
        $wpdb->prepare(
            "DELETE FROM %i WHERE recorded_at < %s",
            $table_name,
            $cutoff_date
        )
    );
}

// Hook for email reports
add_action('nandresmon_send_report', 'nandresmon_send_scheduled_report');

/**
 * Send scheduled email report
 */
function nandresmon_send_scheduled_report()
{
    $options = get_option(NANDRESMON_OPTIONS_KEY);
    if (empty($options['enable_email_reports'])) {
        return;
    }

    nandresmon_send_email_report();
}
