<?php

namespace HugeAddons\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Icons_Manager;
use Elementor\Utils;
use Elementor\Group_Control_Typography;



if ( !defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class HugeProductCarouselWidget extends Widget_Base {

    public function get_name() {
        return 'thha-huge-product-carousel';
    }

    public function get_title() {
        return __( 'Huge Product Carousel', 'huge-addons-for-elementor' );
    }

    public function get_icon() {
        return 'thha-icon hugeicon-product-carousel';
    }

    public function get_categories() {
        return ['thha-addons-category']; 
    }

    public function get_style_depends() {
        $widget_name = strtolower( str_replace( '_', '-', $this->get_name() ) );
        $css_file_path = THHA_PLUGIN_PATH . "public/css/{$widget_name}.min.css";

        wp_enqueue_style( 'ha-owl-carousel', THHA_URL . "public/css/owl.carousel.min.css", [], THHA_VERSION );
        wp_enqueue_style( 'ha-owl-carousel-theme', THHA_URL . "public/css/owl.theme.default.min.css", [], THHA_VERSION );
        if ( file_exists( $css_file_path ) ) {
            wp_register_style( $widget_name, THHA_URL . "public/css/{$widget_name}.min.css", [], THHA_VERSION );
            return [ $widget_name ];
        }
        return [];
    }

    public function get_script_depends() {
        $widget_name = strtolower( str_replace( '_', '-', $this->get_name() ) );
        $js_file_path = THHA_PLUGIN_PATH . "public/js/{$widget_name}.min.js";

        wp_register_script( 'ha-owl-carousel', THHA_URL . "public/js/owl.carousel.min.js", ['jquery'], THHA_VERSION, true );

        if ( file_exists( $js_file_path ) ) {
            wp_register_script( $widget_name, THHA_URL . "public/js/{$widget_name}.min.js", ['jquery', 'ha-owl-carousel'], THHA_VERSION, true );
            return [ $widget_name ];
        }
        return [];
    }

    /**
     * Get WooCommerce products list
     * @return array
     */
    private function get_woocommerce_products() {
        $products = wc_get_products( [
            'status' => 'publish',
            'limit'  => -1, // Be careful with huge catalogs; consider limiting or using AJAX
        ] );

        $options = [];
        foreach ( $products as $product ) {
            $options[ $product->get_id() ] = $product->get_name() . ' (#' . $product->get_id() . ')';
        }

        return $options;
    }

    protected function _register_controls() {
      // General Settings
      $this->start_controls_section(
          'product_carousel_content_section',
          [
              'label' => __( 'General Settings', 'huge-addons-for-elementor' ),
              'tab'   => Controls_Manager::TAB_CONTENT,
          ]
      );

      $this->add_control(
          'title',
          [
              'label'   => __( 'Title', 'huge-addons-for-elementor' ),
              'type'    => Controls_Manager::TEXT,
              'default' => __( 'Product Carousel Title', 'huge-addons-for-elementor' ),
          ]
      );

      $this->add_control(
          'description',
          [
              'label'   => __( 'Description', 'huge-addons-for-elementor' ),
              'type'    => Controls_Manager::TEXTAREA,
              'default' => __( 'This is a description', 'huge-addons-for-elementor' ),
          ]
      );

      // 1. Template Selection
        $this->add_control(
            'product_template',
            [
                'label'   => esc_html__( 'Choose Template', 'huge-addons-for-elementor' ),
                'type'    => \Elementor\Controls_Manager::SELECT,
                'default' => 'template-1',
                'options' => [
                    'template-1' => esc_html__( 'Template 1', 'huge-addons-for-elementor' ),
                    'template-2' => esc_html__( 'Template 2', 'huge-addons-for-elementor' ),
                ],
            ]
        );

      // Select Products
      $this->add_control(
        'selected_products',
        [
            'label'       => esc_html__( 'Select Products', 'huge-addons-for-elementor' ),
            'type'        => \Elementor\Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple'    => true,
            'options'     => $this->get_woocommerce_products(), 
            'placeholder' => esc_html__( 'Search products...', 'huge-addons-for-elementor' ),
        ]
        );


      $this->end_controls_section();

      $this->start_controls_section(
          'product_carousel_settings_section',
          [
              'label' => __( 'Carousel Settings', 'huge-addons-for-elementor' ),
              'tab'   => Controls_Manager::TAB_CONTENT,
          ]
      );

      $this->add_control(
			'autoplay',
            [
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label' => esc_html__( 'Autoplay', 'huge-addons-for-elementor' ),
                'default' => 'yes',
            ]
		);
      $this->add_control(
			'autoplay_timeout',
            [
                'type' => \Elementor\Controls_Manager::NUMBER,
                'label' => esc_html__( 'Autoplay Timeout (ms)', 'huge-addons-for-elementor' ),
                'default' => 5000,
            ]
		);

        $this->add_control(
            'loop',
            [
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label' => esc_html__( 'Loop', 'huge-addons-for-elementor' ),
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'nav',
            [
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label' => esc_html__( 'Show Navigation', 'huge-addons-for-elementor' ),
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'dots',
            [
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label' => esc_html__( 'Show Dots', 'huge-addons-for-elementor' ),
                'default' => 'yes',
                'dependency' => [
                    'nav' => 'yes', // Show dots only if navigation is enabled
                ],
            ]
        );

        $this->add_responsive_control(
            'items',
            [
                'label' => esc_html__( 'Items to Show', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'default' => 4,
                'tablet_default' => 2,
                'mobile_default' => 1,
            ]
        );

      $this->end_controls_section();

      // Style Section
      $this->start_controls_section(
          'product_carousel_style_section',
          [
              'label' => __( 'Style', 'huge-addons-for-elementor' ),
              'tab'   => Controls_Manager::TAB_STYLE,
          ]
      );

      $this->add_control(
        'margin',
        [
            'label' => esc_html__( 'Margin Between Items (px)', 'huge-addons-for-elementor' ),
            'type' => Controls_Manager::NUMBER,
            'default' => 10,
        ]
      );

      
      $this->end_controls_section();


      $this->start_controls_section(
        'section_title_style',
        [
            'label' => esc_html__( 'Widget Title', 'huge-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
        ]
    );

    $this->add_control(
        'title_color',
        [
            'label'     => esc_html__( 'Text Color', 'huge-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .title' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name'     => 'title_typography',
            'label'    => esc_html__( 'Typography', 'huge-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .title',
        ]
    );

    $this->add_control(
        'title_alignment',
        [
            'label'   => esc_html__( 'Alignment', 'huge-addons-for-elementor' ),
            'type'    => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__( 'Left', 'huge-addons-for-elementor' ),
                    'icon'  => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__( 'Center', 'huge-addons-for-elementor' ),
                    'icon'  => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__( 'Right', 'huge-addons-for-elementor' ),
                    'icon'  => 'eicon-text-align-right',
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .title' => 'text-align: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_section();

    $this->start_controls_section(
        'section_description_style',
        [
            'label' => esc_html__( 'Widget Description', 'huge-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
        ]
    );

    $this->add_control(
        'description_color',
        [
            'label'     => esc_html__( 'Text Color', 'huge-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .description' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name'     => 'description_typography',
            'label'    => esc_html__( 'Typography', 'huge-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .description',
        ]
    );

    $this->add_control(
        'description_alignment',
        [
            'label'   => esc_html__( 'Alignment', 'huge-addons-for-elementor' ),
            'type'    => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__( 'Left', 'huge-addons-for-elementor' ),
                    'icon'  => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__( 'Center', 'huge-addons-for-elementor' ),
                    'icon'  => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__( 'Right', 'huge-addons-for-elementor' ),
                    'icon'  => 'eicon-text-align-right',
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .description' => 'text-align: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_section();

    $this->start_controls_section(
        'card_style',
        [
            'label' => esc_html__( 'Card Style', 'huge-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name'     => 'category_typography',
            'label'    => esc_html__( 'Categories Typography', 'huge-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .product-catagory',
        ]
    );
    $this->add_control(
        'category_color',
        [
            'label'     => esc_html__( 'Categories Color', 'huge-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .product-catagory' => 'color: {{VALUE}};',
            ],
        ]
    );


    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name'     => 'product_title_typography',
            'label'    => esc_html__( 'Product title Typography', 'huge-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .product-title',
        ]
    );
    $this->add_control(
        'product_title_color',
        [
            'label'     => esc_html__( 'Product title Color', 'huge-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .product-title' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name'     => 'product_description_typography',
            'label'    => esc_html__( 'Product title Typography', 'huge-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .product-description',
        ]
    );
    $this->add_control(
        'product_description_color',
        [
            'label'     => esc_html__( 'Product title Color', 'huge-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .product-description' => 'color: {{VALUE}};',
            ],
        ]
    );
    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name'     => 'product_sale_price_typography',
            'label'    => esc_html__( 'Product Sale Price Typography', 'huge-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .product-price',
        ]
    );
    $this->add_control(
        'product_sale_price_color',
        [
            'label'     => esc_html__( 'Product Sale Price Color', 'huge-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .product-price' => 'color: {{VALUE}};',
            ],
        ]
    );
    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name'     => 'product_price_typography',
            'label'    => esc_html__( 'Product Price Typography', 'huge-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .product-price small',
        ]
    );
    $this->add_control(
        'product_price_color',
        [
            'label'     => esc_html__( 'Product Price Color', 'huge-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .product-price small' => 'color: {{VALUE}};',
            ],
        ]
    );


    $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $title = !empty($settings['title']) ? $settings['title'] : '';
        $description = !empty($settings['description']) ? $settings['description'] : '';
        $selected_ids = $settings['selected_products'];
        $current_template = $settings['product_template'];

        $carousel_settings = [
            'autoplay'        => 'yes' === $settings['autoplay'],
            'autoplayTimeout' => absint( $settings['autoplay_timeout'] ),
            'loop'            => 'yes' === $settings['loop'],
            'nav'             => 'yes' === $settings['nav'],
            'dots'            => 'yes' === $settings['dots'],
            'items'           => absint( $settings['items'] ),
            'items_tablet'    => !empty($settings['items_tablet']) ? absint( $settings['items_tablet'] ) : 2,
            'items_mobile'    => !empty($settings['items_mobile']) ? absint( $settings['items_mobile'] ) : 1,
            'margin'          => absint( $settings['margin'] ),
        ];

        $this->add_render_attribute( 'carousel', 'class', 'owl-carousel owl-theme' );
        $this->add_render_attribute( 'carousel', 'data-settings', json_encode( $carousel_settings ) );
        if ( 'template-2' === $current_template ) {
            $this->add_render_attribute( 'carousel', 'class', 'owl-carousel-2' );
        }

        // Base query arguments
        $args = [
            'post_type'      => 'product',
            'posts_per_page' => 6,
            'status'         => 'publish',
        ];

        // Check if manual selection exists
        if ( ! empty( $selected_ids ) && is_array( $selected_ids ) ) {
            $args['post__in'] = $selected_ids;
            $args['orderby'] = 'post__in'; // Maintain user's selection order
            $args['posts_per_page'] = -1;  // Show all selected
        } else {
            $args['orderby'] = 'date';
            $args['order']   = 'DESC';
        }

        $query = new \WP_Query( $args );
        if( $current_template == 'template-1'){  ?>
            <section class="thha-huge-product-carousel">
                <?php if( !empty($title) || !empty($description) ) { ?>
                    <div>
                        <?php if( !empty($title) )  { ?>
                            <h1 class="title"> <?php echo esc_html( $title ); ?> </h1>
                        <?php } ?>
                        <?php if( !empty($description) )  { ?>
                                <p class="description"> <?php echo esc_html( $description ); ?></p>
                        <?php } ?>
                    </div>
                <?php } ?>
                <div <?php echo $this->get_render_attribute_string( 'carousel' ); ?>>
                    <?php if ( $query->have_posts() ) { 
                        while ( $query->have_posts() ) {
                            $query->the_post();
                            // Get the WooCommerce product object
                            $product = wc_get_product( get_the_ID() );
                            
                            if ( ! $product ) continue;

                            // Fetching Data
                            $image_url    = get_the_post_thumbnail_url( get_the_ID(), 'medium' ) ?: wc_placeholder_img_src();
                            $categories   = wc_get_product_category_list( get_the_ID(), ', ', '<span class="product-catagory">', '</span>' );
                            $product_tags = $product->get_tag_ids();
                            $badge_text   = !empty($product_tags) ? get_term($product_tags[0])->name : ''; // First tag or default
                        ?>
                        <div class="item">
                            <div class="product-card">
                                <?php if(!empty($badge_text) ): ?>
                                    <div class="badge"><?php echo esc_html( $badge_text ); ?></div>
                                <?php endif; ?>
                                <div class="product-tumb">
                                    <img src="<?php echo esc_url( $image_url ); ?>" alt="<?php the_title_attribute(); ?>">
                                </div>

                                <div class="product-details">
                                    <?php echo $categories; ?>

                                    <h4 class="product-title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
                                    
                                    <p class="product-description"><?php echo wp_trim_words( get_the_excerpt(), 15, '...' ); ?></p>
                                    
                                    <div class="product-bottom-details">
                                        <div class="product-price">
                                            <?php if ( $product->is_on_sale() ) : ?>
                                                <small><?php echo $product->get_regular_price() . get_woocommerce_currency_symbol(); ?></small>
                                            <?php endif; ?>
                                            <?php echo $product->get_price() . get_woocommerce_currency_symbol(); ?>
                                        </div>
                                        
                                        <div class="product-links">
                                            <?php 
                                            // Add to cart URL
                                            $cart_url = $product->add_to_cart_url();
                                            ?>
                                            <a href="<?php the_permalink(); ?>"><i class="fa fa-heart"></i></a>
                                            <a href="<?php echo esc_url( $cart_url ); ?>"><i class="fa fa-shopping-cart"></i></a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php 
                        }
                            wp_reset_postdata();
                        } 
                    ?>
            </div>
            </section>
        <?php 
        } elseif( $current_template == 'template-2' ) { ?>
            <section class="thha-huge-product-carousel">
                <?php if( !empty($title) || !empty($description) ) { ?>
                    <div>
                        <?php if( !empty($title) )  { ?>
                            <h1 class="title"> <?php echo esc_html( $title ); ?> </h1>
                        <?php } ?>
                        <?php if( !empty($description) )  { ?>
                            <p> <?php echo esc_html( $description ); ?></p>
                        <?php } ?>
                    </div>
                <?php } ?>
                <div <?php echo $this->get_render_attribute_string( 'carousel' ); ?>>
                <?php 
                if ( $query->have_posts() ) : 
                    while ( $query->have_posts() ) : $query->the_post(); 
                        $product = wc_get_product( get_the_ID() );
                        if ( ! $product ) continue;

                        // Data Extraction
                        $image_url  = get_the_post_thumbnail_url( get_the_ID(), 'large' ) ?: wc_placeholder_img_src();
                        $brand_name = $product->get_attribute( 'pa_brand' ) ?: 'Brand'; // Change 'pa_brand' to your brand attribute slug
                        
                        // Handle Sizes (Check if it's a variable product)
                        $available_sizes = '';
                        if ( $product->is_type( 'variable' ) ) {
                            $variation_attributes = $product->get_variation_attributes();
                            if ( isset( $variation_attributes['pa_size'] ) ) { // Change 'pa_size' to your size attribute slug
                                $available_sizes = implode( ' , ', array_map( 'strtoupper', $variation_attributes['pa_size'] ) );
                            }
                        }
                ?>
                <div class="item">
                    <div class="container page-wrapper">
                        <div class="page-inner">
                            <div class="row">
                                <div class="el-wrapper">
                                    <div class="box-up">
                                        <img class="img" src="<?php echo esc_url( $image_url ); ?>" alt="<?php the_title_attribute(); ?>">
                                        <div class="img-info">
                                            <div class="info-inner">
                                                <span class="p-name"><?php the_title(); ?></span>
                                                <span class="p-company"><?php echo esc_html( $brand_name ); ?></span>
                                            </div>
                                            <?php if ( ! empty( $available_sizes ) ) : ?>
                                                <div class="a-size">Available sizes : <span class="size"><?php echo esc_html( $available_sizes ); ?></span></div>
                                            <?php endif; ?>
                                        </div>
                                    </div>

                                    <div class="box-down">
                                        <div class="h-bg">
                                            <div class="h-bg-inner"></div>
                                        </div>

                                        <a class="cart" href="<?php echo esc_url( $product->add_to_cart_url() ); ?>">
                                            <span class="price"><?php echo $product->get_price_html(); ?></span>
                                            <span class="add-to-cart">
                                                <span class="txt"><?php esc_html_e( 'Add in cart', 'your-textdomain' ); ?></span>
                                            </span>
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <?php 
                    endwhile; 
                    wp_reset_postdata(); 
                endif; 
                ?>
            </div>

            </section>
        <?php }

    }
}
