<?php
namespace HugeAddons\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Group_Control_Typography;


if ( !defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class HugePostGridWidget extends Widget_Base {
    public function get_name() {
        return 'thha-huge-post-grid';
    }
    
    public function get_title() {
        return __('Huge Post Grid', 'huge-addons-for-elementor');
    }
    
    public function get_icon() {
        // return 'eicon-posts-grid';
        return 'thha-icon hugeicon-post-grid';
    }
    
    public function get_categories(): array {
        return [ 'thha-addons-category' ];
    }

    public function get_style_depends() {
        $widget_name = strtolower( str_replace( '_', '-', $this->get_name() ) );
        $css_file_path = THHA_PLUGIN_PATH . "public/css/{$widget_name}.min.css";

        if ( file_exists( $css_file_path ) ) {
            wp_register_style( $widget_name, THHA_URL . "public/css/{$widget_name}.min.css", [], THHA_VERSION );
            return [ $widget_name ];
        }
        return [];
    }

    public function get_script_depends() {
        $widget_name = strtolower( str_replace( '_', '-', $this->get_name() ) );
        $js_file_path = THHA_PLUGIN_PATH . "public/js/{$widget_name}.min.js";

        if ( file_exists( $js_file_path ) ) {
            wp_register_script( $widget_name, THHA_URL . "public/js/{$widget_name}.min.js", ['jquery'], THHA_VERSION, true );
            return [ $widget_name, 'post-grid-loadmore', 'post-grid-pagination' ];
        }
        return [];
    }


    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Settings', 'huge-addons-for-elementor'),
                'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );


        $this->add_control(
            'post_style',
            [
                'label' => __('Post Design Style', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'huge-style1' => __('Design 1 - Classic Card', 'huge-addons-for-elementor'),
                    'huge-style2' => __('Design 2 - Boxed Layout', 'huge-addons-for-elementor'),
                    'huge-style3' => __('Design 3 - Overlay Card', 'huge-addons-for-elementor'),
                    'huge-style4' => __('Design 4 - Split Card', 'huge-addons-for-elementor'),
                    'huge-style5' => __('Design 5 - Modern Hover', 'huge-addons-for-elementor'),
                    'huge-style6' => __('Design 6 - Classic Overlay', 'huge-addons-for-elementor'),
                    'huge-style7' => __('Design 7 - Sleek Hover', 'huge-addons-for-elementor'),
                    'huge-style8' => __('Design 8 - Elegant Overlay', 'huge-addons-for-elementor'),
                    'huge-style9' => __('Design 9 - Creative Tilt', 'huge-addons-for-elementor'),
                    'huge-style10' => __('Design 10 - Minimal List', 'huge-addons-for-elementor'),
                ],
                'default' => 'huge-style1',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label' => __('Number of Posts', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 20,
                'step' => 1,
                'default' => 6,
            ]
        );

    // Get all categories dynamically
    $categories = get_categories();
    $category_options = ['' => __('All Categories', 'huge-addons-for-elementor')];

    foreach ($categories as $category) {
        $category_options[$category->slug] = $category->name;
    }

    // Add Category Dropdown in Elementor Controls
    $this->add_control(
        'selected_category',
        [
            'label' => __('Filter by Category', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::SELECT,
            'options' => $category_options,
            'default' => '',
        ]
    );

        // Toggle controls
        $this->add_control(
            'show_image',
            [
                'label' => __('Show Image', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'huge-addons-for-elementor'),
                'label_off' => __('Hide', 'huge-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'image_size',
            [
                'label' => __('Image Size', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'medium',
                'options' => [
                    'thumbnail'      => __('Thumbnail - 150x150', 'huge-addons-for-elementor'),
                    'medium'         => __('Medium - 300x300', 'huge-addons-for-elementor'),
                    'medium_large'   => __('Medium Large - 768x0', 'huge-addons-for-elementor'),
                    'large'          => __('Large - 1024x1024', 'huge-addons-for-elementor'),
                    '1536x1536'      => __('1536x1536', 'huge-addons-for-elementor'),
                    '2048x2048'      => __('2048x2048', 'huge-addons-for-elementor'),
                    'full'           => __('Full', 'huge-addons-for-elementor'),
                ],
                'condition' => [
                    'show_image' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'title_word_limit',
            [
                'label' => __('Title Word Limit', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 20,
                'default' => 10,
                'description' => __('Limit the number of words in the post title.', 'huge-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'show_content',
            [
                'label' => __('Show Content', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'content_type',
            [
                'label' => __('Content Type', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'excerpt' => __('Excerpt', 'huge-addons-for-elementor'),
                    'content' => __('Full Content', 'huge-addons-for-elementor'),
                ],
                'default' => 'excerpt',
                'condition' => [
                    'show_content' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'content_word_limit',
            [
                'label' => __('Content Word Limit', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 100,
                'default' => 20,
                'condition' => [
                    'show_content' => 'yes',
                ],
                'description' => __('Limit the number of words in the post excerpt/content.', 'huge-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'show_category',
            [
                'label' => __('Show Category', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'separator' => 'before',
                
            ]
        );

        $this->add_control(
            'show_tags',
            [
                'label' => __('Show Tags', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_author',
            [
                'label' => __('Show Author', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_date',
            [
                'label' => __('Show Date', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'show_readmore',
            [
                'label' => __('Read More', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'show_comments',
            [
                'label' => __('Show Comments', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );
        $this->add_control(
            'show_pagination',
            [
                'label' => __('Show Pagination', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'huge-addons-for-elementor'),
                'label_off' => __('No', 'huge-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'pagination_type',
            [
                'label' => __('Pagination Type', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'pagination' => __('Numbered Pagination', 'huge-addons-for-elementor'),
                    'load_more' => __('Load More Button', 'huge-addons-for-elementor'),
                ],
                'default' => 'pagination',
                'condition' => [
                    'show_pagination' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label'       => __('Columns', 'huge-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::SELECT,
                'description' => __('Set the number of columns for different screen sizes.', 'huge-addons-for-elementor'),
                'options'     => [
                    '1' => __('1 Column', 'huge-addons-for-elementor'),
                    '2' => __('2 Columns', 'huge-addons-for-elementor'),
                    '3' => __('3 Columns', 'huge-addons-for-elementor'),
                    '4' => __('4 Columns', 'huge-addons-for-elementor'),
                    '5' => __('5 Columns', 'huge-addons-for-elementor'),
                    '6' => __('6 Columns', 'huge-addons-for-elementor'),
                    '8' => __('8 Columns', 'huge-addons-for-elementor'),
                ],
                'default'     => '3',
                'separator' => 'before',
                'selectors' => [
                    '{{WRAPPER}} .huge-post-grid' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
                ],
                
            ]
        );

        $this->add_responsive_control(
            'column_gap',
            [
                'label' => __('Column Gap', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem','%'],
                'range' => [
                    'px' => ['min' => 0, 'max' => 50],
                    'em' => ['min' => 0, 'max' => 5],
                    'rem' => ['min' => 0, 'max' => 5],
                    '%' => ['min' => 0, 'max' => 100],
                ],
                'default' => ['size' => 20, 'unit' => 'px'],
                'selectors' => [
                    '{{WRAPPER}} .huge-post-grid' => 'column-gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'row_gap',
            [
                'label' => __('Row Gap', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem','%'],
                'range' => [
                    'px' => ['min' => 0, 'max' => 50],
                    'em' => ['min' => 0, 'max' => 5],
                    'rem' => ['min' => 0, 'max' => 5],
                    '%' => ['min' => 0, 'max' => 100],
                ],
                'default' => ['size' => 20, 'unit' => 'px'],
                'selectors' => [
                    '{{WRAPPER}} .huge-post-grid' => 'row-gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );


        $this->end_controls_section();


    /***************************************************************************************** Style Tab ********************************************************************************************************/
    // Container Style
    $this->start_controls_section(
        'section_container_style',
        [
            'label' => __('Container', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
        ]
    );

    // Background
    $this->add_group_control(
        \Elementor\Group_Control_Background::get_type(),
        [
            'name' => 'container_background',
            'label' => __('Background', 'huge-addons-for-elementor'),
            'types' => ['classic', 'gradient'],
            'selector' => '{{WRAPPER}} .huge-post-item',
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'container_border',
            'selector' => '{{WRAPPER}} .huge-post-item',
        ]
    );

    $this->add_responsive_control(
        'container_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'container_box_shadow',
            'selector' => '{{WRAPPER}} .huge-post-item',
        ]
    );

    $this->add_responsive_control(
        'container_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'container_margin',
        [
            'label' => __('Margin', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    // Hover effects
    $this->start_controls_tabs('container_hover_tabs');

    $this->start_controls_tab(
        'container_normal',
        [
            'label' => __('Normal', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'container_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item' => 'background-color: {{VALUE}}',
            ],
        ]
    );

    $this->end_controls_tab();

    $this->start_controls_tab(
        'container_hover',
        [
            'label' => __('Hover', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'container_bg_color_hover',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover' => 'background-color: {{VALUE}}',
            ],
        ]
    );

    $this->add_control(
        'container_border_color_hover',
        [
            'label' => __('Border Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover' => 'border-color: {{VALUE}}',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'container_box_shadow_hover',
            'label' => __('Box Shadow', 'huge-addons-for-elementor'),
            'selector' => '{{WRAPPER}} .huge-post-item:hover',
        ]
    );

    $this->add_control(
        'container_hover_transition',
        [
            'label' => __('Transition Duration', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::SLIDER,
            'range' => [
                'px' => [
                    'max' => 3,
                    'step' => 0.1,
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .huge-post-item' => 'transition-duration: {{SIZE}}s',
            ],
        ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();

    $this->end_controls_section();

     // Title Style

    $this->start_controls_section(
        'section_title_style',
        [
            'label' => __('Title', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'title_typography',
            'selector' => '{{WRAPPER}} .huge-post-title',
        ]
    );

    $this->add_control(
        'title_color',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-title a' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'title_hover_color',
        [
            'label' => __('Hover Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-title a' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'title_alignment',
        [
            'label' => __('Alignment', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => __('Left', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => __('Center', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => __('Right', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'left',
            'selectors' => [
                '{{WRAPPER}} .huge-post-title' => 'text-align: {{VALUE}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'title_margin',
        [
            'label' => __('Margin', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );
    

    $this->end_controls_section();

    // Content Style
    $this->start_controls_section(
        'section_excerpt_style',
        [
            'label' => __('Content', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                    'show_content' => 'yes'
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'excerpt_typography',
            'selector' => '{{WRAPPER}} .huge-post-excerpt',
        ]
    );

    $this->add_control(
        'excerpt_color',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-excerpt' => 'color: {{VALUE}};',
            ],
        ]
    );
    $this->add_control(
        'content_background',
        [
            'label' => __('Content Background', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-content' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'content_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );
        $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'content_border',
            'selector' => '{{WRAPPER}} .huge-post-content',
        ]
    );

    $this->add_responsive_control(
        'content_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'content_box_shadow',
            'selector' => '{{WRAPPER}} .huge-post-content',
        ]
    );

    $this->add_responsive_control(
        'excerpt_alignment',
        [
            'label' => __('Alignment', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => __('Left', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => __('Center', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => __('Right', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'left',
            'selectors' => [
                '{{WRAPPER}} .huge-post-excerpt' => 'text-align: {{VALUE}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'excerpt_margin',
        [
            'label' => __('Margin', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-excerpt' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->end_controls_section();

    // Category Style
    $this->start_controls_section(
        'section_category_style',
        [
            'label' => __('Category', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                'show_category' => 'yes',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'category_typography',
            'selector' => '{{WRAPPER}} .huge-post-category',
        ]
    );

    // Category color tabs
    $this->start_controls_tabs('category_colors');

    $this->start_controls_tab(
        'category_color_normal',
        [
            'label' => __('Normal', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'category_text_color',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-category' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'category_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-category' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();

    $this->start_controls_tab(
        'category_text_color_hover',
        [
            'label' => __('Hover', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'category_color_hover',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-category' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'category_bg_color_hover',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-category' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();

    $this->add_responsive_control(
        'category_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'category_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-category' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'category_border',
            'selector' => '{{WRAPPER}} .huge-post-category',
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'category_box_shadow',
            'selector' => '{{WRAPPER}} .huge-post-category',
        ]
    );
    $this->add_responsive_control(
        'category_alignment',
        [
            'label' => __('Alignment', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => __('Left', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => __('Center', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => __('Right', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'left',
            'selectors' => [
                '{{WRAPPER}} .huge-post-category' => 'text-align: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_section();

    // Author Style
    $this->start_controls_section(
        'section_author_style',
        [
            'label' => __('Author', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                'show_author' => 'yes',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'author_typography',
            'selector' => '{{WRAPPER}} .huge-post-author',
        ]
    );

    // Author color tabs
    $this->start_controls_tabs('author_colors');

    $this->start_controls_tab(
        'author_color_normal',
        [
            'label' => __('Normal', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'author_text_color',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-author' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'author_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-author' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();

    $this->start_controls_tab(
        'author_color_hover',
        [
            'label' => __('Hover', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'author_text_color_hover',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-author' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'author_bg_color_hover',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-author' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();

    $this->add_responsive_control(
        'author_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-author' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'author_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-author' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'author_border',
            'selector' => '{{WRAPPER}} .huge-post-author',
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'author_box_shadow',
            'selector' => '{{WRAPPER}} .huge-post-author',
        ]
    );

    $this->end_controls_section();

    // Date Style
    $this->start_controls_section(
        'section_date_style',
        [
            'label' => __('Date', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                'show_date' => 'yes',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'date_typography',
            'selector' => '{{WRAPPER}} .huge-post-date',
        ]
    );

    // Date color tabs
    $this->start_controls_tabs('date_colors');

    $this->start_controls_tab(
        'date_color_normal',
        [
            'label' => __('Normal', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'date_text_color',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-date' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'date_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-date' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();

    $this->start_controls_tab(
        'date_color_hover',
        [
            'label' => __('Hover', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'date_text_color_hover',
        [
            'label' => __('Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-date' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'date_bg_color_hover',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-date' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();

    $this->add_responsive_control(
        'date_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-date' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'date_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-date' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'date_border',
            'selector' => '{{WRAPPER}} .huge-post-date',
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'date_box_shadow',
            'selector' => '{{WRAPPER}} .huge-post-date',
        ]
    );

    $this->end_controls_section();

    // Tags Style
    $this->start_controls_section(
        'section_tags_style',
        [
            'label' => __('Tags', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                'show_tags' => 'yes',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'tags_typography',
            'selector' => '{{WRAPPER}} .huge-post-tags, {{WRAPPER}} .huge-post-tags a',
        ]
    );
    $this->add_responsive_control(
        'tag_alignment',
        [
            'label' => __('Alignment', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => __('Left', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => __('Center', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => __('Right', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'left',
            'selectors' => [
                '{{WRAPPER}} .huge-post-tags, {{WRAPPER}} .huge-post-tags a' => 'text-align: {{VALUE}};',
            ],
        ]
    );

    // Tags color
    $this->start_controls_tabs('tags_colors');

    $this->start_controls_tab(
        'tags_color_normal',
        [
            'label' => __('Normal', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'tags_text_color',
        [
            'label' => __('Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-tags a' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'tags_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-tags a' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();

    $this->start_controls_tab(
        'tags_color_hover',
        [
            'label' => __('Hover', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'tags_text_color_hover',
        [
            'label' => __('Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-tags a' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'tags_bg_color_hover',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-post-tags a' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();

    // Tags spacing
    $this->add_responsive_control(
        'tags_gap',
        [
            'label' => __('Gap Between Tags', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::SLIDER,
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 50,
                ],
                'rem' => [
                    'min' => 0,
                    'max' => 5,
                ],
            ],
            'size_units' => ['px', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-tags a:not(:last-child)' => 'margin-right: {{SIZE}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'tags_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-tags a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );
    $this->add_responsive_control(
        'tags_margin',
        [
            'label' => __('Margin', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-tags a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'tags_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-tags a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'tags_box_shadow',
            'selector' => '{{WRAPPER}} .huge-post-tags a',
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'tags_border',
            'selector' => '{{WRAPPER}} .huge-post-tags a',
        ]
    );

    $this->end_controls_section();

    // Image Style
    $this->start_controls_section(
        'section_image_style',
        [
            'label' => __('Image', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                'show_image' => 'yes',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'image_border',
            'selector' => '{{WRAPPER}} .huge-post-thumbnail img',
        ]
    );

    $this->add_responsive_control(
        'image_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-thumbnail img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                '{{WRAPPER}} .huge-post-thumbnail' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'image_box_shadow',
            'selector' => '{{WRAPPER}} .huge-post-thumbnail img',
        ]
    );

    $this->add_responsive_control(
        'image_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-thumbnail' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'image_margin',
        [
            'label' => __('Margin', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-post-thumbnail' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    // Overlay Section
    $this->add_control(
        'image_overlay_heading',
        [
            'label' => __('Image Overlay', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::HEADING,
            'separator' => 'before',
        ]
    );

    // Overlay Type
    $this->add_control(
        'image_overlay_type',
        [
            'label' => __('Overlay Type', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::SELECT,
            'default' => 'default',
            'options' => [
                'default' => __('Default', 'huge-addons-for-elementor'),
                'color' => __('Color', 'huge-addons-for-elementor'),
                'gradient' => __('Gradient', 'huge-addons-for-elementor'),
            ],
        ]
    );

    // Color Overlay
    $this->add_control(
        'image_overlay_color',
        [
            'label' => __('Overlay Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-overlay' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'image_overlay_type' => 'color',
            ],
        ]
    );

    // Gradient Overlay
    $this->add_group_control(
        \Elementor\Group_Control_Background::get_type(),
        [
            'name' => 'image_overlay_gradient',
            'label' => __('Overlay Gradient', 'huge-addons-for-elementor'),
            'types' => ['gradient'],
            'selector' => '{{WRAPPER}} .huge-post-overlay',
            'condition' => [
                'image_overlay_type' => 'gradient',
            ],
        ]
    );

    // Overlay Opacity
    $this->add_control(
        'image_overlay_opacity',
        [
            'label' => __('Overlay Opacity', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::SLIDER,
            'range' => [
                'px' => [
                    'max' => 1,
                    'min' => 0,
                    'step' => 0.01,
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .huge-post-overlay' => 'opacity: {{SIZE}};',
            ],
            'conditions' => [
                'relation' => 'or',
                'terms' => [
                    [
                        'name' => 'image_overlay_type',
                        'operator' => '==',
                        'value' => 'color'
                    ],
                    [
                        'name' => 'image_overlay_type',
                        'operator' => '==',
                        'value' => 'gradient'
                    ],
                ],
            ],
        ]
    );

    // Hover Effects
    $this->add_control(
        'image_hover_effects_heading',
        [
            'label' => __('Hover Effects', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::HEADING,
            'separator' => 'before',
            'condition' => [
                'image_overlay_type!' => 'default',
            ],
        ]
    );

    // Hover Overlay Opacity
    $this->add_control(
        'image_overlay_opacity_hover',
        [
            'label' => __('Hover Overlay Opacity', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::SLIDER,
            'range' => [
                'px' => [
                    'max' => 1,
                    'min' => 0,
                    'step' => 0.01,
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .huge-post-thumbnail:hover .huge-post-overlay' => 'opacity: {{SIZE}};',
            ],
            'conditions' => [
                'relation' => 'or',
                'terms' => [
                    [
                        'name' => 'image_overlay_type',
                        'operator' => '==',
                        'value' => 'color'
                    ],
                    [
                        'name' => 'image_overlay_type',
                        'operator' => '==',
                        'value' => 'gradient'
                    ],
                ],
            ],
        ]
    );

    $this->end_controls_section();

    // Comments Style
$this->start_controls_section(
    'section_comments_style',
    [
        'label' => __('Comments', 'huge-addons-for-elementor'),
        'tab' => \Elementor\Controls_Manager::TAB_STYLE,
        'condition' => [
            'show_comments' => 'yes',
        ],
    ]
);

$this->add_group_control(
    \Elementor\Group_Control_Typography::get_type(),
    [
        'name' => 'comments_typography',
        'selector' => '{{WRAPPER}} .huge-post-comments',
    ]
);

// Comments color tabs
$this->start_controls_tabs('comments_colors');

$this->start_controls_tab(
    'comments_color_normal',
    [
        'label' => __('Normal', 'huge-addons-for-elementor'),
    ]
);

$this->add_control(
    'comments_text_color',
    [
        'label' => __('Color', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::COLOR,
        'selectors' => [
            '{{WRAPPER}} .huge-post-comments' => 'color: {{VALUE}};',
        ],
    ]
);

$this->add_control(
    'comments_bg_color',
    [
        'label' => __('Background Color', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::COLOR,
        'selectors' => [
            '{{WRAPPER}} .huge-post-comments' => 'background-color: {{VALUE}};',
        ],
    ]
);

$this->end_controls_tab();

$this->start_controls_tab(
    'comments_color_hover',
    [
        'label' => __('Hover', 'huge-addons-for-elementor'),
    ]
);

$this->add_control(
    'comments_text_color_hover',
    [
        'label' => __('Color', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::COLOR,
        'selectors' => [
            '{{WRAPPER}} .huge-post-item:hover .huge-post-comments' => 'color: {{VALUE}};',
        ],
    ]
);

$this->add_control(
    'comments_bg_color_hover',
    [
        'label' => __('Background Color', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::COLOR,
        'selectors' => [
            '{{WRAPPER}} .huge-post-item:hover .huge-post-comments' => 'background-color: {{VALUE}};',
        ],
    ]
);

$this->end_controls_tab();
$this->end_controls_tabs();

$this->add_responsive_control(
    'comments_padding',
    [
        'label' => __('Padding', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'rem', 'em'],
        'selectors' => [
            '{{WRAPPER}} .huge-post-comments' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
    ]
);

$this->add_responsive_control(
    'comments_border_radius',
    [
        'label' => __('Border Radius', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'rem'],
        'selectors' => [
            '{{WRAPPER}} .huge-post-comments' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
    ]
);

$this->add_group_control(
    \Elementor\Group_Control_Border::get_type(),
    [
        'name' => 'comments_border',
        'selector' => '{{WRAPPER}} .huge-post-comments',
    ]
);

$this->add_group_control(
    \Elementor\Group_Control_Box_Shadow::get_type(),
    [
        'name' => 'comments_box_shadow',
        'selector' => '{{WRAPPER}} .huge-post-comments',
    ]
);

$this->add_control(
    'comments_icon_heading',
    [
        'label' => __('Icon', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::HEADING,
        'separator' => 'before',
    ]
);

$this->add_control(
    'comments_icon_color',
    [
        'label' => __('Icon Color', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::COLOR,
        'selectors' => [
            '{{WRAPPER}} .huge-post-comments i' => 'color: {{VALUE}};',
        ],
    ]
);

$this->add_control(
    'comments_icon_size',
    [
        'label' => __('Icon Size', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::SLIDER,
        'size_units' => ['px', 'em', 'rem'],
        'range' => [
            'px' => [
                'min' => 1,
                'max' => 100,
            ],
        ],
        'selectors' => [
            '{{WRAPPER}} .huge-post-comments i' => 'font-size: {{SIZE}}{{UNIT}};',
        ],
    ]
);

$this->add_responsive_control(
    'comments_icon_spacing',
    [
        'label' => __('Icon Spacing', 'huge-addons-for-elementor'),
        'type' => \Elementor\Controls_Manager::SLIDER,
        'size_units' => ['px', 'em', 'rem'],
        'range' => [
            'px' => [
                'min' => 0,
                'max' => 50,
            ],
        ],
        'selectors' => [
            '{{WRAPPER}} .huge-post-comments i' => 'margin-right: {{SIZE}}{{UNIT}};',
        ],
    ]
);

$this->end_controls_section();

    // Read More Style
    $this->start_controls_section(
        'section_readmore_style',
        [
            'label' => __('Read More', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                'show_readmore' => 'yes', 
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'readmore_typography',
            'selector' => '{{WRAPPER}} .huge-read-more',
        ]
    );

    // Color tabs (normal/hover)
    $this->start_controls_tabs('readmore_colors');

    // Normal tab
    $this->start_controls_tab(
        'readmore_color_normal',
        [
            'label' => __('Normal', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'readmore_text_color',
        [
            'label' => __('Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-read-more' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'readmore_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-read-more' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();

    // Hover tab
    $this->start_controls_tab(
        'readmore_color_hover',
        [
            'label' => __('Hover', 'huge-addons-for-elementor'),
        ]
    );

    $this->add_control(
        'readmore_text_color_hover',
        [
            'label' => __('Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-read-more' => 'color: {{VALUE}};',
            ],
        ]
    );

    $this->add_control(
        'readmore_bg_color_hover',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .huge-post-item:hover .huge-read-more' => 'background-color: {{VALUE}};',
            ],
        ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();

    // Spacing
    $this->add_responsive_control(
        'readmore_margin',
        [
            'label' => __('Margin', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-read-more' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'readmore_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem', 'em'],
            'selectors' => [
                '{{WRAPPER}} .huge-read-more' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'readmore_border',
            'selector' => '{{WRAPPER}} .huge-read-more',
        ]
    );

    $this->add_responsive_control(
        'readmore_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'rem'],
            'selectors' => [
                '{{WRAPPER}} .huge-read-more' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'readmore_box_shadow',
            'selector' => '{{WRAPPER}} .huge-read-more',
        ]
    );

    $this->end_controls_section();

    // Pagination Style
    $this->start_controls_section(
        'section_pagination_style',
        [
            'label' => __('Pagination', 'huge-addons-for-elementor'),
            'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            'condition' => [
                    'show_pagination' => 'yes',
            ],
        ]
    );

    // Load More Button Tab
    $this->start_controls_tabs('pagination_tabs');

    // Load More Button - Normal
    $this->start_controls_tab(
        'load_more_normal',
        [
            'label' => __('Load More', 'huge-addons-for-elementor'),
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_control(
        'load_more_text_color',
        [
            'label' => __('Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .load-more-btn' => 'color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_control(
        'load_more_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .load-more-btn' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'load_more_typography',
            'selector' => '{{WRAPPER}} .load-more-btn',
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'load_more_border',
            'selector' => '{{WRAPPER}} .load-more-btn',
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_responsive_control(
        'load_more_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'em'],
            'selectors' => [
                '{{WRAPPER}} .load-more-btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_responsive_control(
        'load_more_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', 'em', '%'],
            'selectors' => [
                '{{WRAPPER}} .load-more-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'load_more_box_shadow',
            'selector' => '{{WRAPPER}} .load-more-btn',
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->end_controls_tab();

    // Load More Button - Hover
    $this->start_controls_tab(
        'load_more_hover',
        [
            'label' => __('Hover', 'huge-addons-for-elementor'),
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_control(
        'load_more_text_color_hover',
        [
            'label' => __('Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .load-more-btn:hover' => 'color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_control(
        'load_more_bg_color_hover',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .load-more-btn:hover' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_control(
        'load_more_border_color_hover',
        [
            'label' => __('Border Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .load-more-btn:hover' => 'border-color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Box_Shadow::get_type(),
        [
            'name' => 'load_more_box_shadow_hover',
            'selector' => '{{WRAPPER}} .load-more-btn:hover',
            'condition' => [
                'pagination_type' => 'load_more',
            ],
        ]
    );

    $this->end_controls_tab();
    $this->end_controls_tabs();

    // Numbered Pagination
    $this->add_control(
        'pagination_numbers_heading',
        [
            'label' => __('Numbered Pagination', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::HEADING,
            'separator' => 'before',
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Border::get_type(),
        [
            'name' => 'pagination_numbers_border',
            'selector' => '{{WRAPPER}} .post-grid-pagination a, {{WRAPPER}} .post-grid-pagination span',
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_control(
        'pagination_numbers_color',
        [
            'label' => __('Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination a' => 'color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_control(
        'pagination_numbers_bg_color',
        [
            'label' => __('Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination a' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_control(
        'pagination_numbers_active_color',
        [
            'label' => __('Active Text Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination .current' => 'color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_control(
        'pagination_numbers_active_bg_color',
        [
            'label' => __('Active Background Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination .current' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_control(
        'pagination_numbers_active_border_color',
        [
            'label' => __('Active Border Color', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination .current' => 'border-color: {{VALUE}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_group_control(
        \Elementor\Group_Control_Typography::get_type(),
        [
            'name' => 'pagination_numbers_typography',
            'selector' => '{{WRAPPER}} .post-grid-pagination a, {{WRAPPER}} .post-grid-pagination span',
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_responsive_control(
        'pagination_numbers_padding',
        [
            'label' => __('Padding', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', 'em', '%'],
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination a, {{WRAPPER}} .post-grid-pagination span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_responsive_control(
        'pagination_numbers_border_radius',
        [
            'label' => __('Border Radius', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'em'],
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination a, {{WRAPPER}} .post-grid-pagination span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_responsive_control(
        'pagination_numbers_gap',
        [
            'label' => __('Gap Between Items', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['px'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 50,
                ],
            ],
            'selectors' => [
                '{{WRAPPER}} .post-grid-pagination ul' => 'gap: {{SIZE}}{{UNIT}};',
            ],
            'condition' => [
                'pagination_type' => 'pagination',
            ],
        ]
    );

    $this->add_responsive_control(
        'pagination_alignment',
        [
            'label' => __('Alignment', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => __('Left', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => __('Center', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => __('Right', 'huge-addons-for-elementor'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'center',
            'selectors' => [
                '{{WRAPPER}} .load-more-wrapper' => 'text-align: {{VALUE}};',
                '{{WRAPPER}} .post-grid-pagination' => 'justify-content: {{VALUE}};',
            ],
        ]
    );

    $this->add_responsive_control(
        'pagination_margin',
        [
            'label' => __('Margin', 'huge-addons-for-elementor'),
            'type' => \Elementor\Controls_Manager::DIMENSIONS,
            'size_units' => ['px', '%', 'em'],
            'selectors' => [
                '{{WRAPPER}} .load-more-wrapper, {{WRAPPER}} .post-grid-pagination' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
        ]
    );

    $this->end_controls_section();
    
    }
    
    protected function render() {
        $settings = $this->get_settings_for_display();
        $unique_id = 'huge-post-grid-' . $this->get_id();

        // Include the render file and call the render function
        require_once __DIR__ . '/renders/render-huge-post-grid-widget.php';
        \HugeAddons\Widgets\Renders\render_huge_post_grid_widget( $settings, $unique_id );
    }
}
