<?php

namespace HugeAddons\Widgets;

use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Widget_Base;

if ( !defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class HugeAdvancedImageGalleryWidget extends Widget_Base {

    public function get_name(): string {
        return 'huge-advanced-image-gallery';
    }

    public function get_title(): string {
        return esc_html__( 'Huge Advanced Image Gallery', 'huge-addons-for-elementor' );
    }

    public function get_icon(): string {
        return 'thha-icon hugeicon-advanced-image-gallery';
    }

    public function get_categories(): array {
        return [ 'thha-addons-category' ];
    }
    
    public function get_script_depends() {
        $widget_name = strtolower( str_replace( '_', '-', $this->get_name() ) );
        $js_file_path = THHA_PLUGIN_PATH . "public/js/thha-{$widget_name}.min.js";

        if ( file_exists( $js_file_path ) ) {
            wp_register_script( "thha-{$widget_name}", THHA_URL . "public/js/thha-{$widget_name}.min.js", ['jquery'], THHA_VERSION, true );
            return [ "thha-{$widget_name}", 'huge_imagesloaded', 'huge_masonry' ];
        }
        return [];
    }

    public function get_style_depends() {
        $widget_name = strtolower( str_replace( '_', '-', $this->get_name() ) );
        $css_file_path = THHA_PLUGIN_PATH . "public/css/thha-{$widget_name}.min.css";
        if ( file_exists( $css_file_path ) ) {
            wp_register_style( $widget_name, THHA_URL . "public/css/thha-{$widget_name}.min.css", [], THHA_VERSION );
            return [ $widget_name ];
        }
        return [];
    }
    

    protected function register_controls() {
        $this->start_controls_section(
            'content_section',
            [
                'label' => __( 'Content', 'huge-addons-for-elementor' ),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'images',
            [
                'label' => __( 'Add Images', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::GALLERY,
                'default' => [],
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label'       => __('Columns', 'huge-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::SELECT,
                'description' => __('Set the number of columns for different screen sizes.', 'huge-addons-for-elementor'),
                'options'     => [
                    '1' => __('1 Column', 'huge-addons-for-elementor'),
                    '2' => __('2 Columns', 'huge-addons-for-elementor'),
                    '3' => __('3 Columns', 'huge-addons-for-elementor'),
                    '4' => __('4 Columns', 'huge-addons-for-elementor'),
                    '5' => __('5 Columns', 'huge-addons-for-elementor'),
                    '6' => __('6 Columns', 'huge-addons-for-elementor'),
                    '8' => __('8 Columns', 'huge-addons-for-elementor'),
                    '10' => __('10 Columns', 'huge-addons-for-elementor'),
                    '12' => __('12 Columns', 'huge-addons-for-elementor'),
                ],
                'default'     => '3',
                'selectors'   => [
                    '{{WRAPPER}} .advanced-image-gallery' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
                ],
                'condition' => [
                    'enable_masonry!' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'gap',
            [
                'label' => __('Grid Gap', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem', '%'], 
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 5,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'size' => 10,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .advanced-image-gallery' => 'gap: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'enable_masonry!' => 'yes',
                ],
            ]
        );
        

        $this->add_control(
            'enable_lightbox',
            [
                'label' => __( 'Enable Lightbox', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __( 'Yes', 'huge-addons-for-elementor' ),
                'label_off' => __( 'No', 'huge-addons-for-elementor' ),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'enable_lightbox_gallery',
            [
                'label'        => __( 'Enable Lightbox Gallery', 'huge-addons-for-elementor' ),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'description'  => __( 'If enabled, images will be grouped in a gallery for lightbox navigation.', 'huge-addons-for-elementor' ),
                'default'      => 'yes',
                'condition'    => [
                    'enable_lightbox' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'enable_overlay',
            [
                'label'        => __( 'Enable Overlay', 'huge-addons-for-elementor' ),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'default'      => 'no',
            ]
        );

        $this->add_control(
            'show_icon',
            [
                'label'        => __( 'Show Icon on Overlay', 'huge-addons-for-elementor' ),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'default'      => 'no',
                'condition'    => [
                    'enable_overlay' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'overlay_icon',
            [
                'label'        => __( 'Overlay Icon', 'huge-addons-for-elementor' ),
                'type'         => \Elementor\Controls_Manager::ICONS,
                'default'      => [
                    'value'   => 'fas fa-search',
                    'library' => 'fa-solid',
                ],
                'condition'    => [
                    'enable_overlay' => 'yes',
                    'show_icon'      => 'yes',
                ],
            ]
        );

        $this->add_control(
            'overlay_animation',
            [
                'label'        => __( 'Overlay Animation', 'huge-addons-for-elementor' ),
                'type'         => \Elementor\Controls_Manager::SELECT,
                'options'      => [
                    'fade-in'        => __( 'Fade In', 'huge-addons-for-elementor' ),
                    'fade-out'       => __( 'Fade Out', 'huge-addons-for-elementor' ),
                    'fade-in-up'     => __( 'Fade In Up', 'huge-addons-for-elementor' ),
                    'fade-in-down'   => __( 'Fade In Down', 'huge-addons-for-elementor' ),
                    'fade-in-left'   => __( 'Fade In Left', 'huge-addons-for-elementor' ),
                    'fade-in-right'  => __( 'Fade In Right', 'huge-addons-for-elementor' ),
                    'fade-out-up'    => __( 'Fade Out Up', 'huge-addons-for-elementor' ), 
                    'fade-out-down'  => __( 'Fade Out Down', 'huge-addons-for-elementor' ), 
                    'fade-out-left'  => __( 'Fade Out Left', 'huge-addons-for-elementor' ), 
                    'fade-out-right' => __( 'Fade Out Right', 'huge-addons-for-elementor' ), 
                    'slide-up'       => __( 'Slide Up', 'huge-addons-for-elementor' ),
                    'slide-down'     => __( 'Slide Down', 'huge-addons-for-elementor' ),
                    'slide-left'     => __( 'Slide Left', 'huge-addons-for-elementor' ),
                    'slide-right'    => __( 'Slide Right', 'huge-addons-for-elementor' ),
                    'zoom-in'        => __( 'Zoom In', 'huge-addons-for-elementor' ),
                    'zoom-out'       => __( 'Zoom Out', 'huge-addons-for-elementor' ),
                    'rotate'         => __( 'Rotate', 'huge-addons-for-elementor' ),
                    'flip'           => __( 'Flip', 'huge-addons-for-elementor' ),
                    'scale'          => __( 'Scale', 'huge-addons-for-elementor' ),
                    'wipe'           => __( 'Wipe', 'huge-addons-for-elementor' ),
                    'pulse'          => __( 'Pulse', 'huge-addons-for-elementor' ),
                    'none'           => __( 'None', 'huge-addons-for-elementor' ),
                ],
                'default'      => 'fade-in',
                'condition'    => [
                    'enable_overlay' => 'yes',
                ],
            ]
        );

        // Toggle for enabling fixed height
        $this->add_control(
            'enable_fixed_height',
            [
                'label' => __( 'Enable Fixed Image Height', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __( 'Yes', 'huge-addons-for-elementor' ),
                'label_off' => __( 'No', 'huge-addons-for-elementor' ),
                'return_value' => 'yes',
                'default' => 'no',
                'condition' => [
                    'enable_masonry!' => 'yes',
                ],
            ]
        );

        // Fixed Height Value
        $this->add_control(
            'fixed_height',
            [
                'label' => __( 'Image Height', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => [ 'px', 'vh' ],
                'range' => [
                    'px' => [ 'min' => 50, 'max' => 500 ],
                ],
                'default' => [ 'size' => 200, 'unit' => 'px' ],
                'condition' => [
                    'enable_fixed_height' => 'yes',
                    'enable_masonry!' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'show_caption',
            [
                'label' => __('Show Caption', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'huge-addons-for-elementor'),
                'label_off' => __('No', 'huge-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'no',
                'condition'    => [
                    'enable_overlay' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'enable_masonry',
            [
                'label' => __( 'Enable Masonry', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __( 'Yes', 'huge-addons-for-elementor' ),
                'label_off' => __( 'No', 'huge-addons-for-elementor' ),
                'return_value' => 'yes',
                'default' => '',
            ]
        );
        
        $this->add_control(
            'masonry_columns',
            [
                'label'       => __('Masonry Columns', 'huge-addons-for-elementor'),
                'type'        => \Elementor\Controls_Manager::SELECT,
                'description' => __('Set the number of columns for Masonry layout.', 'huge-addons-for-elementor'),
                'options'     => [
                    '1' => __('1 Column', 'huge-addons-for-elementor'),
                    '2' => __('2 Columns', 'huge-addons-for-elementor'),
                    '3' => __('3 Columns', 'huge-addons-for-elementor'),
                    '4' => __('4 Columns', 'huge-addons-for-elementor'),
                    '5' => __('5 Columns', 'huge-addons-for-elementor'),
                    '6' => __('6 Columns', 'huge-addons-for-elementor'),
                    '8' => __('8 Columns', 'huge-addons-for-elementor'),
                    '10' => __('10 Columns', 'huge-addons-for-elementor'),
                    '12' => __('12 Columns', 'huge-addons-for-elementor'),
                ],
                'default'     => '3',
                'condition'   => [
                    'enable_masonry' => 'yes',
                ],
            ]
        );
        
        $this->add_control(
            'masonry_gap',
            [
                'label' => __('Masonry Gap', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 5,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'size' => 10,
                    'unit' => 'px',
                ],
                'condition' => [
                    'enable_masonry' => 'yes',
                ],
            ]
        );
        

        $this->end_controls_section();

        /***************************************************************** style ************************************************************************** */

        $this->start_controls_section(
            'style_section',
            [
                'label' => __( 'Gallery Style', 'huge-addons-for-elementor' ),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        // Add a Separator for Better UI
        $this->add_control(
            'images_heading',
                [
                    'label' => __( 'Images', 'huge-addons-for-elementor' ),
                    'type' => \Elementor\Controls_Manager::HEADING,
                ]
        );
        
        // Image Border Radius
        $this->add_responsive_control(
            'image_border_radius',
            [
                'label' => __( 'Border Radius', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em', 'rem'],
                'range' => [
                    'px'  => ['min' => 0, 'max' => 50],
                    '%'   => ['min' => 0, 'max' => 100],
                    'em'  => ['min' => 0, 'max' => 5, 'step' => 0.1],
                    'rem' => ['min' => 0, 'max' => 5, 'step' => 0.1],
                ],
                'selectors' => [
                    '{{WRAPPER}} .huge-gallery-item img' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'image_border',
                'label' => __( 'Image Border', 'huge-addons-for-elementor' ),
                'selector' => '{{WRAPPER}} .huge-gallery-item img',
            ]
        );
        
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'image_box_shadow',
                'label' => __( 'Box Shadow', 'huge-addons-for-elementor' ),
                'selector' => '{{WRAPPER}} .huge-gallery-item img',
            ]
        );

        // Add a Separator for Better UI
        $this->add_control(
            'overlay_heading',
                [
                    'label' => __( 'Overlay', 'huge-addons-for-elementor' ),
                    'type' => \Elementor\Controls_Manager::HEADING,
                    'condition' => [
                    'enable_overlay' => 'yes',
                    ],
                ]
        );

        // Overlay Background Control
        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'overlay_background',
                'label' => __( 'Overlay Background', 'huge-addons-for-elementor' ),
                'types' => [ 'classic', 'gradient' ], 
                'selector' => '{{WRAPPER}} .image-overlay',
                'condition' => [
                    'enable_overlay' => 'yes',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => 'rgba(0, 0, 0, 0.5)',
                    ],
                ],
            ]
        );

        // Add Border Control for Overlay
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'overlay_border',
                'label' => __( 'Overlay Border', 'huge-addons-for-elementor' ),
                'selector' => '{{WRAPPER}} .image-overlay',
                'condition' => [
                    'enable_overlay' => 'yes',
                ],
            ]
        );
        // Image Border Radius
        $this->add_responsive_control(
            'overlay_border_radius',
            [
                'label' => __( 'Overlay Border Radius', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em', 'rem'],
                'range' => [
                    'px'  => ['min' => 0, 'max' => 50],
                    '%'   => ['min' => 0, 'max' => 100],
                    'em'  => ['min' => 0, 'max' => 5, 'step' => 0.1],
                    'rem' => ['min' => 0, 'max' => 5, 'step' => 0.1],
                ],
                'selectors' => [
                    '{{WRAPPER}} .image-overlay' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                ],
            ]
        );

        // Add Box Shadow Control for Overlay
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'overlay_box_shadow',
                'label' => __( 'Overlay Box Shadow', 'huge-addons-for-elementor' ),
                'selector' => '{{WRAPPER}} .image-overlay',
                'condition' => [
                    'enable_overlay' => 'yes',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                ],
            ]
        );

        // Add a Separator for Better UI
        $this->add_control(
            'icon_heading',
                [
                    'label' => __( 'Icon', 'huge-addons-for-elementor' ),
                    'type' => \Elementor\Controls_Manager::HEADING,
                    'condition' => [
                        'enable_overlay' => 'yes',
                        'show_icon' => 'yes',
                    ],
                ]
        );

        $this->add_control(
            'icon_bg_color',
            [
                'label' => __('Icon Background Color', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .overlay-icon' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_icon' => 'yes',
                ],
            ]
        );        
        
        // Icon Color
        $this->add_control(
            'icon_color',
            [
                'label' => __( 'Icon Color', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .overlay-icon' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_icon' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Icon Size', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => ['min' => 10, 'max' => 100],
                    'em' => ['min' => 0.5, 'max' => 5],
                    'rem' => ['min' => 0.5, 'max' => 5],
                ],
                'default' => [
                    'size' => 32,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .overlay-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}; fill: currentColor;',
                    '{{WRAPPER}} .overlay-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .overlay-icon img' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_icon' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'icon_border',
                'label' => __('Border', 'huge-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .overlay-icon',
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_icon' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_border_radius',
            [
                'label' => __('Border Radius', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em', 'rem'],
                'range' => [
                    'px' => ['min' => 0, 'max' => 50],
                    '%'  => ['min' => 0, 'max' => 100],
                    'em' => ['min' => 0, 'max' => 5],  
                    'rem' => ['min' => 0, 'max' => 5],
                ],
                'selectors' => [
                    '{{WRAPPER}} .overlay-icon' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_icon' => 'yes',
                ],
            ]
        );
        $this->add_responsive_control(
            'icon_margin',
            [
                'label' => __('Icon Margin', 'huge-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .overlay-icon svg, {{WRAPPER}} .overlay-icon i, {{WRAPPER}} .overlay-icon img' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_icon' => 'yes',
                ],
            ]
        );
        
        
         // Add a Separator for Better UI
         $this->add_control(
            'caption_heading',
                [
                    'label' => __( 'Caption', 'huge-addons-for-elementor' ),
                    'type' => \Elementor\Controls_Manager::HEADING,
                    'condition' => [
                        'enable_overlay' => 'yes',
                        'show_caption' => 'yes',
                    ],
                ]
        );

        // Caption Style
        $this->add_control(
            'caption_color',
            [
                'label' => __( 'Caption Color', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .image-caption' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_caption' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'caption_typography',
                'label' => __( 'Caption Typography', 'huge-addons-for-elementor' ),
                'selector' => '{{WRAPPER}} .image-caption',
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_caption' => 'yes',
                ],
            ]
        );


        $this->add_responsive_control(
            'caption_margin',
            [
                'label' => __( 'Caption Margin', 'huge-addons-for-elementor' ),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%', 'em', 'rem' ],
                'default' => [
                    'top'    => 10,
                    'right'  => 10,
                    'bottom' => 10,
                    'left'   => 10,
                    'unit'   => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .image-caption' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'enable_overlay' => 'yes',
                    'show_caption' => 'yes',
                ],
            ]
        );
        
        
        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        if (empty($settings['images'])) {
            return;
        }

        $enable_masonry = $settings['enable_masonry'] === 'yes';
        $enable_lightbox = $settings['enable_lightbox'] === 'yes';
        $enable_lightbox_gallery = $settings['enable_lightbox_gallery'] === 'yes';
        $enable_overlay = $settings['enable_overlay'] === 'yes';
        $show_icon = $settings['show_icon'] === 'yes';
        $overlay_icon = $settings['overlay_icon'];
        $overlay_animation = $settings['overlay_animation'];
        $show_caption = $settings['show_caption'] === 'yes';
        $enable_fixed_height = $settings['enable_fixed_height'] === 'yes';
        $fixed_height = ($enable_fixed_height && isset($settings['fixed_height']['size'], $settings['fixed_height']['unit'])) 
            ? esc_attr($settings['fixed_height']['size'] . $settings['fixed_height']['unit']) 
            : 'auto';

        if ($enable_masonry) {
            $fixed_height = 'auto';
        }


        // Masonry-specific settings
        $masonry_columns = $enable_masonry ? $settings['masonry_columns'] : '3';
        $masonry_gap = $enable_masonry ? $settings['masonry_gap']['size'] : 10; 
        $masonry_gap_unit = $enable_masonry ? $settings['masonry_gap']['unit'] : 'px'; 

        // Calculate the margin (half of the gap)
        $masonry_margin = $masonry_gap / 2;

        // Add a class for masonry layout if enabled
        $gallery_class = $enable_masonry ? 'advanced-image-gallery huge-masonry' : 'advanced-image-gallery';

        // Inline styles for masonry layout
        $masonry_item_style = $enable_masonry ? sprintf(
            'style="width: calc(100%% / %d - %s); margin: %s;"',
            $masonry_columns,
            $masonry_gap . $masonry_gap_unit, 
            $masonry_margin . $masonry_gap_unit 
        ) : '';
        ?>

        <div class="huge-image-gallery <?php echo esc_attr( $gallery_class ); ?>">
            <?php foreach ($settings['images'] as $image): 
                $image_url = esc_url($image['url']);
                $image_alt = esc_attr(get_post_meta($image['id'], '_wp_attachment_image_alt', true));
                $image_caption = wp_get_attachment_caption($image['id']);
                $lightbox_attributes = $enable_lightbox_gallery ? ' data-elementor-lightbox-slideshow="gallery"' : '';
                ?>
                <div class="huge-gallery-item" <?php echo $masonry_item_style; ?> >
                    <?php if ( $enable_lightbox ): ?>
                        <a href="<?php echo esc_url( $image_url ); ?>" data-elementor-open-lightbox="yes" <?php echo esc_attr( $lightbox_attributes ); ?>>
                    <?php endif; ?>


                    <div class="gallery-image-wrapper">
                        <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_html( $image_alt); ?>" style="height: <?php echo esc_attr( $fixed_height ); ?>; object-fit: cover; width: 100%;">
                        
                        <?php if ($enable_overlay): ?>
                            <div class="image-overlay <?php echo esc_attr($overlay_animation); ?>">
                                <?php if ($show_icon && !empty($overlay_icon['value'])): ?>
                                    <div class="overlay-icon">
                                        <?php \Elementor\Icons_Manager::render_icon($overlay_icon, ['aria-hidden' => 'true']); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if ($show_caption && !empty($image_caption)): ?>
                                    <div class="image-caption"><?php echo esc_html($image_caption); ?></div>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <?php if ($enable_lightbox): ?>
                        </a>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>

    <?php
    }

}
