<?php
namespace HugeAddons\Includes\Extensions;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class THHA_Custom_CSS {
    private static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    public function __construct() {
        // Check if Elementor Pro is active
        if ( $this->is_elementor_pro_active() ) {
            return;
        }

        // Element controls (widgets/sections/columns) - High priority to add at the end
        add_action( 'elementor/element/after_section_end', [ $this, 'register_element_controls' ], 999, 3 );
        add_action( 'elementor/element/parse_css', [ $this, 'add_element_css' ], 10, 2 );

        // Page/document controls & CSS - High priority to add at the end
        add_action( 'elementor/document/after_section_end', [ $this, 'register_document_controls' ], 999, 3 );
        add_action( 'elementor/css-file/post/parse', [ $this, 'add_page_settings_css' ] );
    }

    /**
     * Check if Elementor Pro is active
     */
    private function is_elementor_pro_active() {
        return defined( 'ELEMENTOR_PRO_VERSION' );
    }

    /**
     * Register Custom CSS controls for elements (widgets/sections/columns)
     */
    public function register_element_controls( $element, $section_id, $args ) {
        // Only add in Advanced tab
        if ( \Elementor\Controls_Manager::TAB_ADVANCED !== $args['tab'] ) {
            return;
        }

        // Add only if not already added (prevents multiples)
        $controls = $element->get_controls();
        if ( isset( $controls['section_huge_custom_css'] ) ) {
            return;
        }

        // Skip if Pro's CSS section already exists
        if ( isset( $controls['section_custom_css_pro'] ) ) {
            return;
        }

        $element->start_controls_section(
            'section_huge_custom_css',
            [
                'label' => esc_html__( 'Huge Custom CSS', 'huge-addons' ),
                'tab' => \Elementor\Controls_Manager::TAB_ADVANCED,
            ]
        );

        $element->add_control(
            'huge_custom_css',
            [
                'label' => esc_html__( 'Huge Custom CSS', 'huge-addons' ),
                'type' => \Elementor\Controls_Manager::CODE,
                'language' => 'css',
                'render_type' => 'ui',
                'show_label' => false,
                'separator' => 'none',
                'description' => esc_html__( 'Add your own custom CSS here', 'huge-addons' ),
            ]
        );

        $element->add_control(
            'huge_custom_css_description',
            [
                'raw' => esc_html__( 'Use "selector" to target wrapper element. Examples:<br>selector {color: red;} // For main element<br>selector .child-element {margin: 10px;} // For child element<br>.my-class {text-align: center;} // Or use any custom selector', 'huge-addons' ),
                'type' => \Elementor\Controls_Manager::RAW_HTML,
                'content_classes' => 'elementor-descriptor',
            ]
        );

        $element->end_controls_section();
    }

    /**
     * Register Custom CSS controls for pages/documents
     */
    public function register_document_controls( $document, $section_id, $args ) {
        // Only add in Settings tab
        if ( \Elementor\Controls_Manager::TAB_SETTINGS !== $args['tab'] ) {
            return;
        }

        // Add only if not already added (prevents multiples)
        $controls = $document->get_controls();
        if ( isset( $controls['section_huge_page_custom_css'] ) ) {
            return;
        }

        // Skip if Pro's CSS section already exists
        if ( isset( $controls['section_custom_css_pro'] ) ) {
            return;
        }

        $document->start_controls_section(
            'section_huge_page_custom_css',
            [
                'label' => esc_html__( 'Huge Custom CSS', 'huge-addons' ),
                'tab' => \Elementor\Controls_Manager::TAB_SETTINGS,
            ]
        );

        $document->add_control(
            'huge_custom_css',
            [
                'label' => esc_html__( 'Huge Custom CSS', 'huge-addons' ),
                'type' => \Elementor\Controls_Manager::CODE,
                'language' => 'css',
                'render_type' => 'ui',
                'show_label' => false,
                'separator' => 'none',
                'description' => esc_html__( 'Add custom CSS for this page. Use page selectors as needed.', 'huge-addons' ),
            ]
        );

        $document->end_controls_section();
    }

    /**
     * Add Custom CSS to element
     */
    public function add_element_css( $post_css, $element ) {
        $element_settings = $element->get_settings();

        if ( empty( $element_settings['huge_custom_css'] ) ) {
            return;
        }

        $css = trim( $element_settings['huge_custom_css'] );

        if ( empty( $css ) ) {
            return;
        }

        // Use $element->get_unique_selector() 
        $css = str_replace( 'selector', $element->get_unique_selector(), $css );

        $post_css->get_stylesheet()->add_raw_css( $css );
    }

    /**
     * Add Page Settings CSS
     */
    public function add_page_settings_css( $post_css ) {
        $document = \Elementor\Plugin::$instance->documents->get( $post_css->get_post_id() );
        $custom_css = $document->get_settings( 'huge_custom_css' );

        if ( empty( $custom_css ) ) {
            return;
        }

        $custom_css = trim( $custom_css );

        if ( empty( $custom_css ) ) {
            return;
        }

        $post_css->get_stylesheet()->add_raw_css( $custom_css );
    }
}

// Initialize the extension
THHA_Custom_CSS::instance();