<?php

namespace HugeAddons\Classes;

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Abstract Class AbstractNoticeManager.
 *
 * Provides a base for creating and managing admin notices.
 */
abstract class AbstractNoticeManager {
    /**
     * @var null|static
     */
    protected static $instance = null;
    protected $key = 'thha_notice';


    /**
     * @var array
     */
    protected $notices = [];

    /**
     * Singleton instance.
     *
     * @return static
     */
    public static function instance() {
        if ( static::$instance === null ) {
            static::$instance = new static();
        }
        return static::$instance;
    }

    /**
     * AbstractNoticeManager constructor.
     */
    public function __construct() {
        $this->notices = get_option( $this->key, [] );
        $this->init();
    }

    /**
     * Initialize hooks.
     */
    public function init() {
        add_action( 'admin_notices', [ $this, 'render_notices' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
        add_action( 'wp_ajax_thha_dismiss_notice', [ $this, 'ajax_dismiss_notice' ] );
    }

    /**
     * Add a notice.
     *
     * @param array $args
     */
    public function add_notice( $args ) {
        $defaults = [
            'unique_id' => '',
            'message'   => '',
            'type'      => 'info',
            'buttons'   => [],
            'reminder'  => false,
        ];
        $args = wp_parse_args( $args, $defaults );
        
        if ( empty( $args['unique_id'] ) || empty( $args['message'] ) ) {
            return;
        }
        
        $notice_exist = get_option( 'thha_notice_' . $args['unique_id'], null );
        if (  $notice_exist === null ) {
            $this->notices[] = $args;
            update_option( 'thha_notice', $this->notices );
            update_option( 'thha_notice_' . $args['unique_id'], 1 );
        }else{
            return;
        }

    }

    /**
     * Render notices.
     */
    public function render_notices() {
        foreach ( $this->notices as $notice ) {
            $transient_key = 'thha_notice_' . $notice['unique_id'] ;
    
            $data = get_transient( $transient_key );

            if ( 0 === $data  ) {
                continue;
            }
    
            $classes = [ 'thha-admin-notice', 'notice', 'notice-' . esc_attr( $notice['type'] ) ];
            if ( $notice['reminder'] ) {
                $classes[] = 'is-dismissible';
            }
            ?>
            <div class="<?php echo esc_attr( implode( ' ', array_map( 'sanitize_html_class', $classes ) ) ); ?>"
                 data-unique-id="<?php echo esc_attr( $notice['unique_id'] ); ?>"
                 <?php if ( ! $notice['reminder'] ) echo 'data-autoclose="1"'; ?>
                 style="position:relative;">
                <div class="notice-wrapper">
                    <div class="notice-content">
                        <div class="notice-message">
                            <?php echo wp_kses_post( $notice['message'] ); ?>
                        </div>
                        <div class="notice-button">
                            <?php
                            if ( ! empty( $notice['buttons'] ) ) {
                                foreach ( $notice['buttons'] as $btn ) {
                                    $label  = esc_html( $btn['label'] ?? 'Button' );
                                    $url    = esc_url( $btn['url'] ?? '#' );
                                    $target = ! empty( $btn['target'] ) ? ' target="' . esc_attr( $btn['target'] ) . '"' : '';
                                    $class  = ! empty( $btn['class'] ) ? esc_attr( $btn['class'] ) : 'button';
                                    echo ' <a href="' . esc_url( $url ) . '" class="' . esc_attr( $class ) . '"' . $target . '>' . esc_html( $label ) . '</a>';
                                }
                            }
                            ?>
                        </div>
                    </div>
                    <?php if ( $notice['reminder'] ) : ?>
                        <button type="button" class="notice-dismiss thha-notice-dismiss" aria-label="Dismiss"></button>
                    <?php endif; ?>
                </div>
            </div>
            <?php

            if ( ! $notice['reminder'] ) {
                update_option( 'thha_notice_' . $notice['unique_id'], 0 );
            }
        }
    }

    /**
     * Handle AJAX request to dismiss a notice.
     */
    public function ajax_dismiss_notice() {
        check_ajax_referer( 'thha_dismiss_notice' );
        $id = sanitize_text_field( wp_unslash( $_POST['unique_id'] ?? '' ) );
        if ( $id ) {
            $min_months = 4;
            $max_months = 9;
            $min_seconds = $min_months * 30 * DAY_IN_SECONDS;
            $max_seconds = $max_months * 30 * DAY_IN_SECONDS;
            $random_expiration = mt_rand( $min_seconds, $max_seconds );

            $transient_name = 'thha_notice_' . $id;
            $data_to_cache = '0';

            $set_success = set_transient( $transient_name, $data_to_cache, $random_expiration );
            // update_option( 'thha_notice_' . $id, 0 );
        }
        wp_die();
    }

    /**
     * Enqueue scripts and styles for notices.
     */
    abstract public function enqueue_scripts();
}