(function(wp) {
    const { registerBlockType } = wp.blocks;
    const { InnerBlocks, useBlockProps, BlockControls } = wp.blockEditor;
    const { ToolbarGroup, ToolbarButton } = wp.components;
    const { __ } = wp.i18n;
    const { useDispatch, useSelect } = wp.data;
    
    // Register Article Block
    registerBlockType('html5-semantic/article', {
        title: __('Article', 'html5-semantic-elements'),
        description: __('HTML5 Article element for independent content', 'html5-semantic-elements'),
        category: 'design',
        icon: 'media-document',
        keywords: [
            __('article', 'html5-semantic-elements'),
            __('html5', 'html5-semantic-elements'),
            __('semantic', 'html5-semantic-elements')
        ],
        supports: {
            align: ['wide', 'full'],
            anchor: true,
            className: true,
        },
        transforms: {
            ungroup: function(attributes, innerBlocks) {
                return innerBlocks;
            }
        },
        
        edit: function(props) {
            const { clientId } = props;
            const { replaceBlocks } = useDispatch('core/block-editor');
            const innerBlocks = useSelect(function(select) {
                return select('core/block-editor').getBlocks(clientId);
            }, [clientId]);
            
            const blockProps = useBlockProps({
                className: 'html5-semantic-article-editor'
            });
            
            function removeArticle() {
                replaceBlocks(clientId, innerBlocks);
            }
            
            return wp.element.createElement(
                wp.element.Fragment,
                null,
                [
                    wp.element.createElement(
                        BlockControls,
                        { key: 'controls' },
                        wp.element.createElement(
                            ToolbarGroup,
                            null,
                            wp.element.createElement(
                                ToolbarButton,
                                {
                                    icon: 'remove',
                                    label: __('Remove Article', 'html5-semantic-elements'),
                                    onClick: removeArticle
                                }
                            )
                        )
                    ),
                    wp.element.createElement(
                        'div',
                        blockProps,
                        [
                            wp.element.createElement(
                                'div',
                                {
                                    key: 'label',
                                    className: 'html5-semantic-label'
                                },
                                __('Article', 'html5-semantic-elements')
                            ),
                            wp.element.createElement(
                                InnerBlocks,
                                {
                                    key: 'innerblocks',
                                    template: [
                                        ['core/heading', { level: 2, placeholder: __('Article heading...', 'html5-semantic-elements') }],
                                        ['core/paragraph', { placeholder: __('Article content...', 'html5-semantic-elements') }]
                                    ]
                                }
                            )
                        ]
                    )
                ]
            );
        },
        
        save: function(props) {
            const blockProps = useBlockProps.save();
            
            return wp.element.createElement(
                'article',
                blockProps,
                wp.element.createElement(InnerBlocks.Content)
            );
        }
    });
    
    // Register Aside Block
    registerBlockType('html5-semantic/aside', {
        title: __('Aside', 'html5-semantic-elements'),
        description: __('HTML5 Aside element for tangentially related content', 'html5-semantic-elements'),
        category: 'design',
        icon: 'align-right',
        keywords: [
            __('aside', 'html5-semantic-elements'),
            __('sidebar', 'html5-semantic-elements'),
            __('html5', 'html5-semantic-elements'),
            __('semantic', 'html5-semantic-elements')
        ],
        supports: {
            align: ['left', 'right', 'wide', 'full'],
            anchor: true,
            className: true,
        },
        transforms: {
            from: [
                {
                    type: 'block',
                    blocks: ['core/paragraph'],
                    transform: function(attributes) {
                        return wp.blocks.createBlock('html5-semantic/aside', {}, [
                            wp.blocks.createBlock('core/heading', { 
                                level: 3, 
                                content: __('Aside heading...', 'html5-semantic-elements')
                            }),
                            wp.blocks.createBlock('core/paragraph', attributes)
                        ]);
                    }
                },
                {
                    type: 'block',
                    blocks: ['core/heading'],
                    transform: function(attributes) {
                        return wp.blocks.createBlock('html5-semantic/aside', {}, [
                            wp.blocks.createBlock('core/heading', attributes),
                            wp.blocks.createBlock('core/paragraph', { 
                                content: __('Additional content...', 'html5-semantic-elements')
                            })
                        ]);
                    }
                },
                {
                    type: 'block',
                    blocks: ['core/list', 'core/quote', 'core/pullquote'],
                    transform: function(attributes, innerBlocks) {
                        // Get the block name to recreate it
                        var blockName = 'core/paragraph';
                        if (attributes && attributes.values) {
                            blockName = 'core/list';
                        }
                        
                        return wp.blocks.createBlock('html5-semantic/aside', {}, [
                            wp.blocks.createBlock('core/heading', { 
                                level: 3, 
                                content: __('Aside heading...', 'html5-semantic-elements')
                            }),
                            wp.blocks.createBlock(blockName, attributes, innerBlocks)
                        ]);
                    }
                },
                {
                    type: 'block',
                    blocks: ['core/group'],
                    transform: function(attributes, innerBlocks) {
                        return wp.blocks.createBlock('html5-semantic/aside', {}, innerBlocks);
                    }
                }
            ],
            to: [
                {
                    type: 'block',
                    blocks: ['core/group'],
                    transform: function(attributes, innerBlocks) {
                        return wp.blocks.createBlock('core/group', {}, innerBlocks);
                    }
                }
            ],
            ungroup: function(attributes, innerBlocks) {
                return innerBlocks;
            }
        },
        
        edit: function(props) {
            const { clientId } = props;
            const { replaceBlocks } = useDispatch('core/block-editor');
            const innerBlocks = useSelect(function(select) {
                return select('core/block-editor').getBlocks(clientId);
            }, [clientId]);
            
            const blockProps = useBlockProps({
                className: 'html5-semantic-aside-editor'
            });
            
            function removeAside() {
                replaceBlocks(clientId, innerBlocks);
            }
            
            return wp.element.createElement(
                wp.element.Fragment,
                null,
                [
                    wp.element.createElement(
                        BlockControls,
                        { key: 'controls' },
                        wp.element.createElement(
                            ToolbarGroup,
                            null,
                            wp.element.createElement(
                                ToolbarButton,
                                {
                                    icon: 'remove',
                                    label: __('Remove Aside', 'html5-semantic-elements'),
                                    onClick: removeAside
                                }
                            )
                        )
                    ),
                    wp.element.createElement(
                        'div',
                        blockProps,
                        [
                            wp.element.createElement(
                                'div',
                                {
                                    key: 'label',
                                    className: 'html5-semantic-label'
                                },
                                __('Aside', 'html5-semantic-elements')
                            ),
                            wp.element.createElement(
                                InnerBlocks,
                                {
                                    key: 'innerblocks',
                                    template: [
                                        ['core/heading', { level: 3, placeholder: __('Aside heading...', 'html5-semantic-elements') }],
                                        ['core/paragraph', { placeholder: __('Additional content...', 'html5-semantic-elements') }]
                                    ]
                                }
                            )
                        ]
                    )
                ]
            );
        },
        
        save: function(props) {
            const blockProps = useBlockProps.save();
            
            return wp.element.createElement(
                'aside',
                blockProps,
                wp.element.createElement(InnerBlocks.Content)
            );
        }
    });
    
})(window.wp);