<?php

/**
 * Container for recaptcha v2.
 *
 * @version 1.0.0
 */

namespace Hizzle\Recaptcha;

defined( 'ABSPATH' ) || exit;

/**
 * Recaptcha v2.
 */
class Recaptcha_V2 extends Handler {

	/**
	 * The is valid.
	 *
	 * @var bool|null
	 */
	protected $is_valid = null;

	/**
	 * Returns the handler's slug.
	 *
	 * @return string
	 */
	public function get_slug() {
		return 'recaptcha_v2';
	}

	/**
	 * Returns the handler's settings.
	 *
	 * @return array
	 */
	public function get_settings() {
		$handlers = array(
			'recaptcha_v2',
			'recaptcha_v2_invisible',
			'recaptcha_v3',
		);

		return array(
			'site_key'   => array(
				'type'     => 'text',
				'label'    => __( 'Site Key', 'hizzle-recaptcha' ),
				'default'  => '',
				'handlers' => $handlers,
				'desc'     => sprintf(
					'<strong>%s</strong> %s',
					__( 'Important: Ensure that the site key matches the captcha type you have selected and the domain is correct.', 'hizzle-recaptcha' ),
					sprintf(
						'<a href="%s" target="_blank">%s</a>',
						'https://www.google.com/recaptcha/admin/create',
						__( 'Create a new site key.', 'hizzle-recaptcha' )
					)
				),
			),
			'secret_key' => array(
				'type'     => 'text',
				'label'    => __( 'Secret Key', 'hizzle-recaptcha' ),
				'default'  => '',
				'handlers' => $handlers,
			),
			'load_from'  => array(
				'type'     => 'select',
				'label'    => __( 'Load from', 'hizzle-recaptcha' ),
				'options'  => array(
					'google'    => 'google.com',
					'recaptcha' => 'recaptcha.net',
				),
				'default'  => 'recaptcha',
				'desc'     => __( 'Use recaptcha.net if Google is blocked in your country.', 'hizzle-recaptcha' ),
				'handlers' => $handlers,
			),
		);
	}

	/**
	 * Checks if the reCAPTCHA was validated.
	 *
	 * @since 1.0.0
	 * @return true|WP_Error
	 */
	public function is_valid() {
		$posted = wp_unslash( $_POST ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
		return $this->check_token( isset( $posted['g-recaptcha-response'] ) ? $posted['g-recaptcha-response'] : '' );
	}

	/**
	 * Checks if the reCAPTCHA was validated.
	 *
	 * @since 1.0.0
	 * @param string $token The token to validate.
	 * @return true|WP_Error
	 */
	protected function check_token( $token ) {

		// New token.
		if ( empty( $token ) ) {
			return new \WP_Error( 'token_not_found', __( "Please verify that you're not a robot.", 'hizzle-recaptcha' ) );
		}

		$result = wp_remote_post(
			'https://www.google.com/recaptcha/api/siteverify',
			array(
				'body' => array(
					'secret'   => hizzle_recaptcha_get_option( 'secret_key' ),
					'response' => $token,
				),
			)
		);

		// Site not reachable.
		if ( is_wp_error( $result ) ) {
			return true;
		}

		$result = json_decode( wp_remote_retrieve_body( $result ), true );

		if ( empty( $result['success'] ) && ! in_array( 'missing-input-secret', $result['error-codes'], true ) && ! in_array( 'invalid-input-secret', $result['error-codes'], true ) ) {
			return new \WP_Error( 'invalid_token', __( "Unable to verify that you're not a robot. Please refresh the page and try again.", 'hizzle-recaptcha' ) );
		}

		return true;
	}

	/**
	 * Checks if the handler is set up.
	 *
	 * @return bool
	 */
	public function can_show() {
		$site_key   = hizzle_recaptcha_get_option( 'site_key' );
		$secret_key = hizzle_recaptcha_get_option( 'secret_key' );

		if ( empty( $site_key ) || empty( $secret_key ) ) {
			return false;
		}

		return parent::can_show();
	}

	/**
	 * Displays the checkbox.
	 *
	 * @since 1.0.0
	 */
	public function display() {

		$attributes = array(
			'class'        => 'g-recaptcha hizzle-recaptcha',
			'style'        => 'max-width: 100%; overflow: hidden; margin-top: 10px; margin-bottom: 10px;',
			'data-sitekey' => hizzle_recaptcha_get_option( 'site_key' ),
		);

		echo '<div';

		foreach ( apply_filters( 'hizzle_recaptcha_attributes', $attributes ) as $key => $value ) {
			printf(
				' %s="%s"',
				esc_attr( $key ),
				esc_attr( $value )
			);
		}

		echo '></div>';
	}

	/**
	 * Loads the recaptcha code if we're displaying it on this page.
	 *
	 * @since 1.0.0
	 */
	public function maybe_add_scripts() {
		if ( $this->load_scripts && $this->can_show() ) {

			if ( 'recaptcha' === hizzle_recaptcha_get_option( 'load_from', 'recaptcha' ) ) {
				$url = 'https://www.recaptcha.net/recaptcha/api.js';
			} else {
				$url = 'https://www.google.com/recaptcha/api.js';
			}

			$url = apply_filters(
				'hizzle_recaptcha_api_url',
				add_query_arg(
					array(
						'render' => 'explicit',
						'onload' => 'hizzleRecaptchaOnload',
					),
					$url
				)
			);

			wp_enqueue_script( 'recaptcha', $url, array(), null, true ); // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion

			ob_start();
			?>
			<script>

				var hizzleRecaptchaLoop = function( cb ) {
					var elements = document.getElementsByClassName("hizzle-recaptcha");
					for (var i = 0; i < elements.length; i++) {
						cb( elements[i], i );
					}
				}

				var hizzleRecaptchaOnload = function() {
					hizzleRecaptchaLoop( function( element, index ) {
						if (element.hasAttribute("data-hizzle-widget-id")) {
							return;
						}

						element.setAttribute("data-hizzle-widget-id", index);

						grecaptcha.render(
							element,
							{
								size: 'normal',
								theme: 'light',
							},
							true
						);
					} );
				};
			</script>
			<?php
			$script = ob_get_clean();
			$script = str_replace( array( '<script>', '</script>' ), '', $script );
			wp_add_inline_script( 'recaptcha', $script, 'before' );
		}
	}
}
