<?php

/**
 * Main plugin class.
 *
 * @version 1.0.0
 */

namespace Hizzle\Recaptcha;

defined( 'ABSPATH' ) || exit;

/**
 * Main plugin class.
 */
class Main {

	/**
	 * The current handler.
	 *
	 * @var Handler|null
	 */
	public $handler = null;

	/**
	 * Integrations instance.
	 *
	 * @var Integrations\Main|null
	 */
	public $integrations = null;

	/**
	 * Admin instance.
	 *
	 * @var Admin|null
	 */
	public $admin = null;

	/**
	 * The single instance of the class.
	 *
	 * @var Main|null
	 */
	private static $instance = null;

	/**
	 * Main Instance.
	 *
	 * Ensures only one instance is loaded or can be loaded.
	 *
	 * @return Main
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'init', array( $this, 'init' ), 0 );
	}

	/**
	 * Plugins loaded.
	 */
	public function init() {
		do_action( 'hizzle_recaptcha_init' );

		$handlers = apply_filters(
			'hizzle_recaptcha_handler_classes',
			array(
				'recaptcha_v2' => __NAMESPACE__ . '\Recaptcha_V2',
			)
		);

		if ( is_admin() ) {
			foreach ( $handlers as $class ) {
				new $class();
			}
		} else {
			$selected = hizzle_recaptcha_get_option( 'handler', 'recaptcha_v2' );

			if ( empty( $selected ) ) {
				$selected = 'recaptcha_v2';
			}

			if ( isset( $handlers[ $selected ] ) ) {
				new $handlers[ $selected ]();
			}
		}

		$this->handler = apply_filters( 'hizzle_recaptcha_handler', null );

		// Only initialize integrations if the handler is active.
		if ( $this->handler instanceof Handler && $this->handler->can_show() ) {
			$this->integrations = new Integrations\Main();
		}

		if ( is_admin() ) {
			$this->admin = new Admin();
		}
	}

	/**
	 * Encrypts a text string.
	 *
	 * @param string $plaintext The plain text string to encrypt.
	 * @return string
	 */
	public static function encrypt( $plaintext ) {
		$cipher_algo = 'AES-128-CBC';
		// Generate a random IV for each encryption
		$iv         = openssl_random_pseudo_bytes( openssl_cipher_iv_length( $cipher_algo ) );
		$passphrase = wp_salt( 'secure_auth' );

		// Encrypt then encode
		$encrypted = openssl_encrypt( $plaintext, $cipher_algo, $passphrase, OPENSSL_RAW_DATA, $iv );

		// Combine IV and encrypted data
		$encoded = base64_encode( $iv . $encrypted ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode

		// Make URL safe
		return strtr( $encoded, '+/=', '._-' );
	}

	/**
	 * Decrypts a text string.
	 *
	 * @param string $encoded The string to decode.
	 * @return string
	 */
	public static function decrypt( $encoded ) {
		if ( empty( $encoded ) ) {
			return '';
		}

		// Decode
		$decoded = base64_decode( strtr( $encoded, '._-', '+/=' ) ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode

		if ( empty( $decoded ) ) {
			return '';
		}

		$cipher_algo = 'AES-128-CBC';
		$iv_length   = openssl_cipher_iv_length( $cipher_algo );

		// Extract IV and ciphertext
		$iv         = substr( $decoded, 0, $iv_length );
		$ciphertext = substr( $decoded, $iv_length );

		$passphrase = wp_salt( 'secure_auth' );

		$result = openssl_decrypt( $ciphertext, $cipher_algo, $passphrase, OPENSSL_RAW_DATA, $iv );
		return false === $result ? '' : $result;
	}
}
