<?php

/**
 * Container for a spam handler.
 *
 * @version 1.0.0
 */

namespace Hizzle\Recaptcha;

defined( 'ABSPATH' ) || exit;

/**
 * Handler.
 */
abstract class Handler {

	/**
	 * Whether to load the scripts.
	 *
	 * @var bool
	 */
	protected $load_scripts = false;

	/**
	 * Returns the handler's slug.
	 *
	 * @return string
	 */
	abstract public function get_slug();

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_filter( 'hizzle_recaptcha_settings', array( $this, 'register_settings' ) );
		add_filter( 'hizzle_recaptcha_has_handler_' . $this->get_slug(), '__return_true' );

		if ( $this->is_active() && $this->can_show() ) {
			$this->init();
		}
	}

	/**
	 * Initializes the handler.
	 */
	public function init() {
		do_action( 'hizzle_recaptcha_init_' . $this->get_slug() );
		add_filter( 'hizzle_recaptcha_handler', array( $this, 'get_handler' ) );

		if ( is_callable( array( $this, 'maybe_add_scripts' ) ) ) {
			add_action( 'wp_footer', array( $this, 'maybe_add_scripts' ), 11 );
			add_action( 'login_footer', array( $this, 'maybe_add_scripts' ), 11 );
		}
	}

	/**
	 * Returns the handler.
	 *
	 * @return Handler
	 */
	public function get_handler() {
		return $this;
	}

	/**
	 * Registers the handler.
	 *
	 * @param array $handlers The registered handlers.
	 * @return array
	 */
	public function register_handler( $handlers ) {
		$handlers[ $this->get_slug() ] = $this->get_slug();
		return $handlers;
	}

	/**
	 * Checks if this is the active handler.
	 *
	 * @return bool
	 */
	public function is_active() {
		$handler = hizzle_recaptcha_get_option( 'handler', 'recaptcha_v2' );

		if ( $handler === $this->get_slug() ) {
			return true;
		}

		return empty( $handler ) ? 'recaptcha_v2' === $this->get_slug() : false;
	}

	/**
	 * Checks if the handler is set up.
	 *
	 * @return bool
	 */
	public function can_show() {
		$hide_logged = hizzle_recaptcha_get_option( 'hide_logged_in', false );
		$should_show = 0 === get_current_user_id() || ! $hide_logged;

		return apply_filters( 'hizzle_recaptcha_show_captcha', $should_show, $this );
	}

	/**
	 * Returns the handler's settings.
	 *
	 * @return array
	 */
	abstract public function get_settings();

	/**
	 * Returns the handler's settings.
	 *
	 * @return array
	 */
	public function register_settings( $settings ) {
		return array_merge( $this->get_settings(), $settings );
	}

	/**
	 * Checks if the captcha was validated.
	 *
	 * @since 1.0.0
	 * @return true|WP_Error
	 */
	abstract public function is_valid();

	/**
	 * Displays the checkbox.
	 *
	 * @since 1.0.0
	 */
	public function render() {
		$this->load_scripts = true;
		$this->display();
	}

	/**
	 * Displays the checkbox.
	 *
	 * @since 1.0.0
	 */
	abstract public function display();
}
