/*!
 * File: quote-chart.js
 * Description: Manages quote card chart behavior and historical data formatting.
 * Author: Ideal Managed Solutions, LLC
 * License: GPLv2 or later
 * Plugin: Historical Charts for Goldback
 * Version: 1.0.0
 */
document.addEventListener('DOMContentLoaded', function () {
    // Grab every chart canvas that has our JSON data
    const canvases = document.querySelectorAll('canvas[data-records]');
    canvases.forEach(canvas => {
        const allData = JSON.parse(canvas.dataset.records);
        let chart;

        // Figure out which rate this is from the canvas ID
        const id = canvas.id.toLowerCase();
        const isUpmaRate = id.includes('upma');
        const isGoldSpot = id.includes('gold');
        const isSilverSpot = id.includes('silver');

        // Build a human label
        let rateLabel = 'Goldback\u00AE Exchange Rate';
        if (isUpmaRate) rateLabel = 'UPMA Exchange Rate';
        if (isGoldSpot) rateLabel = 'UPMA Gold Spot';
        if (isSilverSpot) rateLabel = 'UPMA Silver Spot';

        // Find the parent container (id starts with gbcharts-)
        const container = canvas.closest('div[id^="gbcharts-"]');
        if (!container) return;

        // Now grab any buttons that switch the day-range
        const buttons = container.querySelectorAll('button[data-days]');

        function renderChart(days) {
            // Filter records newer than X days ago
            const cutoff = new Date();
            cutoff.setDate(cutoff.getDate() - days);
            const filtered = allData.filter(d => new Date(d.date) >= cutoff);

            // Destroy old instance
            const ctx = canvas.getContext('2d');
            if (chart) chart.destroy();

            // Build new Chart.js line chart
            chart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: filtered.map(d => d.date),
                    datasets: [{
                        label: rateLabel,
                        data: filtered.map(d => d.value),  // <-- use value field
                        fill: true,
                        backgroundColor: 'rgba(198, 169, 77, 0.1)',
                        borderColor: '#C6A94D',
                        tension: 0.3
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: { ticks: { maxTicksLimit: 5 } },
                        y: { beginAtZero: false }
                    },
                    plugins: { legend: { display: false } }
                }
            });
        }

        // Wire up the buttons
        buttons.forEach(btn => {
            btn.addEventListener('click', () => {
                buttons.forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
                renderChart(parseInt(btn.dataset.days, 10));
            });
        });

        // Kick off the default 7-day view
        renderChart(7);
        const defaultBtn = container.querySelector('button[data-days="7"]');
        if (defaultBtn) defaultBtn.classList.add('active');
    });
});
