﻿/*!
 * File: chart-handler.js
 * Description: Handles rendering and configuration of Chart.js charts for the Historical Charts for Goldback plugin.
 * Author: Ideal Managed Solutions, LLC
 * License: GPLv2 or later
 * Plugin: Historical Charts for Goldback
 * Version: 1.0.0
 */
document.addEventListener('DOMContentLoaded', () => {
    // Find ALL of our chart containers
    document.querySelectorAll('.gbcharts-graph-chart-container').forEach(container => {
        // 1) Grab rate & records for *this* container
        const rateKey = (container.dataset.rate || 'GB').toUpperCase();
        const canvas = container.querySelector('.gbcharts-price-chart');
        const rawData = canvas.dataset.records || '[]';
        const allData = JSON.parse(rawData);
        const ctx = canvas.getContext('2d');

        // 2) Label map
        const labelMap = {
            'GB': 'Goldback\u00AE Exchange Rate',
            'UPMA': 'UPMA Exchange Rate',
            'GOLD': 'UPMA Gold Spot',
            'SILVER': 'UPMA Silver Spot'
        };
        const rateLabel = labelMap[rateKey] || labelMap['GB'];

        // 3) Render function
        let chart;
        function renderChart(data) {
            if (chart) chart.destroy();

            chart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: data.map(d => d.date),
                    datasets: [{
                        label: rateLabel,
                        data: data.map(d => d.value),
                        fill: true,
                        backgroundColor: 'rgba(198, 169, 77, 0.1)',
                        borderColor: '#C6A94D',
                        tension: 0.3,
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    scales: {
                        x: { display: true, title: { display: true, text: 'Date' } },
                        y: { display: true, title: { display: true, text: 'Value (USD)' } }
                    },
                    plugins: {
                        legend: { position: 'top' }
                    }
                }
            });
        }


        // 4) Wire up this container’s buttons
        const buttons = container.querySelectorAll('.filter-buttons button');
        buttons.forEach(btn => {
            btn.addEventListener('click', () => {
                const now = new Date();
                let cutoff;
                switch (btn.dataset.range) {
                    case '1W': cutoff = new Date(now.setDate(now.getDate() - 7)); break;
                    case '1M': cutoff = new Date(now.setMonth(now.getMonth() - 1)); break;
                    case '6M': cutoff = new Date(now.setMonth(now.getMonth() - 6)); break;
                    case '1Y': cutoff = new Date(now.setFullYear(now.getFullYear() - 1)); break;
                    case '5Y': cutoff = new Date(now.setFullYear(now.getFullYear() - 5)); break;
                    default: renderChart(allData); activate(btn); return;
                }
                renderChart(allData.filter(d => new Date(d.date) >= cutoff));
                activate(btn);
            });
        });
        function activate(activeBtn) {
            buttons.forEach(b => b.classList.remove('active'));
            activeBtn.classList.add('active');
        }

        // 5) Initial draw (ALL)
        const defaultBtn = container.querySelector('.filter-buttons button[data-range="1M"]');
        if (defaultBtn) {
            // trigger the same logic as a click
            defaultBtn.click();
        } else {
            // fallback if no 1M button exists
            renderChart(allData);
        }
    });
});
