<?php
/*
Plugin Name: Historical Charts for Goldback
Plugin URI: https://idealmanagedsolutions.com/plugins/goldback-charts/
Description: Display Goldback quote cards, quote cards with charts, and historical price data as a chart with time-based filters from a remote API.
Version: 1.2.1
Author: Ideal Managed Solutions, LLC
Author URI: https://idealmanagedsolutions.com
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
Text Domain: historical-charts-for-goldback

Copyright 2025 Ideal Managed Solutions, LLC (email: info@idealmsp.com)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

function gbcharts_enqueue_scripts() {
    // Chart.js v4.4.9 - https://github.com/chartjs/Chart.js
    wp_enqueue_script('chartjs', plugin_dir_url(__FILE__) . 'js/chart.js', [], null, true);
    // Luxon (Time Adapter Dependency) Luxon v3.6.1 - https://github.com/moment/luxon
    wp_enqueue_script('luxon', plugin_dir_url(__FILE__) . 'js/luxon.min.js', [], null, true);
    // // Chart.js Luxon Time Adapter - https://github.com/chartjs/chartjs-adapter-luxon
    wp_enqueue_script('chartjs-luxon-adapter', plugin_dir_url(__FILE__) .'js/chartjs-adapter-luxon.umd.min.js', ['chartjs', 'luxon'], null, true);
	// DataTables v2.3.1 - https://datatables.net
    wp_enqueue_script('datatables-js', plugin_dir_url(__FILE__) . 'js/jquery.dataTables.min.js', [ 'jquery' ], null, true);
    // Custom chart handler plugin script
	wp_enqueue_script('chart-handler', plugin_dir_url(__FILE__) . 'js/chart-handler.js', [ 'chartjs', 'chartjs-luxon-adapter' ], '1.0', true);
    //Custom quote combo chart plugin script
    wp_enqueue_style('quote-chart', plugin_dir_url(__FILE__) . 'css/quote-chart.css');
	// Custom quote card standalone plugin script
    wp_enqueue_script('quote-chart-js', plugin_dir_url(__FILE__) . 'js/quote-chart.js', ['chartjs'], time(), true);
	// Custom quote card plugin styling css
    wp_enqueue_style('quote-card', plugin_dir_url(__FILE__) . 'css/quote-card.css');
	// Standalone chart plugin styling css
    wp_enqueue_style('gbcharts-style', plugin_dir_url(__FILE__) . 'css/chart-style.css');
	// Datatables plugin styling css
	wp_enqueue_style('gbcharts-datatable-style', plugin_dir_url(__FILE__).'css/datatable-style.css');
	// jquery datatables styling
	wp_enqueue_style('datatables-css', plugin_dir_url(__FILE__) . 'css/jquery.dataTables.min.css');
	
	// Historical data tables plugin script
    wp_enqueue_script('gbcharts-data-table-js', plugin_dir_url(__FILE__) . 'js/gbcharts-data-table.js', [ 'jquery', 'datatables-js' ], '1.0', true);
}
add_action('wp_enqueue_scripts', 'gbcharts_enqueue_scripts');

function gbcharts_graph_shortcode( $atts ) {
    // Pull & normalize rate
    $atts = shortcode_atts( [ 'rate' => 'GB' ], $atts, 'gbcharts_graph_chart' );
    $rate = strtoupper( $atts['rate'] );

    // Fetch & parse XML in PHP
    $xml_url  = 'https://services.idealmsp.com/IMSPlugins/goldback-info/goldback_data.xml';
    $response = wp_remote_get( $xml_url );
    if ( is_wp_error( $response ) ) {
        return '<p>Error loading data.</p>';
    }
    $xml_string = wp_remote_retrieve_body( $response );

	// Remove BOM if present
	$xml_string = preg_replace('/^\xEF\xBB\xBF/', '', $xml_string);

	// Parse XML
	$xml = simplexml_load_string( $xml_string );
    if ( ! $xml ) {
        return '<p>Invalid data.</p>';
    }

    // Pick field by rate and build records
    $map = [
      'GB'     => 'GBER',
      'UPMA'   => 'UPMAER',
      'GOLD'   => 'UPMAAU',
      'SILVER' => 'UPMAAG',
    ];
    $field   = $map[ $rate ] ?? $map['GB'];
    $records = [];
    foreach ( $xml->Record as $r ) {
        $v = floatval( $r->{$field} );
        if ( $v > 0 ) {
            $records[] = [
                'date'  => (string)$r->Date,
                'value' => $v,
            ];
        }
    }

    // Emit a container with a CLASS and embed the JSON
    ob_start(); ?>
    <div class="gbcharts-graph-chart-container" data-rate="<?php echo esc_attr( $rate ); ?>">
      <canvas class="gbcharts-price-chart"
              data-records='<?php echo wp_json_encode( $records ); ?>'
              style="width:100%;height:550px;">
      </canvas>
      <div class="filter-buttons">
        <button data-range="1W">1W</button>
        <button data-range="1M">1M</button>
        <button data-range="6M">6M</button>
        <button data-range="1Y">1Y</button>
        <button data-range="5Y">5Y</button>
        <button data-range="ALL">ALL</button>
      </div>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode( 'gbcharts_graph_chart', 'gbcharts_graph_shortcode' );


//quote cards
function gbcharts_quote_card_shortcode( $atts ) {
    // Pull & normalize rate attribute
    $atts = shortcode_atts( [
        'rate' => 'GB',
    ], $atts, 'gbcharts_quote_card' );
    $rate = strtoupper( $atts['rate'] );

    // Map rate → XML field & human title
    $fieldMap = [
        'GB'     => 'GBER',
        'UPMA'   => 'UPMAER',
        'GOLD'   => 'UPMAAU',
        'SILVER' => 'UPMAAG',
    ];
    $titleMap = [
        'GB'     => 'Goldback Exchange Rate',
        'UPMA'   => 'UPMA Exchange Rate',
        'GOLD'   => 'UPMA Gold Spot',
        'SILVER' => 'UPMA Silver Spot',
    ];
    $field = $fieldMap[ $rate ]   ?? $fieldMap['GB'];
    $title = $titleMap[ $rate ]   ?? $titleMap['GB'];

    // Fetch & parse XML
    $xml_url  = 'https://services.idealmsp.com/IMSPlugins/goldback-info/goldback_data.xml';
    $response = wp_remote_get( $xml_url );
    if ( is_wp_error( $response ) ) {
        return '<p>Error loading data.</p>';
    }
    $xml_string = wp_remote_retrieve_body( $response );

	// Remove BOM if present
	$xml_string = preg_replace('/^\xEF\xBB\xBF/', '', $xml_string);

	// Parse XML
	$xml = simplexml_load_string( $xml_string );
    if ( ! $xml ) {
        return '<p>Invalid data format.</p>';
    }

    // Build a uniform records array
    $records = [];
    foreach ( $xml->Record as $rec ) {
        $dateStr   = (string) $rec->Date;
        $timestamp = strtotime( $dateStr );
        $value     = floatval( $rec->{$field} );
        if ( $value > 0 ) {
            $records[] = [
                'date'      => $dateStr,
                'timestamp' => $timestamp,
                'value'     => $value,
            ];
        }
    }
    usort( $records, fn( $a, $b ) => $a['timestamp'] <=> $b['timestamp'] );

    // Compute latest, previous, gain & pct
    $latest   = end( $records );
    $previous = prev( $records );
    $gain     = $latest['value'] - $previous['value'];
    $gain_pct = $previous['value'] > 0
              ? ( $gain / $previous['value'] ) * 100
              : 0;
    $gain_cls = $gain >= 0 ? 'positive' : 'negative';

    // Helper to get high/low in past N days
    $get_range = function( $recs, $days ) {
        $cutoff = strtotime( "-{$days} days" );
        $vals   = array_column(
            array_filter( $recs, fn( $r ) => $r['timestamp'] >= $cutoff ),
            'value'
        );
        return [ max( $vals ), min( $vals ) ];
    };
    list( $h1w, $l1w )   = $get_range( $records, 7 );
    list( $h1m, $l1m )   = $get_range( $records, 30 );
    list( $h52w, $l52w ) = $get_range( $records, 365 );

    // Output the card
    ob_start();
    ?>
    <div class="gbcharts-quote-card">
      <h2><?php echo esc_html( $title ); ?></h2>
      <h2>
        $<?php echo esc_html(number_format( $latest['value'], 2 )); ?>
        <span style="font-size:14px;color:#666;">USD</span>
      </h2>
      <div class="gain <?php echo esc_attr( $gain_cls ); ?>">
        <?php echo $gain >= 0 ? '↑' : '↓'; ?>
        <?php echo ( $gain >= 0 ? '+' : '' ) . number_format( $gain, 2 ); ?>
        (<?php echo ( $gain >= 0 ? '+' : '' ) . number_format( $gain_pct, 2 ); ?>%)
        | <span style="color:#000;">Previous: $<?php echo number_format( $previous['value'], 2 ); ?></span>
      </div>
      <div class="as-of-date">
        As of <?php echo esc_html(gmdate( 'F j, Y', $latest['timestamp'] )); ?>
      </div>
      <table>
        <tr>
          <td>7 Day High</td><td><?php echo number_format( $h1w, 2 ); ?></td>
          <td>7 Day Low</td><td><?php echo number_format( $l1w, 2 ); ?></td>
        </tr>
        <tr>
          <td>30 Day High</td><td><?php echo number_format( $h1m, 2 ); ?></td>
          <td>30 Day Low</td><td><?php echo number_format( $l1m, 2 ); ?></td>
        </tr>
        <tr>
          <td>52 Week High</td><td><?php echo number_format( $h52w, 2 ); ?></td>
          <td>52 Week Low</td><td><?php echo number_format( $l52w, 2 ); ?></td>
        </tr>
      </table>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode( 'gbcharts_quote_card', 'gbcharts_quote_card_shortcode' );

//quote and chart combo cards
function gbcharts_quote_combo_shortcode( $atts ) {
    // pull the rate attribute
    $atts = shortcode_atts( [
        'rate' => 'GB',   // default
    ], $atts, 'gbcharts_quote_combo' );

    $rate = strtoupper( $atts['rate'] );

switch ( $rate ) {
    case 'UPMA':
        $field       = 'UPMAER';
        $title       = 'UPMA Exchange Rate';
        $containerID = 'gbcharts-upma-quote-combo';
        $canvasID    = 'upmaChart';
        $buttonClass = 'filter-buttons';
        $valueKey    = 'upma';
        break;

    case 'GB':
        $field       = 'GBER';
        $title       = 'Goldback Exchange Rate';
        $containerID = 'gbcharts-quote-combo';
        $canvasID    = 'gberChart';
        $buttonClass = 'filter-buttons';
        $valueKey    = 'gber';
        break;

    case 'GOLD':
        $field       = 'UPMAAU';
        $title       = 'UPMA Gold Spot';
        $containerID = 'gbcharts-gold-quote-combo';
        $canvasID    = 'goldChart';
        $buttonClass = 'filter-buttons';
        $valueKey    = 'gold';
        break;

    case 'SILVER':
        $field       = 'UPMAAG';
        $title       = 'UPMA Silver Spot';
        $containerID = 'gbcharts-silver-quote-combo';
        $canvasID    = 'silverChart';
        $buttonClass = 'filter-buttons';
        $valueKey    = 'silver';
        break;

    default:
        // default to GB
        $field       = 'GBER';
        $title       = 'Goldback Exchange Rate';
        $containerID = 'gbcharts-quote-combo';
        $canvasID    = 'gberChart';
        $buttonClass = 'filter-buttons';
        $valueKey    = 'gber';
        break;
}

    // fetch & parse the XML
    $xml_url  = 'https://services.idealmsp.com/IMSPlugins/goldback-info/goldback_data.xml';
    $response = wp_remote_get( $xml_url );
    if ( is_wp_error( $response ) ) {
        return '<p>Error loading data.</p>';
    }
    $xml_string = wp_remote_retrieve_body( $response );

	// Remove BOM if present
	$xml_string = preg_replace('/^\xEF\xBB\xBF/', '', $xml_string);

	// Parse XML
	$xml = simplexml_load_string( $xml_string );
    if ( ! $xml ) {
        return '<p>Invalid data format.</p>';
    }

    // build a uniform records array
    $records = [];
    foreach ( $xml->Record as $rec ) {
        $date      = (string) $rec->Date;
        $timestamp = strtotime( $date );
        $value     = floatval( $rec->{$field} );
        if ( $value > 0 ) {
            $records[] = [
                'date'      => $date,
                'timestamp' => $timestamp,
                'value'     => $value,
            ];
        }
    }
    usort( $records, fn( $a, $b ) => $a['timestamp'] <=> $b['timestamp'] );

    // compute latest, gains, ranges
    $latest   = end( $records );
    $previous = prev( $records );
    $gain     = $latest['value'] - $previous['value'];
    $gain_pct = $previous['value'] > 0
              ? ( $gain / $previous['value'] ) * 100
              : 0;
    $gain_cls = $gain >= 0 ? 'positive' : 'negative';

    // helper for highs/lows
    $get_range = function( $recs, $days ) {
        $cutoff = strtotime( "-{$days} days" );
        $vals   = array_column(
            array_filter( $recs, fn( $r ) => $r['timestamp'] >= $cutoff ),
            'value'
        );
        return [ max( $vals ), min( $vals ) ];
    };
    list( $h1w, $l1w )   = $get_range( $records, 7 );
    list( $h1m, $l1m )   = $get_range( $records, 30 );
    list( $h52w, $l52w ) = $get_range( $records, 365 );

    // render
    ob_start(); ?>
    <div id="<?php echo esc_attr( $containerID ); ?>">
      <h2><?php echo esc_html( $title ); ?></h2>
      <h2>
        $<?php echo number_format( $latest['value'], 2 ); ?>
        <span style="font-size:14px;color:#666;">USD</span>
      </h2>
      <div class="gain <?php echo esc_attr( $gain_cls ); ?>">
        <?php echo $gain >= 0 ? "&#8593;" : "&#8595;"; ?>
        <?php echo ( $gain >= 0 ? '+' : '' ) . number_format( $gain, 2 ); ?>
        (<?php echo ( $gain >= 0 ? '+' : '' ) . number_format( $gain_pct, 2 ); ?>%)
         | <span style="color:#000;">Previous: $<?php echo number_format( $previous['value'], 2 ); ?></span>
      </div>
      <div class="as-of-date">As of <?php echo esc_html(gmdate( "F j, Y", $latest['timestamp'] )); ?></div>
      <div class="<?php echo esc_attr( $buttonClass ); ?>">
        <button data-days="7">1W</button>
        <button data-days="30">1M</button>
        <button data-days="365">1Y</button>
      </div>
      <div style="height:200px;">
        <canvas
          id="<?php echo esc_attr( $canvasID ); ?>"
          data-records='<?php echo wp_json_encode( $records ); ?>'
        ></canvas>
      </div>
      <table>
        <tr><td>7 Day High</td><td><?php echo number_format( $h1w, 2 ); ?></td>
            <td>7 Day Low</td><td><?php echo number_format( $l1w, 2 ); ?></td></tr>
        <tr><td>30 Day High</td><td><?php echo number_format( $h1m, 2 ); ?></td>
            <td>30 Day Low</td><td><?php echo number_format( $l1m, 2 ); ?></td></tr>
        <tr><td>52 Week High</td><td><?php echo number_format( $h52w, 2 ); ?></td>
            <td>52 Week Low</td><td><?php echo number_format( $l52w, 2 ); ?></td></tr>
      </table>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode( 'gbcharts_quote_combo', 'gbcharts_quote_combo_shortcode' );

// Historical data table shortcode
function gbcharts_data_table_shortcode( $atts ) {
    // Parse & normalize rate
    $atts = shortcode_atts( [
        'rate' => 'GB',
    ], $atts, 'gbcharts_data_table' );
    $rate = strtoupper( $atts['rate'] );

    // Map rate → XML field
    $fieldMap = [
        'GB'     => 'GBER',
        'UPMA'   => 'UPMAER',
        'GOLD'   => 'UPMAAU',
        'SILVER' => 'UPMAAG',
    ];
    $field = $fieldMap[ $rate ] ?? $fieldMap['GB'];

    // Fetch & parse XML
    $xml_url  = 'https://services.idealmsp.com/IMSPlugins/goldback-info/goldback_data.xml';
    $response = wp_remote_get( $xml_url );
    if ( is_wp_error( $response ) ) {
        return '<p>Error loading data.</p>';
    }
    $xml_string = wp_remote_retrieve_body( $response );

	// Remove BOM if present
	$xml_string = preg_replace('/^\xEF\xBB\xBF/', '', $xml_string);

	// Parse XML
	$xml = simplexml_load_string( $xml_string );
    if ( ! $xml ) {
        return '<p>Invalid data format.</p>';
    }

    // Build rows: date & value
    $raw = [];
    foreach ( $xml->Record as $rec ) {
        $dateStr = (string) $rec->Date;
        $value   = floatval( $rec->{$field} );
        if ( $value > 0 ) {
            $raw[] = [
                'date'  => $dateStr,
                'value' => $value,
            ];
        }
    }
    usort( $raw, fn( $a, $b ) => strtotime($a['date']) <=> strtotime($b['date']) );

    // Enrich with gain/loss
    $rows = [];
    for ( $i = 0; $i < count($raw); $i++ ) {
        $date  = $raw[$i]['date'];
        $value = $raw[$i]['value'];
        if ( $i === 0 ) {
            $gain = $loss = '';
        } else {
            $diff = $value - $raw[$i-1]['value'];
            if ( $diff >= 0 ) {
                $gain = number_format( $diff, 2 );
                $loss = '';
            } else {
                $gain = '';
                $loss = number_format( abs($diff), 2 );
            }
        }
        $rows[] = [
            'date'  => $date,
            'value' => number_format( $value, 2 ),
            'gain'  => $gain,
            'loss'  => $loss,
        ];
    }

   // Render the table HTML
    $table_id = 'gbcharts-data-table-' . uniqid();
    ob_start();
    ?>
    <table id="<?php echo esc_attr( $table_id ); ?>"
           class="gbcharts-data-table display"
           style="width:100%">
      <thead>
        <tr>
          <th>Date</th>
          <th>Value (USD)</th>
          <th>Gain</th>
          <th>Loss</th>
        </tr>
      </thead>
      <tbody>
  <?php foreach ( $rows as $r ) : ?>
    <tr>
      <td><?php echo esc_html( $r['date'] ); ?></td>
      <td>
        <?php
          echo $r['value'] !== ''
            ? '$' . esc_html( $r['value'] )
            : 'N/A';
        ?>
      </td>
      <td>
        <?php
          echo $r['gain'] !== ''
            ? '$' . esc_html( $r['gain'] )
            : 'N/A';
        ?>
      </td>
      <td>
        <?php
          echo $r['loss'] !== ''
            ? '$' . esc_html( $r['loss'] )
            : 'N/A';
        ?>
      </td>
    </tr>
  <?php endforeach; ?>
</tbody>
    </table>
    <?php
    return ob_get_clean();
}
add_shortcode( 'gbcharts_data_table', 'gbcharts_data_table_shortcode' );


