<?php
namespace Highpots\SpamProtection\Integrations;

use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class HPSP_Gravity_Forms extends HPSP_Form_Integration {
    public function __construct() {
        parent::__construct('gravityforms');
    }

    public function should_activate(): bool {
        return HPSP_Utils::is_plugin_active('gravityforms');
    }

    public function is_enabled(): bool {
        return HPSP_Constants::get_enable_gravity_forms_integration_option();
    }

    public function init(): void {
        if (!$this->should_activate() || !$this->is_enabled()) {
            return;
        }

        // Inject fields via native hook
        add_filter('gform_form_tag', [$this, 'inject_fields'], 10, 2);

        // Hook into Gravity Forms validation
        add_filter('gform_validation', [$this, 'validate_submission'], 10, 1);

        // Also use spam filter as backup
        add_filter('gform_entry_is_spam', [$this, 'validate_spam'], 10, 3);
    }

    /**
     * Inject spam protection fields into Gravity Forms
     */
    public function inject_fields(string $form_tag, array $form): string {
        // Prevent double injection
        if (strpos($form_tag, 'name="hpsp_form_id"') !== false) {
            return $form_tag;
        }

        $form_id = $this->generate_form_id($form, 'gravityforms');
        $fields_html = $this->field_renderer->render_as_html($form_id);

        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized with wp_kses() in HPSP_Utils::sanitize_spam_field_html()
        return $form_tag . $this->sanitize_field_html($fields_html);
    }

    /**
     * Validate Gravity Forms submission during validation phase
     */
    public function validate_submission($validation_result): array {
        $form = $validation_result['form'];
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by Gravity Forms
        $form_data = map_deep(wp_unslash($_POST), 'sanitize_text_field');

        $spam_validation = $this->handle_validation($form_data);

        if (!$spam_validation['success']) {
            $validation_result['is_valid'] = false;

            // Add error to first field or create general error
            if (!empty($form['fields'])) {
                $validation_result['form']['fields'][0]['failed_validation'] = true;
                $validation_result['form']['fields'][0]['validation_message'] = $spam_validation['message'];
            }
        }

        return $validation_result;
    }

    /**
     * Validate spam for Gravity Forms (backup method)
     */
    public function validate_spam($is_spam, $form, $entry): bool {
        if ($is_spam) {
            return $is_spam; // Already marked as spam
        }

        $validation_result = $this->handle_validation();

        return !$validation_result['success'];
    }

}
