<?php
namespace Highpots\SpamProtection\Integrations;

use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generic form integration for spam protection.
 *
 * This integration handles validation for forms protected by the
 * frontend JavaScript injection. It serves as a fallback for forms
 * that don't have specific plugin integrations.
 */
class HPSP_Generic_Form extends HPSP_Form_Integration {

    public function __construct() {
        parent::__construct('generic-content');
    }

    public function should_activate(): bool {
        return true; // Always active as fallback
    }

    public function is_enabled(): bool {
        return HPSP_Constants::get_enable_generic_integration_option();
    }

    public function init(): void {
        if (!$this->should_activate() || !$this->is_enabled()) {
            return;
        }

        // Handle generic form submissions - validation only
        // Field injection is handled by frontend JavaScript
        add_action('template_redirect', [$this, 'handle_form_submission']);
    }

    /**
     * Handle generic form submissions
     */
    public function handle_form_submission(): void {
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated -- Only checking request method
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            return;
        }

        // Check if this is a spam-protected form
        // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.ValidatedSanitizedInput.InputNotValidated -- Custom nonce handled by spam protection token, only checking key existence
        if (!isset($_POST['hpsp_form_id'])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            return;
        }

        $validation_result = $this->handle_validation();

        if (!$validation_result['success']) {
            // For generic forms, we'll set a query parameter and redirect
            $redirect_url = add_query_arg([
                'hpsp_error' => '1',
                'hpsp_message' => urlencode($validation_result['message'])
            ], wp_get_referer() ?: home_url());

            wp_safe_redirect($redirect_url);
            exit;
        }
    }

}
