<?php
namespace Highpots\SpamProtection\Integrations;

use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class HPSP_Formidable extends HPSP_Form_Integration {
    public function __construct() {
        parent::__construct('formidable');
    }

    public function should_activate(): bool {
        return HPSP_Utils::is_plugin_active('formidable');
    }

    public function is_enabled(): bool {
        return HPSP_Constants::get_enable_formidable_integration_option();
    }

    public function init(): void {
        if (!$this->should_activate() || !$this->is_enabled()) {
            return;
        }

        // Inject fields via native hook
        add_filter('frm_filter_final_form', [$this, 'inject_fields'], 10, 2);

        // Hook into Formidable validation
        add_filter('frm_validate_entry', [$this, 'validate_submission'], 10, 2);
    }

    /**
     * Inject spam protection fields into Formidable Forms
     */
    public function inject_fields(string $form_html, $form): string {
        // Prevent double injection
        if (strpos($form_html, 'name="hpsp_form_id"') !== false) {
            return $form_html;
        }

        $form_id = $this->generate_form_id($form, 'formidable');
        $fields_html = $this->field_renderer->render_as_html($form_id);

        // Inject before closing form tag (like Contact Form 7/Elementor)
        if (strpos($form_html, '</form>') !== false) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized with wp_kses() in HPSP_Utils::sanitize_spam_field_html()
            return str_replace('</form>', $this->sanitize_field_html($fields_html) . '</form>', $form_html);
        }

        // Fallback: append if no closing tag found
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized with wp_kses() in HPSP_Utils::sanitize_spam_field_html()
        return $form_html . $this->sanitize_field_html($fields_html);
    }

    /**
     * Validate Formidable form submission
     */
    public function validate_submission($errors, $values): array {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by Formidable Forms
        $form_data = map_deep(wp_unslash($_POST), 'sanitize_text_field');

        $validation_result = $this->handle_validation($form_data);

        if (!$validation_result['success']) {
            $errors['spam_protection'] = $validation_result['message'];
        }

        return $errors;
    }

}
