<?php

namespace Highpots\SpamProtection\Integrations;

use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\HPSP_Form_Field_Extractor;
use Highpots\SpamProtection\HPSP_Validator;
use Highpots\SpamProtection\HPSP_Field_Renderer;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

abstract class HPSP_Form_Integration
{

    protected HPSP_Validator $validator;
    protected HPSP_Form_Field_Extractor $field_extractor;
    protected HPSP_Field_Renderer $field_renderer;


    protected string $integration_name;

    public function __construct(string $integration_name)
    {
        $this->integration_name = $integration_name;
        $this->validator = new HPSP_Validator();
        $this->field_extractor = new HPSP_Form_Field_Extractor();
        $this->field_renderer = new HPSP_Field_Renderer();
    }

    /**
     * Initialize the integration
     */
    abstract public function init(): void;

    /**
     * Check if this integration should be active
     */
    abstract public function should_activate(): bool;

    /**
     * Check if this integration is enabled in settings
     */
    abstract public function is_enabled(): bool;

    /**
     * Handle form validation
     */
    protected function handle_validation(array $form_data = []): array
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by parent form plugin
        if (!empty($form_data)) {
            $data = $form_data;
        } else {
            // Sanitize early - deep sanitization of $_POST data
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verification handled by parent form plugin
            $data = map_deep(wp_unslash($_POST), 'sanitize_text_field');
        }

        return $this->validator->validate_submission($data);
    }

    /**
     * Send error response and exit
     */
    protected function send_error_response(array $validation_result): void
    {
        $message = $validation_result['message'] ?? esc_html__('Spam protection triggered.', 'highpots-spam-protection');
        $response = [
            'success' => false,
            'status' => 'validation_failed',
            'message' => $message,
            'data' => [
                'message' => $message,
                'form_id' => $validation_result['form_id'] ?? '',
                //'log_type' => $validation_result['log_type'] ?? '',
            ]
        ];

        wp_send_json($response);
    }

    /**
     * Get integration name
     */
    public function get_name(): string
    {
        return $this->integration_name;
    }

    /**
     * Generate form ID with integration prefix
     *
     * @param mixed $form Form object or ID
     * @param string $prefix Integration prefix (e.g., 'contact-form-7', 'gravityforms')
     * @return string Prefixed form ID
     */
    protected function generate_form_id($form, string $prefix): string
    {
        $id = '';

        // Try to extract ID from form object/array
        if (is_object($form)) {
            if (isset($form->id)) {
                $id = $form->id;
            } elseif (isset($form->ID)) {
                $id = $form->ID;
            } elseif (method_exists($form, 'id')) {
                $id = $form->id();
            } elseif (method_exists($form, 'get_id')) {
                $id = $form->get_id();
            }
        } elseif (is_array($form)) {
            $id = $form['id'] ?? $form['ID'] ?? '';
        } elseif (is_numeric($form)) {
            $id = $form;
        }

        // Fallback to timestamp if no ID found
        if (empty($id)) {
            $id = 'auto_' . time();
        }

        return $prefix . '_' . $id;
    }

    /**
     * Sanitize rendered HTML fields with wp_kses
     *
     * @param string $html HTML string from field renderer
     * @return string Sanitized HTML safe for output
     */
    protected function sanitize_field_html(string $html): string
    {
        return HPSP_Utils::sanitize_spam_field_html($html);
    }
}
