<?php

namespace Highpots\SpamProtection\Integrations;

use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class HPSP_Elementor_Form extends HPSP_Form_Integration
{
    public function __construct()
    {
        parent::__construct('elementor');
    }

    public function should_activate(): bool
    {
        return HPSP_Utils::is_plugin_active('elementor');
    }

    public function is_enabled(): bool
    {
        return HPSP_Constants::get_enable_elementor_integration_option();
    }

    public function init(): void
    {
        if (!$this->should_activate() || !$this->is_enabled()) {
            return;
        }

        // Inject fields via native hook
        add_filter('elementor/widget/render_content', [$this, 'inject_fields'], 10, 2);

        // Hook into Elementor form validation
        add_action('elementor_pro/forms/validation', [$this, 'validate_submission'], 5, 2);
    }

    /**
     * Inject spam protection fields into Elementor form
     */
    public function inject_fields(string $content, $widget): string
    {
        // Only process Elementor form widgets
        if ('form' !== $widget->get_name()) {
            return $content;
        }

        // Prevent double injection
        if (strpos($content, 'name="hpsp_form_id"') !== false) {
            return $content;
        }

        // Get widget ID for form identification
        $widget_id = $widget->get_id();
        $form_id = $this->generate_form_id($widget_id, 'elementor');

        // Render protection fields
        $fields_html = $this->field_renderer->render_as_html($form_id);

        // Inject before closing form tag (like Contact Form 7/WPForms)
        if (strpos($content, '</form>') !== false) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized with wp_kses() in HPSP_Utils::sanitize_spam_field_html()
            return str_replace('</form>', $this->sanitize_field_html($fields_html) . '</form>', $content);
        }

        // Fallback: append if no closing tag found
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized with wp_kses() in HPSP_Utils::sanitize_spam_field_html()
        return $content . $this->sanitize_field_html($fields_html);
    }

    /**
     * Validate Elementor form submission during validation phase
     */
    public function validate_submission($record, $ajax_handler): void
    {
        // Use handle_validation() without params to read from $_POST
        // This includes both Elementor's form fields and our custom injected fields
        $validation_result = $this->handle_validation();

        if (!$validation_result['success']) {
            // Use add_error() to properly register the error and prevent submission
            $ajax_handler->add_error('hpsp_spam_protection', $validation_result['message']);
            $ajax_handler->add_error_message($validation_result['message']);
        }
    }
}
