<?php

namespace Highpots\SpamProtection\Integrations;

use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class HPSP_Contact_Form7 extends HPSP_Form_Integration
{

    public function __construct()
    {
        parent::__construct('contact-form-7');
    }

    public function should_activate(): bool
    {
        return HPSP_Utils::is_plugin_active('contact-form-7');
    }

    public function is_enabled(): bool
    {
        return HPSP_Constants::get_enable_cf7_integration_option();
    }

    public function init(): void
    {
        if (!$this->should_activate() || !$this->is_enabled()) {
            return;
        }

        // Inject fields via native hook
        add_filter('wpcf7_form_elements', [$this, 'inject_fields'], 10, 1);

        // Validate submission
        add_action('wpcf7_submit', [$this, 'validate_submission'], 9, 2);
    }

    /**
     * Inject spam protection fields into CF7 form
     */
    public function inject_fields(string $form_html): string
    {
        // Prevent double injection
        if (strpos($form_html, 'name="hpsp_form_id"') !== false) {
            return $form_html;
        }

        // Extract form ID from current CF7 form
        $contact_form = \WPCF7_ContactForm::get_current();
        if (!$contact_form) {
            return $form_html;
        }

        $form_id = $this->generate_form_id($contact_form->id(), 'contact-form-7');
        $fields_html = $this->field_renderer->render_as_html($form_id);

        // Try to inject before closing form tag
        if (strpos($form_html, '</form>') !== false) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized with wp_kses() in HPSP_Utils::sanitize_spam_field_html()
            return str_replace('</form>', $this->sanitize_field_html($fields_html) . '</form>', $form_html);
        }

        // Fallback: append if no closing tag found
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized with wp_kses() in HPSP_Utils::sanitize_spam_field_html()
        return $form_html . $this->sanitize_field_html($fields_html);
    }

    /**
     * Validate CF7 form submission
     */
    public function validate_submission($contact_form, $abort = false): void
    {
        // Don't validate if submission is already aborted

        if ($abort) {
            return;
        }

        $submission = \WPCF7_Submission::get_instance();

        if (!$submission) {
            return;
        }

        // Get posted data from submission
        $posted_data = $submission->get_posted_data();


        $validation_result = $this->handle_validation($posted_data);

        if (!$validation_result['success']) {
            // Send JSON response and exit (CF7 expects this for AJAX submissions)
            $this->send_cf7_error_response($validation_result, $contact_form, $submission);
        }
    }

    /**
     * Send CF7-specific error response
     */
    private function send_cf7_error_response(array $validation_result, $contact_form, $submission): void
    {
        // Create CF7-compatible response
        $message = $validation_result['message'] ?? esc_html__('Spam protection triggered.', 'highpots-spam-protection');
        $response = [
            'contact_form_id' => $contact_form->id(),
            'success' => false,
            'status' => 'validation_failed',
            'message' => $message,
            'posted_data_hash' => wp_hash(serialize($submission->get_posted_data())),
            'data' => [
                'message' => $message,
            ]
        ];

        wp_send_json($response);
    }
}
