<?php
namespace Highpots\SpamProtection\Helpers;

use DOMDocument;
use Highpots\SpamProtection\HPSP_Dom_Processor;

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}


/**
 * Utility class for common operations
 */
class HPSP_Utils {
   /**
     * Get client IP address
     */
    public static function get_client_ip_address(): string {
        $ip = 'UNKNOWN';
        
        $headers = [
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ];

        foreach ($headers as $header) {
            if (isset($_SERVER[$header]) && !empty($_SERVER[$header])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- IP address validated with filter_var below
                $ip = wp_unslash($_SERVER[$header]);
                break;
            }
        }

        // Handle comma-separated IPs (X-Forwarded-For)
        if (strpos($ip, ',') !== false) {
            $ip = trim(explode(',', $ip)[0]);
        }

        // Validate and return IP
        $validated_ip = filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
        return $validated_ip ?: filter_var($ip, FILTER_VALIDATE_IP) ?: 'UNKNOWN';
    }

    /**
     * Mask IP address for privacy
     */
    public static function mask_ip_address(string $ip_address): string {
        if (filter_var($ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $parts = explode('.', $ip_address);
            if (count($parts) === 4) {
                return "{$parts[0]}.{$parts[1]}.xxx.xxx";
            }
        } elseif (filter_var($ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $parts = explode(':', $ip_address);
            if (count($parts) >= 4) {
                return implode(':', array_slice($parts, 0, 4)) . '::xxxx';
            }
        }
        
        return $ip_address;
    }

    /**
     * Get unique user identifier without using PHP sessions
     * Uses WordPress transients and cookies for session-less identification
     *
     * Prefixes:
     * - 'user_' for logged-in WordPress users (user_123)
     * - 'anon_' for anonymous visitors tracked by cookie (anon_abc123...)
     *
     * @return string Unique identifier for current user/visitor
     */
    public static function get_user_identifier(): string {
        // For logged-in users, use user ID
        if (is_user_logged_in()) {
            return 'user_' . get_current_user_id();
        }

        // For anonymous users, use cookie-based identifier
        $cookie_name = 'hpsp_uid';

        if (isset($_COOKIE[$cookie_name]) && !empty($_COOKIE[$cookie_name])) {
            // Validate existing identifier
            $identifier = sanitize_text_field(wp_unslash($_COOKIE[$cookie_name]));
            if (preg_match('/^[a-f0-9]{32}$/', $identifier)) {
                return 'anon_' . $identifier;
            }
        }

        // Generate new identifier for anonymous user
        $identifier = md5(wp_generate_password(32, false) . time());

        // Set cookie (24 hours expiry, httponly for security)
        // WordPress constants with fallbacks for edge cases
        $expiry = time() + (defined('DAY_IN_SECONDS') ? DAY_IN_SECONDS : 86400);
        $path = defined('COOKIEPATH') ? COOKIEPATH : '/';
        $domain = defined('COOKIE_DOMAIN') ? COOKIE_DOMAIN : '';
        $secure = function_exists('is_ssl') ? is_ssl() : (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on');

        setcookie($cookie_name, $identifier, $expiry, $path, $domain, $secure, true);

        return 'anon_' . $identifier;
    }

    /**
     * Check if plugin is active
     */
    public static function is_plugin_active(string $name): bool {
        if (!function_exists('is_plugin_active')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins = [
            //'coblocks' => 'coblocks/class-coblocks.php',
            //'forminator' => 'forminator/forminator.php',
            //'wpdiscuz' => 'wpdiscuz/class.WpdiscuzCore.php',
            //'wpmembers' => 'wp-members/wp-members.php',
            //'html-forms' => 'html-forms/html-forms.php',
            //'woocommerce' => 'woocommerce/woocommerce.php',
            'elementor' => 'elementor/elementor.php',
            'formidable' => 'formidable/formidable.php',
            'gravityforms' => 'gravityforms/gravityforms.php',
            'contact-form-7' => 'contact-form-7/wp-contact-form-7.php',
            'wpforms' => ['wpforms/wpforms.php', 'wpforms-lite/wpforms.php'],
        ];

        if (!isset($plugins[$name])) {
            return apply_filters('hpsp_plugin_active', false, $name);
        }

        $plugin_files = is_array($plugins[$name]) ? $plugins[$name] : [$plugins[$name]];
        
        foreach ($plugin_files as $plugin_file) {
            if (is_plugin_active($plugin_file)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get allowed HTML tags and attributes for spam protection fields
     *
     * @return array Allowed HTML structure for wp_kses
     */
    public static function get_spam_field_allowed_html(): array {
        return [
            'div' => [
                'style'             => true,
                'data-hpsp-fields' => true,
            ],
            'input' => [
                'type'         => true,
                'name'         => true,
                'value'        => true,
                'style'        => true,
                'aria-hidden'  => true,
                'tabindex'     => true,
                'autocomplete' => true,
            ],
        ];
    }

    /**
     * Sanitize spam protection field HTML with wp_kses
     *
     * @param string $html HTML string to sanitize
     * @return string Sanitized HTML
     */
    public static function sanitize_spam_field_html(string $html): string {
        return wp_kses($html, self::get_spam_field_allowed_html());
    }
}