<?php
namespace Highpots\SpamProtection\Helpers;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles generation and management of security salts.
 *
 * @package Highpots\SpamProtection\Helpers
 */
class HPSP_Salt_Generator {

	/**
	 * Generates a cryptographically secure random salt.
	 *
	 * @param int $length The length of the salt (default 64 characters).
	 * @return string The generated salt.
	 */
	public static function generate_salt( int $length = 64 ): string {
		// Use WordPress's secure random string generator
		$salt = wp_generate_password( $length, true, true );
		
		// Alternative: Use random_bytes for even stronger entropy
		// $salt = bin2hex( random_bytes( $length / 2 ) );
		
		return $salt;
	}

	/**
	 * Checks if required salts are defined in wp-config.php.
	 *
	 * @return array Array of missing salt constants.
	 */
	public static function get_missing_salts(): array {
		$required_salts = [
			'HPSP_HMAC_SECRET',
			'HPSP_HONEY_POT_SALT',
		];

		$missing = [];
		foreach ( $required_salts as $salt ) {
			if ( ! defined( $salt ) ) {
				$missing[] = $salt;
			}
		}

		return $missing;
	}

	/**
	 * Checks if wp-config.php is writable.
	 *
	 * @return bool True if writable, false otherwise.
	 */
	public static function is_wp_config_writable(): bool {
		$wp_config_path = self::get_wp_config_path();
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Checking wp-config.php permissions
		return $wp_config_path && is_writable( $wp_config_path );
	}

	/**
	 * Gets the path to wp-config.php.
	 *
	 * @return string|false The path to wp-config.php or false if not found.
	 */
	public static function get_wp_config_path() {
		// Check in ABSPATH first
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_exists -- Locating wp-config.php
		if ( file_exists( ABSPATH . 'wp-config.php' ) ) {
			return ABSPATH . 'wp-config.php';
		}

		// Check one level up (common setup)
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_exists -- Locating wp-config.php
		if ( file_exists( dirname( ABSPATH ) . '/wp-config.php' ) ) {
			return dirname( ABSPATH ) . '/wp-config.php';
		}

		return false;
	}

	/**
	 * Generates configuration code for wp-config.php.
	 *
	 * @return string The configuration code to add.
	 */
	public static function generate_config_code(): string {
		$hmac_salt = self::generate_salt( 64 );
		$honeypot_salt = self::generate_salt( 64 );

		$code = "\n" . str_repeat( '/', 80 ) . "\n";
		$code .= "// HighPots Spam Protection Salts\n";
		$code .= "// Generated on " . gmdate( 'Y-m-d H:i:s' ) . "\n";
		$code .= str_repeat( '/', 80 ) . "\n\n";
		$code .= "define( 'HPSP_HMAC_SECRET', '{$hmac_salt}' );\n";
		$code .= "define( 'HPSP_HONEY_POT_SALT', '{$honeypot_salt}' );\n\n";

		return $code;
	}

	/**
	 * Attempts to write salts to wp-config.php automatically.
	 *
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public static function write_salts_to_config() {
		$wp_config_path = self::get_wp_config_path();

		if ( ! $wp_config_path ) {
			return new \WP_Error(
				'config_not_found',
				esc_html__( 'wp-config.php file not found.', 'highpots-spam-protection' )
			);
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Checking wp-config.php permissions
		if ( ! is_writable( $wp_config_path ) ) {
			return new \WP_Error(
				'config_not_writable',
				esc_html__( 'wp-config.php is not writable.',  'highpots-spam-protection')
			);
		}

		// Read current config
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents -- Reading wp-config.php content
		$config_content = file_get_contents( $wp_config_path );

		// Check if salts already exist
		if ( strpos( $config_content, 'HPSP_HMAC_SECRET' ) !== false ) {
			return new \WP_Error(
				'salts_exist',
				esc_html__( 'Salts already exist in wp-config.php.', 'highpots-spam-protection' )
			);
		}

		// Find the position to insert (before "That's all, stop editing!")
		$insert_marker = "/* That's all, stop editing!";
		$insert_position = strpos( $config_content, $insert_marker );

		if ( $insert_position === false ) {
			// Fallback: insert before closing PHP tag
			$insert_marker = '?>';
			$insert_position = strpos( $config_content, $insert_marker );
		}

		if ( $insert_position === false ) {
			// Last resort: append to end
			$new_content = $config_content . self::generate_config_code();
		} else {
			// Insert before marker
			$new_content = substr_replace( 
				$config_content, 
				self::generate_config_code(), 
				$insert_position, 
				0 
			);
		}

		// Create backup (also requiresroot folder to be writable)
	/* 	$backup_path = $wp_config_path . '.backup-' . time();
		if ( ! copy( $wp_config_path, $backup_path ) ) {
			return new \WP_Error( 
				'backup_failed', 
				__( 'Failed to create backup of wp-config.php.', 'highpots-spam-protection' ) 
			);
		} */

		// Write new content
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Writing to wp-config.php
		if ( file_put_contents( $wp_config_path, $new_content ) === false ) {
			return new \WP_Error(
				'write_failed',
				esc_html__( 'Failed to write to wp-config.php.', 'highpots-spam-protection' )
			);
		}

		return true;
	}

	/**
	 * Stores salts in options table as fallback.
	 *
	 * @return bool True on success.
	 */
	public static function store_salts_in_options(): bool {
		if ( ! defined( 'HPSP_HMAC_SECRET' ) ) {
			update_option(
				HPSP_Constants::HMAC_SECRET_OPTION_KEY,
				self::generate_salt( 64 ),
				false // Don't autoload
			);
		}

		if ( ! defined( 'HPSP_HONEY_POT_SALT' ) ) {
			update_option(
				HPSP_Constants::HONEYPOT_SALT_OPTION_KEY,
				self::generate_salt( 64 ),
				false // Don't autoload
			);
		}

		return true;
	}
}