<?php

namespace Highpots\SpamProtection\Enums;

if (! defined('ABSPATH')) {
  exit;
}

/**
 * Enum-like class for spam protection log types (PHP 7.4 compatible).
 *
 * Defines all possible validation failure reasons and successful submissions
 * that can be logged by the spam protection system.
 *
 * @package Highpots\SpamProtection\Enums
 */
class HPSP_Log_Type
{

  /** @var string Successful form submission */
  const VALID = 'valid';

  /** @var string Invalid or expired HMAC token */
  const INVALID_TOKEN = 'invalid_token';

  /** @var string Honeypot field was filled (bot detected) */
  const HONEYPOT = 'honeypot';

  /** @var string Form submitted too quickly (likely bot) */
  const FAST_SUBMISSION = 'fast_submission';

  /** @var string Form submission took too long (token expired) */
  const SLOW_SUBMISSION = 'slow_submission';

  /** @var string Too many submissions from same IP within time window */
  const TOO_MANY_SUBMISSIONS = 'too_many_submissions';

  /** @var string Submission contains blocked writing systems/characters */
  const BLOCKED_WRITING_SYSTEMS = 'blocked_writing_systems';

  /** @var string Form ID is missing from submission */
  const FORM_ID_MISSING = 'form_id_missing';

  /** @var string Suspicious or missing User-Agent header */
  const SUSPICIOUS_USER_AGENT = 'suspicious_user_agent';

  /** @var string Invalid or missing referrer header */
  const SUSPICIOUS_REFERER = 'suspicious_referer';


  /**
   * Get the human-readable label for a log type.
   *
   * @param string $type The log type constant value
   * @return string Translated label text
   */
  public static function label(string $type): string
  {
    switch ($type) {
      case self::VALID:
        return esc_html__('Valid', 'highpots-spam-protection');
      case self::TOO_MANY_SUBMISSIONS:
        return esc_html__('Too many submissions', 'highpots-spam-protection');
      case self::INVALID_TOKEN:
        return esc_html__('Invalid token', 'highpots-spam-protection');
      case self::SUSPICIOUS_USER_AGENT:
        return esc_html__('Suspicious user agent', 'highpots-spam-protection');
      case self::HONEYPOT:
        return esc_html__('Honeypot', 'highpots-spam-protection');
      case self::FAST_SUBMISSION:
        return esc_html__('Fast submission', 'highpots-spam-protection');
      case self::SLOW_SUBMISSION:
        return esc_html__('Slow submission', 'highpots-spam-protection');
      case self::BLOCKED_WRITING_SYSTEMS:
        return esc_html__('Blocked writing systems', 'highpots-spam-protection');
      case self::FORM_ID_MISSING:
        return esc_html__('Form ID missing', 'highpots-spam-protection');
      default:
        return '';
    }
  }


  /**
   * Get the user-facing error message for a log type.
   *
   * Returns an appropriate error message to display to users when
   * their form submission is blocked by this validation rule.
   *
   * @param string $type The log type constant value
   * @return string Translated error message
   */
  public static function error_message(string $type): string
  {
    switch ($type) {
      case self::TOO_MANY_SUBMISSIONS:
        return esc_html__('Too many submissions', 'highpots-spam-protection');
      case self::INVALID_TOKEN:
        return esc_html__('Invalid token', 'highpots-spam-protection');
      case self::HONEYPOT:
        return esc_html__('Honeypot triggered', 'highpots-spam-protection');
      case self::FAST_SUBMISSION:
        return esc_html__('Submitted Too Quickly', 'highpots-spam-protection');
      case self::SLOW_SUBMISSION:
        return esc_html__('Form Expired', 'highpots-spam-protection');
      case self::BLOCKED_WRITING_SYSTEMS:
        return esc_html__('Disallowed characters found', 'highpots-spam-protection');
      case self::FORM_ID_MISSING:
        return esc_html__('Form ID missing', 'highpots-spam-protection');
      case self::SUSPICIOUS_USER_AGENT:
        return esc_html__('Suspicious user agent detected', 'highpots-spam-protection');
      case self::SUSPICIOUS_REFERER:
        return esc_html__('Invalid referrer', 'highpots-spam-protection');
      default:
        return '';
    }
  }


  /**
   * Try to get a label from a string key.
   *
   * Safely attempts to get a label for a log type value.
   * Returns the original key if the type is invalid.
   *
   * @param string $key The log type value
   * @return string The label or original key if invalid
   */
  public static function tryLabel(string $key): string
  {
    $label = self::label($key);
    return $label !== '' ? $label : $key;
  }


  /**
   * Get the details array for a log type.
   *
   * Returns an associative array with the type's value and label.
   *
   * @param string $type The log type constant value
   * @return array{value: string, label: string}
   */
  public static function getDetails(string $type): array
  {
    return [
      'value' => $type,
      'label' => self::label($type),
    ];
  }

  /**
   * Get all log type constants.
   *
   * @return array<string> Array of all log type constant values
   */
  public static function cases(): array
  {
    return [
      self::VALID,
      self::INVALID_TOKEN,
      self::HONEYPOT,
      self::FAST_SUBMISSION,
      self::SLOW_SUBMISSION,
      self::TOO_MANY_SUBMISSIONS,
      self::BLOCKED_WRITING_SYSTEMS,
      self::FORM_ID_MISSING,
      self::SUSPICIOUS_USER_AGENT,
      self::SUSPICIOUS_REFERER,
    ];
  }

  /**
   * Convert all log types to an array.
   *
   * Returns an array of all log types with their details (value and label).
   *
   * @return array<int, array{value: string, label: string}>
   */
  public static function toArray(): array
  {
    $result = [];
    foreach (self::cases() as $case) {
      $result[] = self::getDetails($case);
    }
    return $result;
  }
}
