<?php

namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Enums\HPSP_Log_Type;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

if (! defined('ABSPATH')) {
	exit;
}

/**
 * Centralizes all spam validation logic using the other manager classes.
 *
 * @package Highpots\SpamProtection
 * 
 */
class HPSP_Validator
{

	/**
	 * Token manager instance.
	 *
	 * @var HPSP_Token_Manager
	 */
	private HPSP_Token_Manager $token_manager;

	/**
	 * Honeypot manager instance.
	 *
	 * @var HPSP_Honeypot_Manager
	 */
	private HPSP_Honeypot_Manager $honeypot_manager;

	/**
	 * Rate limiter instance.
	 *
	 * @var HPSP_Rate_Limiter
	 */
	private HPSP_Rate_Limiter $rate_limiter;

	/**
	 * HPSP_Logger instance.
	 *
	 * @var HPSP_Logger
	 */
	private HPSP_Logger $logger;


	/**
	 * HPSP_Form_Field_Extractor instance.
	 *
	 * @var HPSP_Form_Field_Extractor
	 */
	private HPSP_Form_Field_Extractor $field_extractor;

	/**
	 * Constructor.
	 *
	 * @param HPSP_Token_Manager|null $token_manager Token manager instance.
	 * @param HPSP_Honeypot_Manager|null $honeypot_manager Honeypot manager instance.
	 * @param HPSP_Rate_Limiter|null $rate_limiter Rate limiter instance.
	 * @param HPSP_Logger|null $logger HPSP_Logger instance.
	 * @param HPSP_Form_Field_Extractor|null $field_extractor HPSP_Form_Field_Extractor instance.
	 */
	public function __construct(
		?HPSP_Token_Manager $token_manager = null,
		?HPSP_Honeypot_Manager $honeypot_manager = null,
		?HPSP_Rate_Limiter $rate_limiter = null,
		?HPSP_Logger $logger = null,
		?HPSP_Form_Field_Extractor $field_extractor = null,
	) {
		$this->token_manager = $token_manager ?? new HPSP_Token_Manager();
		$this->honeypot_manager = $honeypot_manager ?? new HPSP_Honeypot_Manager();
		$this->rate_limiter = $rate_limiter ?? new HPSP_Rate_Limiter();
		$this->logger = $logger ?? new HPSP_Logger();
		$this->field_extractor = $field_extractor ?? new HPSP_Form_Field_Extractor();
	}

	/**
	 * Validates submission timing to detect bots.
	 *
	 * @param array $post_data The POST data containing timing information.
	 * @return array Validation result with status, difference, and message.
	 */
	public function validate_timing(int $rendered_at): array
	{
		$min_time = HPSP_Constants::get_min_time_option();
		$max_time = HPSP_Constants::get_max_time_option();

		$diff = time() - $rendered_at;
		$status = 'valid';

		if ($diff < $min_time) {
			$status = 'fast';
		} elseif ($diff > $max_time) {
			$status = 'slow';
		}

		return array(
			'status'     => $status,
			'difference' => $diff,
			'message'    => sprintf('Form submitted in %d seconds.', $diff),
		);
	}

	/**
	 * Detects blocked writing systems in form data.
	 *
	 * @param array $form_fields Array of form field values to check.
	 * @return array Array of detected blocked writing systems.
	 */
	public function detect_blocked_writing_systems(array $form_fields): array
	{
		$regexes = array(
			'latin'       => '/[a-zA-Z]/',
			'cyrillic'    => '/[\p{Cyrillic}]/u',
			'arabic'      => '/[\p{Arabic}]/u',
			'hebrew'      => '/[\p{Hebrew}]/u',
			'armenian'    => '/[\p{Armenian}]/u',
			'georgian'    => '/[\p{Georgian}]/u',
			'bengali'     => '/[\p{Bengali}]/u',
			'chinese'     => '/[\p{Han}]/u',
			'japanese'    => '/[\p{Hiragana}\p{Katakana}]/u',
			'korean'      => '/[\p{Hangul}]/u',
			'greek'       => '/[\p{Greek}]/u',
			'devanagari'  => '/[\p{Devanagari}]/u',
			'thai'        => '/[\p{Thai}]/u',
			'tamil'       => '/[\p{Tamil}]/u',
			'ethiopic'    => '/[\p{Ethiopic}]/u',
		);

		$blocked_systems = HPSP_Constants::get_blocked_writing_systems_option();

		if (empty($blocked_systems)) {
			return array();
		}

		$detected_systems = array();

		foreach ($form_fields as $value) {
			if (! is_string($value)) {
				continue;
			}

			foreach ($blocked_systems as $system) {
				if (! isset($regexes[$system])) {
					continue;
				}

				if (preg_match($regexes[$system], $value) && ! in_array($system, $detected_systems, true)) {
					$detected_systems[] = $system;
				}
			}
		}

		return $detected_systems;
	}


	/**
	 * Validates the User-Agent header of the request.
	 *
	 * Checks if the User-Agent header is present and doesn't match
	 * known bot patterns. Helps prevent automated spam submissions.
	 *
	 * @return bool True if User-Agent is valid, false if suspicious or missing
	 */
	public function validate_user_agent(): bool
	{
		$user_agent = '';
		if (isset($_SERVER['HTTP_USER_AGENT'])) {
			$user_agent = sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT']));
		}

		// First check if empty
		if (empty($user_agent)) {
			return false;
		}


		// Block empty user agents or known bot patterns
		$blocked_patterns = [
			'/bot/i',
			'/crawler/i',
			'/spider/i',
		];

		foreach ($blocked_patterns as $pattern) {
			if (preg_match($pattern, $user_agent)) {
				return false;
			}
		}

		return true;
	}


	/**
	 * Validates the referrer header of the request.
	 *
	 * Checks if the HTTP_REFERER header matches the site URL,
	 * ensuring the form submission originated from this website.
	 *
	 * @return bool True if referrer is valid and matches site URL, false otherwise
	 */
	public function validate_referrer(): bool
	{
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Referrer URL used for comparison only, not output
		$referrer = isset($_SERVER['HTTP_REFERER']) ? wp_unslash($_SERVER['HTTP_REFERER']) : '';

		// Allow empty referrer (some browsers/privacy tools strip it)
		if (empty($referrer)) {
			return true;
		}

		$site_url = get_site_url();
		return strpos($referrer, $site_url) === 0;
	}



	/**
	 * Validates a complete form submission.
	 *
	 * @param array $post_data The POST data to validate.
	 * @return array Validation result with success status, message, and error details.
	 */
	public function validate_submission(array $post_data): array
	{
		$form_id = isset($post_data['hpsp_form_id'])
			? sanitize_text_field($post_data['hpsp_form_id'])
			: '';


		if (empty($form_id)) {

			return $this->format_error_response($form_id, HPSP_Log_Type::FORM_ID_MISSING);
		}


		// Extract spam protection fields
		$token = sanitize_text_field($post_data['hpsp_hmac_token'] ?? '');
		$rendered_at = absint($post_data['hpsp_rendered_at'] ?? 0);
		$nonce = sanitize_text_field($post_data['hpsp_hmac_nonce'] ?? '');



		// Honeypot validation
		if (! $this->honeypot_manager->validate_honeypot($form_id, $post_data)) {
			$this->cleanup_nonce($nonce);
			return $this->format_error_response($form_id, HPSP_Log_Type::HONEYPOT);
		}


		// User Agent validation (if enabled)
		if (HPSP_Constants::get_enable_user_agent_validation_option() && !$this->validate_user_agent()) {
			return $this->format_error_response($form_id, HPSP_Log_Type::SUSPICIOUS_USER_AGENT);
		}

		// Referrer validation (if enabled)
		if (HPSP_Constants::get_enable_referrer_validation_option() && !$this->validate_referrer()) {
			return $this->format_error_response($form_id, HPSP_Log_Type::SUSPICIOUS_REFERER);
		}



		// Rate limiting validation (if enabled)
		if (HPSP_Constants::get_enable_rate_limiting_option() && ! $this->rate_limiter->validate_submission($form_id)) {
			return $this->format_error_response($form_id, HPSP_Log_Type::TOO_MANY_SUBMISSIONS);
		}

		// Extract form fields for content validation
		$form_fields = $this->field_extractor->extract($post_data);

		// Check for blocked writing systems
		if (! empty($form_fields)) {
			$blocked_writing_systems = $this->detect_blocked_writing_systems($form_fields);
			if (! empty($blocked_writing_systems)) {
				return $this->format_error_response(
					$form_id,
					HPSP_Log_Type::BLOCKED_WRITING_SYSTEMS,
					implode(', ', $blocked_writing_systems)
				);
			}
		}

		// HMAC token validation
		if (! $this->token_manager->validate_token($token, $rendered_at, $nonce, $form_id)) {
			return $this->format_error_response($form_id, HPSP_Log_Type::INVALID_TOKEN);
		}

		// Timing validation
		$timing_result = $this->validate_timing($rendered_at);
		$timing_message = $timing_result['message'];

		if ('slow' === $timing_result['status']) {
			return $this->format_error_response($form_id, HPSP_Log_Type::SLOW_SUBMISSION, $timing_message);
		}

		if ('fast' === $timing_result['status']) {
			return $this->format_error_response($form_id, HPSP_Log_Type::FAST_SUBMISSION, $timing_message);
		}

		// All validations passed
		$this->logger->log_event($form_id, HPSP_Log_Type::VALID, $timing_message);

		return array(
			'success' => true,
			'message' => 'Validation passed.',
		);
	}

	/**
	 * Formats an error response and logs the event.
	 *
	 * @param string $message The error message.
	 * @param string $form_id The form identifier.
	 * @param string $log_type The log type constant value.
	 * @param string|null $note Optional additional note.
	 * @return array The formatted error response.
	 */
	private function format_error_response(string $form_id, string $log_type, ?string $note = null): array
	{
		$this->logger->log_event($form_id, $log_type, $note);

		return array(
			'success'  => false,
			'status'   => 'validation_error',
			'message'  => HPSP_Log_Type::error_message($log_type),
			'form_id'  => $form_id,
			'log_type' => $log_type,
			'log_note' => $note,
		);
	}

	/**
	 * Cleans up a nonce transient.
	 *
	 * @param string $nonce The nonce to clean up.
	 */
	private function cleanup_nonce(string $nonce): void
	{
		if (! empty($nonce)) {
			delete_transient(Helpers\HPSP_Constants::HMAC_NONCE_TRANSIENT_PREFIX . $nonce);
		}
	}

	/**
	 * Gets the token manager instance.
	 *
	 * @return HPSP_Token_Manager The token manager.
	 */
	public function get_token_manager(): HPSP_Token_Manager
	{
		return $this->token_manager;
	}

	/**
	 * Gets the honeypot manager instance.
	 *
	 * @return HPSP_Honeypot_Manager The honeypot manager.
	 */
	public function get_honeypot_manager(): HPSP_Honeypot_Manager
	{
		return $this->honeypot_manager;
	}

	/**
	 * Gets the rate limiter instance.
	 *
	 * @return HPSP_Rate_Limiter The rate limiter.
	 */
	public function get_rate_limiter(): HPSP_Rate_Limiter
	{
		return $this->rate_limiter;
	}

	/**
	 * Gets the HPSP_Logger instance.
	 *
	 * @return HPSP_Logger The HPSP_Logger.
	 */
	public function get_logger(): HPSP_Logger
	{
		return $this->logger;
	}
}
