<?php
namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Helpers\HPSP_Constants;


if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
    exit;
}

class HPSP_Uninstaller {
    
    /**
     * Run uninstall process
     */
    public static function uninstall() {
        if ( ! current_user_can( 'activate_plugins' ) ) {
            return;
        }
        
        if ( is_multisite() ) {
            self::uninstall_multisite();
        } else {
            self::uninstall_single_site();
        }
        
        self::clear_cache();
    }
    
    /**
     * Uninstall for single site
     */
    private static function uninstall_single_site() {
        self::delete_options();
        self::delete_tables();
        self::delete_user_meta();
        self::delete_post_meta();
    }
    
    /**
     * Uninstall for multisite
     */
    private static function uninstall_multisite() {
        global $wpdb;

        // Get all blog IDs
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, using WordPress core table
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for uninstall, getting all blog IDs
        $blog_ids = $wpdb->get_col( "SELECT blog_id FROM {$wpdb->blogs}" );

        foreach ( $blog_ids as $blog_id ) {
            switch_to_blog( $blog_id );
            self::uninstall_single_site();
            restore_current_blog();
        }

        // Delete network-wide options
        self::delete_network_options();
    }
    
    /**
     * Delete network options
     */
    private static function delete_network_options() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, using WordPress core table
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for uninstall, deleting network options
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->sitemeta} WHERE meta_key LIKE %s",
                $wpdb->esc_like( HPSP_Constants::WP_OPTION_PREFIX ) . '%'
            )
        );
    }
    
    /**
     * Delete all plugin options
     */
    private static function delete_options() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, using WordPress core table
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for uninstall, deleting plugin options
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                $wpdb->esc_like( HPSP_Constants::WP_OPTION_PREFIX ) . '%'
            )
        );
    }
    
    /**
     * Delete custom database tables
     */
    private static function delete_tables() {
        global $wpdb;

        $tables = array(
            HPSP_Constants::get_logs_table_name(),
        );

        foreach ( $tables as $table ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table names are safe plugin constants defined in HPSP_Constants, required for uninstall
            $wpdb->query( "DROP TABLE IF EXISTS {$table}" );
        }
    }
    
    /**
     * Delete user meta
     */
    private static function delete_user_meta() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, using WordPress core table
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for uninstall, deleting user meta
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE %s",
                $wpdb->esc_like( HPSP_Constants::WP_OPTION_PREFIX ) . '%'
            )
        );
    }
    
    /**
     * Delete post meta
     */
    private static function delete_post_meta() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, using WordPress core table
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for uninstall, deleting post meta
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
                $wpdb->esc_like( HPSP_Constants::WP_OPTION_PREFIX ) . '%'
            )
        );
    }
    
    /**
     * Clear WordPress cache
     */
    private static function clear_cache() {
        wp_cache_flush();
    }
}