<?php
namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Helpers\HPSP_Constants;
use Highpots\SpamProtection\Helpers\HPSP_Salt_Generator;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles HMAC token generation and validation for spam protection.
 *
 * @package Highpots\SpamProtection
 * 
 */
class HPSP_Token_Manager {

	/**
	 * HMAC secret key for token generation.
	 *
	 * @var string
	 */
	private string $hmac_secret;

	/**
	 * Default token expiry time in seconds.
	 *
	 * @var int
	 */
	private int $expiry_time;

	/**
	 * Grace period for token reuse (in seconds).
	 *
	 * @var int
	 */
	private int $grace_period;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->hmac_secret = $this->get_hmac_secret();
		$this->expiry_time = $this->get_expiry_time();
		$this->grace_period = $this->get_grace_period();

		//$this->monitor_token_usage();
	}



	public function monitor_token_usage() {
		add_action( 'hpsp_token_reused', function( $form_id, $use_count, $nonce_data ) {
			// Monitor suspicious reuse patterns
			// Token reuse monitoring - can be extended with custom logging if needed
		}, 10, 3 );

		add_action( 'hpsp_token_replay_blocked', function( $form_id, $time_since_use, $nonce_data ) {
			// Monitor blocked replay attacks
			// Replay attack monitoring - can be extended with custom logging if needed
		}, 10, 3 );
	}


	/**
	 * Gets the HMAC secret key.
	 *
	 * @return string The HMAC secret key.
	 */
	private function get_hmac_secret(): string {
		if ( defined( 'HPSP_HMAC_SECRET' ) ) {
			return HPSP_HMAC_SECRET;
		}

		// Fallback to options
		$secret = get_option( HPSP_Constants::HMAC_SECRET_OPTION_KEY );

		if ( ! $secret ) {
			// Generate and store if missing
			$secret = HPSP_Salt_Generator::generate_salt( 64 );
			update_option( HPSP_Constants::HMAC_SECRET_OPTION_KEY, $secret, false );
		}

		return $secret;
	}

	/**
	 * Gets the current expiry time setting.
	 *
	 * @return int The expiry time in seconds.
	 */
	private function get_expiry_time(): int {
		return HPSP_Constants::get_max_time_option();
	}



	/**
	 * Gets the grace period for token reuse.
	 *
	 * @return int The grace period in seconds.
	 */
	private function get_grace_period(): int {
		// Get from settings
		$grace_period = HPSP_Constants::get_token_grace_period_option();
		
		/**
		 * Filters the grace period for token reuse.
		 *
		 * Allows tokens to be reused within this time window for legitimate
		 * retries (network failures, accidental double-clicks, etc.).
		 *
		 * @param int $grace_period Grace period in seconds from settings.
		 */
		return apply_filters( 'hpsp_token_grace_period', $grace_period );
	}

	/**
	 * Generates a unique HMAC token for form protection.
	 *
	 * @param string $form_id A unique identifier for the form.
	 * @return array The generated HMAC token data including token, timestamp, and nonce.
	 */
	public function generate_token( string $form_id ): array {
		$timestamp = time();
		$user_identifier = Helpers\HPSP_Utils::get_user_identifier();
		$nonce = wp_generate_password( 16, false );

		$data = $form_id . '|' . $timestamp . '|' . $user_identifier . '|' . $nonce;
		$token = hash_hmac( 'sha256', $data, $this->hmac_secret );

		// Store nonce data in transient for validation
		$transient_key = HPSP_Constants::HMAC_NONCE_TRANSIENT_PREFIX . $nonce;
		$nonce_data = array(
			'data'       => $data,
			'form_id'    => $form_id,
			'created_at' => $timestamp,
			'used'       => false,
			'used_at'    => null,
			'use_count'  => 0,
		);
		
		set_transient( $transient_key, $nonce_data, $this->expiry_time );

		return array(
			'token'     => $token,
			'timestamp' => $timestamp,
			'nonce'     => $nonce,
		);
	}

	/**
	 * Validates an HMAC token with replay protection.
	 *
	 * @param string $token The HMAC token to validate.
	 * @param int $timestamp The timestamp associated with the token.
	 * @param string $nonce The nonce associated with the token.
	 * @param string $form_id A unique identifier for the form.
	 * @return bool True if the token is valid, false otherwise.
	 */
	public function validate_token( string $token, int $timestamp, string $nonce, string $form_id ): bool {
		$user_identifier = Helpers\HPSP_Utils::get_user_identifier();
		$data = $form_id . '|' . $timestamp . '|' . $user_identifier . '|' . $nonce;
		$expected_token = hash_hmac( 'sha256', $data, $this->hmac_secret );

		// Check if token has expired
		if ( time() - $timestamp > $this->expiry_time ) {
			return false;
		}

		// Check if tokens match using timing-safe comparison
		if ( ! hash_equals( $token, $expected_token ) ) {
			return false;
		}

		// Get nonce data
		$nonce_key = HPSP_Constants::HMAC_NONCE_TRANSIENT_PREFIX . $nonce;
		$nonce_data = get_transient( $nonce_key );

		if ( false === $nonce_data ) {
			return false; // Nonce doesn't exist or has expired
		}

		// Verify data integrity
		if ( ! hash_equals( $nonce_data['data'], $data ) ) {
			return false; // Data mismatch
		}

		// Check if token was already used
		if ( true === $nonce_data['used'] ) {
			$time_since_use = time() - ( $nonce_data['used_at'] ?? 0 );

			// Reject if used outside grace period
			if ( $time_since_use > $this->grace_period ) {
				/**
				 * Fires when a token is reused outside the grace period.
				 *
				 * @param string $form_id The form identifier.
				 * @param int $time_since_use Seconds since token was first used.
				 * @param array $nonce_data The nonce data array.
				 */
				do_action( 'hpsp_token_replay_blocked', $form_id, $time_since_use, $nonce_data );
				return false;
			}

			// Within grace period - allow for legitimate retries
		}

		// Mark token as used
		$nonce_data['used'] = true;
		$nonce_data['used_at'] = $nonce_data['used_at'] ?? time(); // Keep original use time
		$nonce_data['use_count'] = ( $nonce_data['use_count'] ?? 0 ) + 1;

		// Update transient with remaining TTL
		$remaining_ttl = $this->get_remaining_ttl( $nonce_key );
		set_transient( $nonce_key, $nonce_data, $remaining_ttl );

		// Log if token was reused (for monitoring)
		if ( $nonce_data['use_count'] > 1 ) {
			/**
			 * Fires when a token is reused within the grace period.
			 *
			 * @param string $form_id The form identifier.
			 * @param int $use_count Number of times token has been used.
			 * @param array $nonce_data The nonce data array.
			 */
			do_action( 'hpsp_token_reused', $form_id, $nonce_data['use_count'], $nonce_data );
		}

		return true;
	}

	/**
	 * Gets remaining TTL for a transient.
	 *
	 * @param string $transient_key The transient key.
	 * @return int Remaining seconds until expiration.
	 */
	private function get_remaining_ttl( string $transient_key ): int {
		$timeout = get_option( '_transient_timeout_' . $transient_key );

		if ( false === $timeout ) {
			return $this->expiry_time; // Default if not found
		}

		// Ensure at least grace period remains
		return max( $this->grace_period, $timeout - time() );
	}
}