<?php

namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Helpers\HPSP_Utils;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles rate limiting for form submissions to prevent spam.
 *
 * @package Highpots\SpamProtection
 * 
 */
class HPSP_Rate_Limiter {

	/**
	 * Default time window in seconds.
	 *
	 * @var int
	 */
	private int $default_time_window;

	/**
	 * Default maximum submissions allowed within time window.
	 *
	 * @var int
	 */
	private int $default_max_submissions;

	/**
	 * Constructor.
	 *
	 * @param int|null $time_window Custom time window or null to use default.
	 * @param int|null $max_submissions Custom max submissions or null to use default.
	 */
	public function __construct( ?int $time_window = null, ?int $max_submissions = null ) {
		$this->default_time_window = $time_window ?? HPSP_Constants::get_rate_limit_window_option();
		$this->default_max_submissions = $max_submissions ?? HPSP_Constants::get_rate_limit_max_option();
	}

	

	/**
	 * Generates a unique transient key for rate limiting.
	 *
	 * @param string $form_id The form identifier.
	 * @param string|null $ip_address Optional IP address, will auto-detect if null.
	 * @return string The transient key.
	 */
	private function get_transient_key( string $form_id, ?string $ip_address = null ): string {
		$ip = $ip_address ?? HPSP_Utils::get_client_ip_address();
		$sanitized_form_id = sanitize_text_field( $form_id );

		return HPSP_Constants::RATE_LIMIT_TRANSIENT_PREFIX . 'submissions_' . md5( $sanitized_form_id . $ip );
	}

	/**
	 * Checks if a form submission should be rate limited.
	 *
	 * @param string $form_id The form identifier.
	 * @param int|null $time_window Custom time window or null to use default.
	 * @param int|null $max_submissions Custom max submissions or null to use default.
	 * @return bool True if submission is allowed, false if rate limited.
	 */
	public function check_rate_limit( string $form_id, ?int $time_window = null, ?int $max_submissions = null ): bool {
		$time_window = $time_window ?? $this->default_time_window;
		$max_submissions = $max_submissions ?? $this->default_max_submissions;
		$transient_key = $this->get_transient_key( $form_id );

		$submission_data = get_transient( $transient_key );

		if ( ! $submission_data || ! is_array( $submission_data ) ) {
			$submission_data = array(
				'count'           => 0,
				'last_submission' => 0,
			);
		}

		$current_time = time();

		// Reset count if time window has elapsed
		if ( $submission_data['last_submission'] > 0 && ( $current_time - $submission_data['last_submission'] ) > $time_window ) {
			$submission_data['count'] = 0;
			$submission_data['last_submission'] = 0;
			// Update transient to reset the counter
			set_transient( $transient_key, $submission_data, $time_window );
		}

		// Check if rate limit is exceeded
		if ( $submission_data['count'] >= $max_submissions ) {
			return false; // Rate limit exceeded
		}

		return true; // Submission allowed
	}

	/**
	 * Records a form submission for rate limiting.
	 *
	 * @param string $form_id The form identifier.
	 * @param int|null $time_window Custom time window or null to use default.
	 * @return bool True if submission was recorded successfully.
	 */
	public function record_submission( string $form_id, ?int $time_window = null ): bool {
		$time_window = $time_window ?? $this->default_time_window;
		$transient_key = $this->get_transient_key( $form_id );

		$submission_data = get_transient( $transient_key );

		if ( ! $submission_data || ! is_array( $submission_data ) ) {
			$submission_data = array(
				'count'           => 0,
				'last_submission' => 0,
			);
		}

		$current_time = time();

		// Reset count if time window has elapsed
		if ( $submission_data['last_submission'] > 0 && ( $current_time - $submission_data['last_submission'] ) > $time_window ) {
			$submission_data['count'] = 0;
		}

		// Increment submission count
		$submission_data['count']++;
		$submission_data['last_submission'] = $current_time;

		// Store updated data with proper expiration
		return set_transient( $transient_key, $submission_data, $time_window + 60 );
	}

	/**
	 * Validates and processes rate limiting for a form submission.
	 *
	 * @param string $form_id The form identifier.
	 * @param int|null $time_window Custom time window or null to use default.
	 * @param int|null $max_submissions Custom max submissions or null to use default.
	 * @return bool True if submission should proceed, false if rate limited.
	 */
	public function validate_submission( string $form_id, ?int $time_window = null, ?int $max_submissions = null ): bool {
		if ( ! $this->check_rate_limit( $form_id, $time_window, $max_submissions ) ) {
			return false; // Rate limited
		}

		// Record this submission
		$this->record_submission( $form_id, $time_window );
		return true; // Allowed
	}

	/**
	 * Gets current submission data for a form/IP combination.
	 *
	 * @param string $form_id The form identifier.
	 * @return array|false The submission data or false if none exists.
	 */
	public function get_submission_data( string $form_id ) {
		$transient_key = $this->get_transient_key( $form_id );
		return get_transient( $transient_key );
	}

	/**
	 * Clears rate limiting data for a form/IP combination.
	 *
	 * @param string $form_id The form identifier.
	 * @return bool True if cleared successfully.
	 */
	public function clear_rate_limit_data( string $form_id ): bool {
		$transient_key = $this->get_transient_key( $form_id );
		return delete_transient( $transient_key );
	}

	/**
	 * Sanitizes form ID input.
	 *
	 * @param string $form_id The form ID to sanitize.
	 * @return string The sanitized form ID.
	 */
	public function sanitize_form_id( string $form_id ): string {
		return sanitize_text_field( $form_id );
	}
}