<?php

namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Admin\HPSP_Admin_Manager;
use Highpots\SpamProtection\Admin\HPSP_Salt_Admin_Notice;
use Highpots\SpamProtection\Integrations\HPSP_Form_Integration_Manager;
use Highpots\SpamProtection\Helpers\HPSP_Constants;

if (! defined('ABSPATH')) {
    exit;
}


class HPSP_Plugin
{


    /**
     * The unique identifier of this plugin.
     *
     * @var string
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @var string
     */
    protected $version;



    private static $instance = null;


    /**
     * Initialize the class and set its properties.
     * Private constructor to enforce singleton pattern.
     */
    private function __construct()
    {
        $this->plugin_name = HPSP_PLUGIN_NAME;
        $this->version = HPSP_PLUGIN_VERSION;

        $this->load_dependencies();
        $this->register_hooks();
    }


    public static function get_instance(): HPSP_Plugin
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Load the required dependencies for this plugin.
     */
    private function load_dependencies() {}







    /**
     * Register all actions and filters for the plugin.
     */
    public function register_hooks()
    {
        add_action('plugins_loaded', [$this, 'plugins_loaded_handler'], 10);
        add_action('plugins_loaded', [$this, 'render_admin_pages'], 20);
        add_action('init', [$this, 'init']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_frontend_scripts']);

        // Initialize REST API
        $rest_api = new HPSP_Rest_Api();
        $rest_api->init();

        // Initialize salt admin notice early so AJAX hooks are registered
        if (is_admin()) {
            add_action('admin_init', [$this, 'initialize_admin_components']);
        }
    }

    /**
     * Load the plugin's text domain for internationalization.
     */
    public function plugins_loaded_handler()
    {
        // phpcs:ignore PluginCheck.CodeAnalysis.DiscouragedFunctions.load_plugin_textdomainFound -- Keeping manual loading for compatibility
        //load_plugin_textdomain('highpots-spam-protection', false, HPSP_PLUGIN_LANG_DIR);
    }


    public function render_admin_pages()
    {
        // Initialize the integration system
        HPSP_Form_Integration_Manager::get_instance();
        if (is_admin()) {
            HPSP_Admin_Manager::get_instance();
        }
    }


    /**
     * Initialize admin components early for AJAX support.
     * Fires on admin_init hook.
     */
    public function initialize_admin_components(): void
    {
        // Initialize salt admin notice (must be early for AJAX hooks)
        //HPSP_Salt_Admin_Notice::get_instance();
    }


    public function init()
    {
        HPSP_Scheduler::init();
    }

    /**
     * Enqueue the plugin's admin-specific scripts and styles.
     */
    public function enqueue_admin_scripts($hook)
    {

        // Get the current screen
        $screen = get_current_screen();

        if ($hook !== 'toplevel_page_hpsp-logs' && !str_contains($screen->id, 'page_hpsp-settings')) {
            return;
        }


        wp_enqueue_style('hpsp-style', HPSP_PLUGIN_DIR_URL . 'assets/css/hpsp-admin.css', array(), HPSP_PLUGIN_VERSION);
        wp_enqueue_script('hpsp-admin-script', HPSP_PLUGIN_DIR_URL . 'assets/js/hpsp-admin.js', array('jquery'), HPSP_PLUGIN_VERSION, true);

        wp_localize_script('hpsp-admin-script', 'hpsp_vars', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('hpsp_nonce'),
            'hpsp_export_csv_nonce'  => wp_create_nonce('hpsp_export_csv'),
            'hpsp_filter_logs_nonce'  => wp_create_nonce('hpsp_filter_logs'),
            'hpsp_dismiss_notice_nonce'  => wp_create_nonce('hpsp_dismiss_notice'),
            'i18n' => array(
                // Export messages
                'exporting' => esc_html__('Exporting...',  'highpots-spam-protection'),
                'no_data_export' => esc_html__('No data to export.',  'highpots-spam-protection'),
                'export_error' => esc_html__('An error occurred while preparing the export. Please try again.',  'highpots-spam-protection'),
                'export_error_generic' => esc_html__('An error occurred while processing your request. Please try again.',  'highpots-spam-protection'),
                'dismiss_notice' => esc_html__('Dismiss this notice.',  'highpots-spam-protection'),
            ),
            'error_messages' => array(
                'sett_min_max_error' => esc_html__("Minimum time must be less than maximum time.",  'highpots-spam-protection')
            )
        ));
    }

    /**
     * Enqueue frontend scripts for form protection.
     *
     * Only loads JavaScript if generic integration is enabled,
     * saving bandwidth when only using supported form plugins.
     */
    public function enqueue_frontend_scripts(): void
    {
        // Skip admin, AJAX, and cron contexts
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }

        // Only load JS if generic integration is enabled
        if (!HPSP_Constants::get_enable_generic_integration_option()) {
            return;
        }

        wp_enqueue_script(
            'hpsp-frontend',
            HPSP_PLUGIN_DIR_URL . 'assets/js/hpsp-frontend.js',
            [],
            HPSP_PLUGIN_VERSION,
            true
        );

        wp_localize_script('hpsp-frontend', 'hpsp_config', [
            'rest_url' => esc_url_raw(rest_url('hpsp/v1/')),
            'nonce'    => wp_create_nonce('wp_rest'),
        ]);
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @return string The name of the plugin.
     */
    public function get_plugin_name()
    {
        return $this->plugin_name;
    }

    /**
     * The version of the plugin.
     *
     * @return string The version of the plugin.
     */
    public function get_version()
    {
        return $this->version;
    }
}
