<?php

namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Admin\HPSP_Settings;
use Highpots\SpamProtection\Helpers\HPSP_Constants;
use Highpots\SpamProtection\Helpers\HPSP_Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles logging of spam protection events.
 *
 * @package Highpots\SpamProtection
 * 
 */
class HPSP_Logger {

	/**
	 * Database table name for logging.
	 *
	 * @var string
	 */
	private string $table_name;




	/**
	 * Constructor.
	 */
	public function __construct() {
		global $wpdb;
		$this->table_name = HPSP_Constants::get_logs_table_name();
	}

	 /**
     * Check if logging is enabled
     */
    private function is_logging_enabled(): bool {
        return HPSP_Constants::get_enable_logging_option();
    }

    /**
     * Cleanup old logs based on retention settings
     */
    public function cleanup_old_logs(): int {
        if (!$this->is_logging_enabled()) {
            return 0;
        }

        global $wpdb;


		$retention_days = HPSP_Constants::get_log_retention_days_option();

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed for log cleanup, caching not applicable
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE timestamp < DATE_SUB(NOW(), INTERVAL %d DAY)", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
            $retention_days
        ));

        return (int) $deleted ?: 0;
    }
	

	/**
	 * Logs a spam protection event.
	 *
	 * @param string $form_id The form identifier.
	 * @param string $log_type The type of event being logged (use HPSP_Log_Type constants).
	 * @param string|null $note Optional additional notes.
	 * @return bool True if logged successfully, false otherwise.
	 */
	public function log_event( string $form_id, string $log_type, ?string $note = null ): bool {


		if (!$this->is_logging_enabled()) {
            return false;
        }

		global $wpdb;

		$ip_address = HPSP_Utils::get_client_ip_address();

		$data = array(
			'timestamp' => current_time( 'mysql' ),
			'form_id'   => sanitize_text_field( $form_id ),
			'reason'    => $log_type,
			'ip_address'   => $ip_address,
			'location'  => null, // Can be expanded later for geolocation
		);

		if ( ! empty( $note ) ) {
			$data['note'] = sanitize_text_field( $note );
		}

		
		// Define format for each field
		$format = array(
			'%s', // timestamp
			'%s', // form_id
			'%s', // reason
			'%s', // ip_address
			'%s', // location (null will be handled correctly)
		);

		if ( ! empty( $note ) ) {
			$format[] = '%s'; // note
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct insert needed for logging, caching not applicable
		$result = $wpdb->insert(
			$this->table_name,
			$data,
			$format
		);

		if ( false === $result ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Failed to log spam protection event: ' . $wpdb->last_error ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Database error logging for debugging
			}
			return false;
		}

		return true;
	}

	/**
	 * Gets recent log entries.
	 *
	 * @param int $limit Maximum number of entries to retrieve.
	 * @param string|null $form_id Optional form ID filter.
	 * @param string|null $reason Optional reason filter.
	 * @return array Array of log entries.
	 */
	public function get_recent_logs( int $limit = 100, ?string $form_id = null, ?string $reason = null ): array {
		global $wpdb;

		$where_conditions = array( '1=1' );
		$where_values = array();

		if ( ! empty( $form_id ) ) {
			$where_conditions[] = 'form_id = %s';
			$where_values[] = sanitize_text_field( $form_id );
		}

		if ( ! empty( $reason ) ) {
			$where_conditions[] = 'reason = %s';
			$where_values[] = sanitize_text_field( $reason );
		}

		$where_clause = implode( ' AND ', $where_conditions );
		$limit = max( 1, min( 1000, intval( $limit ) ) ); // Limit between 1 and 1000

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
		$query = "SELECT id, timestamp, form_id, reason, ip_address, location, note FROM {$this->table_name} WHERE {$where_clause} ORDER BY timestamp DESC LIMIT %d";
		$where_values[] = $limit;

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- SQL is prepared with $wpdb->prepare()
		$prepared_query = $wpdb->prepare( $query, ...$where_values );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Direct query needed for log retrieval, SQL is properly prepared above with wpdb->prepare()
		$results = $wpdb->get_results( $prepared_query );

		return $results ?: array();
	}

	/**
	 * Gets statistics for a date range.
	 *
	 * @param string $start_date Start date in YYYY-MM-DD format.
	 * @param string $end_date End date in YYYY-MM-DD format.
	 * @return array Statistics grouped by date and reason.
	 */
	public function get_statistics( string $start_date, string $end_date ): array {
		global $wpdb;

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
		$query = "
			SELECT DATE(timestamp) as date,
				   reason,
				   COUNT(*) as count
			FROM {$this->table_name}
			WHERE DATE(timestamp) BETWEEN %s AND %s
			GROUP BY DATE(timestamp), reason
			ORDER BY date DESC, reason
		";

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- SQL is prepared with $wpdb->prepare()
		$prepared_query = $wpdb->prepare( $query, sanitize_text_field( $start_date ), sanitize_text_field( $end_date ) );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared -- Direct query needed for statistics, SQL is prepared above
		$results = $wpdb->get_results( $prepared_query );

		return $results ?: array();
	}



	/**
	 * Gets the total count of log entries.
	 *
	 * @param string|null $reason Optional reason filter.
	 * @return int Total count of entries.
	 */
	public function get_total_count( ?string $reason = null ): int {
		global $wpdb;

		if ( ! empty( $reason ) ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed for count, caching not applicable
			$count = $wpdb->get_var( $wpdb->prepare(
				"SELECT COUNT(*) FROM {$this->table_name} WHERE reason = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
				sanitize_text_field( $reason )
			) );
		} else {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query needed for count, caching not applicable
			$count = $wpdb->get_var(
				"SELECT COUNT(*) FROM {$this->table_name}" // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
			);
		}

		return intval( $count );
	}

	/**
	 * Exports log data as CSV.
	 *
	 * @param string $start_date Start date in YYYY-MM-DD format.
	 * @param string $end_date End date in YYYY-MM-DD format.
	 * @return string CSV formatted data.
	 */
	public function export_csv( string $start_date, string $end_date ): string {
		global $wpdb;

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe, defined by plugin constant
		$query = "
			SELECT timestamp, form_id, reason, ip_address, location, note
			FROM {$this->table_name}
			WHERE DATE(timestamp) BETWEEN %s AND %s
			ORDER BY timestamp DESC
		";

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- SQL is prepared with $wpdb->prepare()
		$prepared_query = $wpdb->prepare( $query, sanitize_text_field( $start_date ), sanitize_text_field( $end_date ) );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared -- Direct query needed for CSV export, SQL is prepared above
		$results = $wpdb->get_results( $prepared_query, ARRAY_A );

		if ( empty( $results ) ) {
			return '';
		}

		$csv_data = array();
		$csv_data[] = array_keys( $results[0] ); // Headers

		foreach ( $results as $row ) {
			$csv_data[] = array_values( $row );
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Using php://temp for CSV generation
		$output = fopen( 'php://temp', 'r+' );
		foreach ( $csv_data as $row ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv -- Writing CSV to temp stream
			fputcsv( $output, $row );
		}
		rewind( $output );
		$csv_string = stream_get_contents( $output );
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing temp stream
		fclose( $output );

		return $csv_string;
	}
}