<?php

namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Helpers\HPSP_Constants;
use Highpots\SpamProtection\Helpers\HPSP_Salt_Generator;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles honeypot field generation and validation for spam protection.
 *
 * @package Highpots\SpamProtection
 * 
 */
class HPSP_Honeypot_Manager {

	/**
	 * Salt for honeypot field name generation.
	 *
	 * @var string
	 */
	private string $honeypot_salt;

	/**
	 * Constructor.
	 *
	 * @param string|null $honeypot_salt Custom salt or null to use default.
	 */
	public function __construct( ?string $honeypot_salt = null ) {
		$this->honeypot_salt = $honeypot_salt ?? $this->get_honeypot_salt();
	}

	
	/**
	 * Gets the honeypot salt.
	 *
	 * @return string The honeypot salt.
	 */
	private function get_honeypot_salt(): string {
		if ( defined( 'HPSP_HONEY_POT_SALT' ) ) {
			return HPSP_HONEY_POT_SALT;
		}

		// Fallback to options
		$salt = get_option( HPSP_Constants::HONEYPOT_SALT_OPTION_KEY );

		if ( ! $salt ) {
			// Generate and store if missing
			$salt = HPSP_Salt_Generator::generate_salt( 64 );
			update_option( HPSP_Constants::HONEYPOT_SALT_OPTION_KEY, $salt, false );
		}

		return $salt;
	}

	/**
	 * Generates a randomized honeypot field name based on form ID.
	 *
	 * @param string $form_id The form identifier.
	 * @return string The randomized honeypot field name.
	 */
	public function generate_field_name( string $form_id = '' ): string {
		$sanitized_form_id = sanitize_text_field( $form_id );
		return 'hpsp_' . md5( $this->honeypot_salt . $sanitized_form_id );
	}

	/**
	 * Validates honeypot field submission.
	 *
	 * @param string $form_id The form identifier.
	 * @param array $post_data The POST data to check.
	 * @return bool True if honeypot validation passes (field is empty), false if spam detected.
	 */
	public function validate_honeypot( string $form_id, array $post_data ): bool {
		$honeypot_field_name = $this->generate_field_name( $form_id );

		// If honeypot field exists and is not empty, it's spam
		if ( isset( $post_data[ $honeypot_field_name ] ) && ! empty( $post_data[ $honeypot_field_name ] ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Creates honeypot field HTML for form injection.
	 *
	 * @param string $form_id The form identifier.
	 * @return string The honeypot field HTML.
	 */
	public function create_field_html( string $form_id ): string {
		$field_name = $this->generate_field_name( $form_id );

		return sprintf(
			'<input type="text" name="%s" style="position:absolute; left:-9999px;" aria-hidden="true" tabindex="-1" autocomplete="off">',
			esc_attr( $field_name )
		);
	}

	/**
	 * Creates honeypot field as DOM element attributes.
	 *
	 * @param string $form_id The form identifier.
	 * @return array Array of attributes for DOM creation.
	 */
	public function create_field_attributes( string $form_id ): array {
		return array(
			'type'         => 'text',
			'name'         => $this->generate_field_name( $form_id ),
			'style'        => 'position:absolute; left:-9999px;',
			'aria-hidden'  => 'true',
			'tabindex'     => '-1',
			'autocomplete' => 'off',
		);
	}

	/**
	 * Sanitizes form ID input.
	 *
	 * @param string $form_id The form ID to sanitize.
	 * @return string The sanitized form ID.
	 */
	public function sanitize_form_id( string $form_id ): string {
		return sanitize_text_field( $form_id );
	}
}