<?php
namespace Highpots\SpamProtection;

use Highpots\SpamProtection\Helpers\HPSP_Constants;
use Highpots\SpamProtection\Helpers\HPSP_Salt_Generator;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


class HPSP_Activator {

    /**
     * Minimum required PHP version
     */
    const MINIMUM_PHP_VERSION = '7.4';

    /**
     * Activate the plugin
     *
     * @return void
     */
    public static function activate() {
        // Check PHP version first
        if ( ! self::check_php_version() ) {
            return;
        }

        self::create_logs_table();


        // Try to generate salts on activation
        if ( HPSP_Salt_Generator::is_wp_config_writable() ) {
            HPSP_Salt_Generator::write_salts_to_config();
        } else {
            // Store in options as fallback
            HPSP_Salt_Generator::store_salts_in_options();
        }

        HPSP_Scheduler::activate();
    }



    private static function create_logs_table() {
        global $wpdb;
        $table_name = HPSP_Constants::get_logs_table_name();
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            timestamp DATETIME NOT NULL,
            form_id VARCHAR(100) NOT NULL,
            reason VARCHAR(50) NOT NULL,
            ip_address CHAR(64) NOT NULL,
            location VARCHAR(250) DEFAULT NULL,
            note VARCHAR(250) DEFAULT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Check if the server meets the minimum PHP version requirement
     *
     * @return bool True if PHP version is sufficient, false otherwise
     */
    private static function check_php_version() {
        if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
            // Deactivate the plugin
            deactivate_plugins( plugin_basename( dirname( __DIR__ ) . '/highpots-spam-protection.php' ) );

            // Show error message
            wp_die(
                wp_kses(
                    sprintf(
                        /* translators: 1: Plugin name, 2: Required PHP version, 3: Current PHP version */
                        __( '<strong>%1$s</strong> requires PHP version %2$s or higher. You are currently running PHP version %3$s.<br><br>The plugin has been deactivated. Please upgrade your PHP version to activate this plugin.', 'highpots-spam-protection' ),
                        HPSP_PLUGIN_DISPLAY_NAME,
                        self::MINIMUM_PHP_VERSION,
                        PHP_VERSION
                    ),
                    array(
                        'strong' => array(),
                        'br' => array()
                    )
                ),
                esc_html__( 'Plugin Activation Error',  'highpots-spam-protection'),
                array(
                    'back_link' => true,
                    'response'  => 200
                )
            );
        }

        return true;
    }
}